# -*- coding: utf-8 -*-

from datetime import datetime
import logging
import re

from passport.backend.api.common.account import (
    build_default_person_registration_info,
    build_empty_person_info,
    default_account,
)
from passport.backend.api.common.authorization import (
    build_auth_cookies_and_session,
    SessionScope,
    set_authorization_track_fields,
    users_from_multisession,
)
from passport.backend.api.common.processes import PROCESS_WEB_REGISTRATION
from passport.backend.api.common.profile.profile import process_env_profile
from passport.backend.api.common.social_api import (
    get_max_size_avatar_from_profile_fail_safe,
    get_social_api,
)
from passport.backend.api.exceptions import (
    AccountGlobalLogoutError,
    SessionExpiredError,
)
from passport.backend.api.views.bundle.auth.base import BundleBaseAuthorizationMixin
from passport.backend.api.views.bundle.auth.exceptions import (
    AuthNotAllowedError,
    AuthSessionExpiredError,
    AuthSessionOverflowError,
)
from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.exceptions import (
    Account2FAEnabledError,
    AccountDisabledError,
    AccountDisabledOnDeletionError,
    AccountGlobalLogoutError as BundleAccountGlobalLogoutError,
    AccountNotFoundError,
    AccountRequiredChangePasswordError,
    InternalPermanentError,
    InvalidTrackStateError,
    OAuthInvalidClientIdError,
    OAuthInvalidClientSecretError,
    PasswordNotMatchedError,
    TaskNotFoundError,
)
from passport.backend.api.views.bundle.headers import (
    HEADER_CLIENT_ACCEPT_LANGUAGE,
    HEADER_CLIENT_USER_AGENT,
    HEADER_CONSUMER_CLIENT_IP,
)
from passport.backend.api.views.bundle.mixins import (
    BundleAccountGetterMixin,
    BundleAccountPropertiesMixin,
    BundleAccountResponseRendererMixin,
    BundleAccountSubscribeMixin,
    BundleAssertCaptchaMixin,
    BundleAuthNotificationsMixin,
    BundleDeviceInfoMixin,
    BundleFrodoMixin,
    BundlePasswordChangeMixin,
    BundlePasswordVerificationMethodMixin,
    BundlePhoneMixin,
    CookieCheckStatus,
    MailSubscriptionsMixin,
)
from passport.backend.api.views.bundle.mixins.challenge import (
    BundleChallengeMixin,
    MobileProfile,
)
from passport.backend.api.views.bundle.mixins.oauth import BundleOAuthMixin
from passport.backend.api.views.bundle.states import (
    RedirectToForcedLiteCompletion,
    RedirectToPasswordChange,
)
from passport.backend.api.views.bundle.utils import (
    assert_valid_host,
    write_phone_to_log,
)
from passport.backend.core import validators
from passport.backend.core.avatars import upload_avatar_async
from passport.backend.core.builders.blackbox.blackbox import get_blackbox
from passport.backend.core.builders.blackbox.utils import add_phone_arguments
from passport.backend.core.builders.oauth.oauth import (
    get_oauth,
    OAUTH_CODE_STRENGTH_LONG,
    OAUTH_INVALID_GRANT_ERROR,
    OAUTH_FORBIDDEN_ACCOUNT_TYPE_ERROR_DESCRIPTION,
    OAuthPermanentError,
)
from passport.backend.core.builders.social_api import exceptions as social_api_exceptions
from passport.backend.core.builders.social_api.profiles import convert_task_profile
from passport.backend.core.builders.social_broker import (
    exceptions as social_broker_exceptions,
    SocialBroker,
)
from passport.backend.core.conf import settings
from passport.backend.core.counters import social_registration_captcha
from passport.backend.core.exceptions import UnknownUid
from passport.backend.core.logging_utils.loggers.statbox import StatboxLogger
from passport.backend.core.models.account import (
    Account,
    ACCOUNT_DISABLED_ON_DELETION,
)
from passport.backend.core.models.email import Email
from passport.backend.core.models.person import DisplayName
from passport.backend.core.runner.context_managers import (
    CREATE,
    UPDATE,
)
from passport.backend.core.services import get_service
from passport.backend.core.subscription import add_subscription
from passport.backend.core.types.account.account import (
    ACCOUNT_TYPE_LITE,
    ACCOUNT_TYPE_MAILISH,
    ACCOUNT_TYPE_PHONISH,
)
from passport.backend.core.types.phone_number.phone_number import PhoneNumber
from passport.backend.core.types.social_business_info import BusinessInfo
from passport.backend.core.utils.decorators import cached_property
from passport.backend.utils.common import remove_none_values
from passport.backend.utils.string import (
    smart_str,
    smart_text,
)
import six
from six.moves.urllib.parse import (
    parse_qsl,
    urlencode,
    urlparse,
    urlunparse,
)

from .exceptions import (
    ApplicationInvalidError,
    InvalidProviderTokenError,
    NameRequiredError,
    ProviderInvalidError,
)
from .states import (
    RedirectToChoose,
    RedirectToRegister,
    RedirectToRetpath,
    SuggestSocialState,
)


_MORDA_COM_HOSTNAME = re.compile(r'^(?:www\.)?yandex\.com$')
_QUERY_PARSED_URL_INDEX = 4


log = logging.getLogger('passport.api.view.bundle.auth.social')


OUTPUT_MODE_SESSIONID = 'sessionid'
OUTPUT_MODE_XTOKEN = 'xtoken'
OUTPUT_MODE_AUTHORIZATION_CODE = 'authorization_code'  # код для получения х-токена


class BaseSocialAuthView(BaseBundleView,
                         BundleBaseAuthorizationMixin,
                         BundleAccountSubscribeMixin,
                         BundleAccountResponseRendererMixin,
                         BundleAccountGetterMixin,
                         BundleAccountPropertiesMixin,
                         BundleAssertCaptchaMixin,
                         BundlePasswordChangeMixin,
                         BundlePasswordVerificationMethodMixin,
                         BundlePhoneMixin,
                         BundleDeviceInfoMixin,
                         BundleAuthNotificationsMixin,
                         BundleChallengeMixin,
                         ):
    required_grants = ['auth_social.base']
    required_headers = [
        HEADER_CLIENT_USER_AGENT,
        HEADER_CONSUMER_CLIENT_IP,
        HEADER_CLIENT_ACCEPT_LANGUAGE,
    ]
    track_type = 'authorize'

    event_action = 'account_register_social'
    antifraud_auth_type = None

    @cached_property
    def social_api(self):
        return get_social_api()

    @cached_property
    def blackbox(self):
        return get_blackbox()

    @cached_property
    def social_broker(self):
        return SocialBroker()

    @cached_property
    def oauth(self):
        return get_oauth()

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode='social',
            track_id=self.track_id,
            consumer=self.consumer,
        )

    def __init__(self):
        super(BaseSocialAuthView, self).__init__()
        self.task_data = None
        self.accounts = None
        self.profile_id = None
        self.provider = None

    def initialize_track(self, social_output_mode, code_challenge=None, code_challenge_method=None):
        self.create_track(self.track_type)

        # При дальнейшей работе нам будут нужны некоторые данные, сохраним их в track.
        with self.track_transaction.rollback_on_error():
            if 'retpath' in self.form_values:
                self.track.retpath = self.form_values['retpath']
            if 'place' in self.form_values:
                self.track.social_place = self.form_values['place']
            self.track.social_return_brief_profile = self.form_values.get('return_brief_profile') or False
            self.track.social_output_mode = social_output_mode
            if code_challenge:
                self.track.oauth_code_challenge = code_challenge
                self.track.oauth_code_challenge_method = code_challenge_method

            service = self.form_values.get('service')
            if service:
                self.track.service = service.slug

            if self.form_values.get('process_uuid'):
                self.track.process_uuid = self.form_values.get('process_uuid')

            if self.form_values.get('broker_consumer'):
                self.track.social_broker_consumer = self.form_values.get('broker_consumer')

    def bind_profile_to_enabled_account(self, task_id=None):
        """
        Привязываем профиль к аккаунту. Если аккаунт заблокирован - выбрасываем исключение.
        """

        if task_id is None:
            task_id = self.track.social_task_id
        assert task_id is not None

        self.response_values['account'] = self.get_account_short_info(self.account)
        self.check_account_enabled()

        try:
            self.profile_id = self.social_api.bind_task_profile(task_id, self.account.uid)
        except social_api_exceptions.TaskNotFoundError:
            raise TaskNotFoundError()
        self.response_values['profile_id'] = self.profile_id
        self.account.originated_from_profile_id = self.profile_id

    def load_accounts_and_select_state(
        self,
        suggest_uid=None,
    ):
        self.accounts = self.get_accounts_for_profile(log_to_statbox=True)

        if len(self.accounts) == 0:
            if self.is_suggest_related_account_enabled():
                self.suggest_register_or_login_to_related_account()
            else:
                self.state = RedirectToRegister()
        elif len(self.accounts) == 1:
            # Авторизуем пользователя.
            self.account = self.accounts[0]
            self.state = RedirectToRetpath()
        else:
            # Авторизуем пользователя, если у нас есть аккаунт с uid из саджеста
            if suggest_uid:
                suggest_accounts = [a for a in self.accounts if a.uid == suggest_uid]
                if len(suggest_accounts) == 1:
                    self.accounts = suggest_accounts
                    self.account = self.accounts[0]
                    self.state = RedirectToRetpath()
                    return

            # Отправляем пользователя на выбор аккаунта.
            self.state = RedirectToChoose()

            # Фронтенду нужны только uid и display_name.
            account_display_names = map(self.get_account_short_info, self.accounts)
            self.response_values['accounts'] = account_display_names

    def suggest_register_or_login_to_related_account(self):
        social_profile_email = self.profile.get('email')

        if not social_profile_email:
            self.state = RedirectToRegister()
            return

        try:
            self.get_account_by_login(
                login=social_profile_email,
                email_attributes='all',
                emails=True,
                enabled_required=False,
            )
            suggested_account = self.account
        except AccountNotFoundError:
            suggested_account = None

        if not suggested_account:
            if not (
                self.is_lite_login(social_profile_email) and
                self.is_available_lite_login(social_profile_email)
            ):
                self.state = RedirectToRegister()
                return

            register_lite_track = self.create_register_lite_track()
            self.response_values.update(
                can_register_lite=True,
                register_lite_track_id=register_lite_track.track_id,
            )
            self.state = SuggestSocialState()
            return

        if not suggested_account.is_enabled:
            self.state = RedirectToRegister()
            return

        if not (
            suggested_account.is_normal and suggested_account.have_password or
            suggested_account.is_lite
        ):
            self.state = RedirectToRegister()
            return

        self.state = SuggestSocialState()

        account_info = self.account_to_response(
            account=suggested_account,
            account_info_required=False,
            display_name_required=True,
            personal_data_required=False,
        )
        if suggested_account.emails.default:
            account_info.update(
                default_email=suggested_account.emails.default.address,
            )

        self.response_values.update(
            suggested_accounts=[account_info],
            auth_retpath=self.track.retpath,
            auth_track_id=self.track.track_id,
        )

        if (
            not suggested_account.is_lite or
            suggested_account.is_strong_password_required or
            suggested_account.totp_secret.is_set
        ):
            self.response_values.update(
                can_register_social=True,
                register_social_track_id=self.track.track_id,
            )

    def is_lite_login(self, login):
        try:
            validators.LiteLogin().to_python(login)
            return True
        except validators.Invalid:
            return False

    def is_available_lite_login(self, login):
        try:
            validators.Availability(ACCOUNT_TYPE_LITE).to_python(
                dict(login=login),
                validators.State(self.request.env),
            )
            return True
        except validators.Invalid:
            return False

    def create_register_lite_track(self):
        track = self.track_manager.create(
            'register',
            self.consumer,
            process_name=PROCESS_WEB_REGISTRATION,
        )
        with self.track_manager.transaction(track=track).rollback_on_error():
            track.process_uuid = self.track.process_uuid
            track.retpath = self.track.retpath
            track.social_task_data = self.track.social_task_data
            track.social_task_id = self.track.social_task_id
            track.social_track_id = self.track.track_id
        return track

    def check_user_policies(self, check_strong_password_policy=True):
        """
        Проверим, нужно ли перенаправить пользователя на страницу смены пароля.
        Сюда не может попасть недорегистрированный пользователь
        типа автозарегистрированного или недорегистрированный ПДД,
        но сюда может попасть недорегистрированный лайт с паролем, отправляем его
        на дорегистрацию.

        Если пользователю необходимо сменить пароль по любой причине, то возвращаем ошибку,
        которую обрабатывает фронтенд и предлагает авторизоваться паролем и пройти весь процесс.
        """
        redirect_state = super(BaseSocialAuthView, self).check_user_policies(check_strong_password_policy)
        if redirect_state is not None:
            if isinstance(redirect_state, RedirectToPasswordChange):
                raise AccountRequiredChangePasswordError()
            if isinstance(redirect_state, RedirectToForcedLiteCompletion):
                self.response_values['has_recovery_method'] = self.check_has_recovery_method()

                self.fill_response_with_account(personal_data_required=True, account_info_required=True)
                self.state = redirect_state
                return redirect_state

        self.check_otp_required()

    def update_subscriptions_and_statistics(self):
        self.subscribe_if_allow_and_update_account(self.account)

        # позже тут еще в статистику будем что-то писать

    def check_account_enabled(self):
        """
        Если аккаунт заблокирован, заполняем в ответе краткую информацию об аккаунте
        и выкидываем правильное исключение в зависимости от подписки на блокирующие SID'ы.
        """
        if self.account.is_user_enabled:
            return

        if self.account.disabled_status == ACCOUNT_DISABLED_ON_DELETION:
            raise AccountDisabledOnDeletionError()

        raise AccountDisabledError()

    def load_task_data(self, use_track=True, allowed_providers=None):
        """
        Загружает из трека или SocialApi социальный профиль и провайдера.
        """
        if allowed_providers is None:
            allowed_providers = settings.AUTH_ALLOWED_PROVIDERS

        self.task_data = None

        if use_track and self.track.social_task_data:
            self.task_data = self.track.social_task_data

        if not self.task_data:
            if not self.form_values.get('task_id'):
                raise TaskNotFoundError()
            try:
                self.task_data = self.social_api.get_task_data(self.form_values['task_id'])
            except social_api_exceptions.TaskNotFoundError:
                raise TaskNotFoundError()

        if use_track:
            self.track.social_task_data = self.task_data

        self.response_values['profile'] = self.task_data['profile']
        self.profile = convert_task_profile(self.task_data['profile'])

        self.provider = self.profile['provider']
        self.response_values['provider'] = self.provider

        if self.provider['code'].lower() not in allowed_providers:
            raise ProviderInvalidError()

    def load_task_data_from_track(self):
        if not self.track.social_task_data:
            log.debug('Track social task data should not be empty')
            raise InvalidTrackStateError()
        self.load_task_data()

    def get_accounts(
        self,
        uids,
        exclude_disabled=False,
        log_to_statbox=False,
        need_phones=True,
        need_emails=True,
    ):
        """
        По всем найденным профилям проведем проверку их через ЧЯ.
        uids - список uid'ов.
        Возвращает список существующих аккаунтов.
        """
        self.response_values['has_enabled_accounts'] = False
        if log_to_statbox:
            self.statbox.bind(
                enabled_accounts_count=0,
                disabled_accounts_count=0,
            )

        if not uids:
            return []
        uids_string = ','.join(map(str, uids))

        kwargs = {'uid': uids_string}
        if need_phones:
            kwargs = add_phone_arguments(**kwargs)
        if need_emails:
            kwargs.update(emails=True)
        try:
            accounts_data = self.blackbox.userinfo(**kwargs)
        except UnknownUid:
            # Аккаунт, соответствующий единственному переданному uid, не существует.
            return []

        if not isinstance(accounts_data, list):  # ответ для одиночного uid
            accounts_data = [accounts_data]

        accounts = [
            Account().parse(d)
            for d in accounts_data
            if d['uid']
        ]

        # Убираем аккаунты, которыми запрещена соц авторизация. Наличие у таких
        # аккаунтов профилей с allow_auth==1 - неожиданность, от которой мы на всякий случай защитимся.
        accounts = filter(
            lambda acc: acc.type not in [ACCOUNT_TYPE_MAILISH, ACCOUNT_TYPE_PHONISH],
            accounts,
        )

        accounts = filter(
            lambda acc: acc.disabled_status != ACCOUNT_DISABLED_ON_DELETION,
            accounts,
        )
        enabled_accounts = [
            account
            for account in accounts
            if account.is_user_enabled
        ]

        self.response_values['has_enabled_accounts'] = bool(enabled_accounts)
        if log_to_statbox:
            enabled_accounts_count = len(enabled_accounts)
            uids_string = ','.join(str(a.uid) for a in accounts)
            self.statbox.bind(
                enabled_accounts_count=enabled_accounts_count,
                disabled_accounts_count=len(accounts) - enabled_accounts_count,
                accounts=uids_string,
            )

        if exclude_disabled:
            accounts = enabled_accounts

        return accounts

    @staticmethod
    def get_account_short_info(account):
        output = {
            'uid': account.uid,
            'login': account.login,
            'is_pdd': account.is_pdd,
        }
        if account.person and account.person.display_name:
            output['display_name'] = account.person.display_name.as_dict()
        return output

    def get_accounts_for_profile(
        self,
        exclude_disabled=False,
        log_to_statbox=False,
        need_phones=True,
        need_emails=True,
    ):
        """
        Ходим в SocialApi, выбираем профили для текущего социального пользователя,
        для каждого профиля в ЧЯ получаем аккаунт, возвращаем список живых аккаунтов,
        для которых разрешена авторизация.
        """

        profiles_raw = self.social_api.get_profiles(
            userid=self.profile['userid'],
            provider=self.provider['code'],
            allow_auth=True,
            business_info=BusinessInfo.from_dict(self.profile.get('business')),
        )

        profile_id_by_uid = dict((profile['uid'], profile['profile_id']) for profile in profiles_raw)
        uids = profile_id_by_uid.keys()

        accounts = self.get_accounts(
            uids,
            exclude_disabled=exclude_disabled,
            log_to_statbox=log_to_statbox,
            need_emails=need_emails,
            need_phones=need_phones,
        )

        for account in accounts:
            account.originated_from_profile_id = profile_id_by_uid[account.uid]

        return accounts

    def fill_response_with_track_fields(self):
        """
        Достаем из трека всевозможные параметры, нужные фронту
        """
        is_native = self.track.social_output_mode == 'xtoken'

        data = dict(
            broker_consumer=self.track.social_broker_consumer,
            is_native=is_native,
            place=self.track.social_place,
            retpath=self.track.retpath,
            return_brief_profile=self.track.social_return_brief_profile,
            track_id=self.track_id,
        )
        data = remove_none_values(data)

        self.response_values.update(data)

    def is_socialreg_captcha_required(self):
        """
        Увеличиваем нужные счетчики, проверяем необходимость показа captcha пользователю.
        """
        captcha_required = social_registration_captcha.is_required(
            self.request.env.user_ip,
            self.provider['name'],
        )

        if captcha_required and self.track.is_captcha_recognized:
            captcha_required = False

        if captcha_required:
            self.track.is_captcha_required = True
        self.track.socialreg_captcha_required_count.incr()

        return captcha_required

    def check_otp_required(self):
        """
        Если на аккаунте включена 2FA, нужно попросить его ввести otp
        """
        if self.account.totp_secret.is_set:
            self.fill_response_with_account(
                personal_data_required=True,
                account_info_required=True,
            )
            cookie_session_info = self.check_session_cookie(
                dbfields=[],
                attributes=[],
            )
            self.response_values['accounts'] = self.get_multisession_accounts(cookie_session_info)

            raise Account2FAEnabledError()

    def check_host_if_required(self):
        if self.track.social_output_mode == OUTPUT_MODE_SESSIONID:
            assert_valid_host(self.request.env)

    def statbox_user_come_back_with_task(self):
        app_attrs = self.task_data['token']['application_attributes']
        self.statbox.log(
            action='callback_end',
            state=self.state.state,
            provider=self.provider.get('name'),
            application=app_attrs['id'],
            third_party_app=app_attrs['third_party'],
            userid=self.profile.get('userid'),
        )

    def _fill_response_with_retpath(self):
        if self.form_values.get('retpath'):
            self.response_values.update({'retpath': self.form_values['retpath']})

    def _fix_morda_retpath(self):
        retpath = self.form_values.get('retpath')
        if not retpath:
            return
        parsed_retpath = urlparse(smart_str(retpath))
        if not _MORDA_COM_HOSTNAME.match(parsed_retpath.hostname or ''):
            return
        query_list = parse_qsl(parsed_retpath.query)
        query_list.append(('redirect', '0'))
        query = urlencode(query_list)
        parsed_retpath = list(parsed_retpath)
        parsed_retpath[_QUERY_PARSED_URL_INDEX] = query
        new_retpath = urlunparse(parsed_retpath)
        if isinstance(retpath, six.text_type):
            new_retpath = smart_text(new_retpath)
        self.form_values['retpath'] = new_retpath

    def set_limited_oauth_token_for_music_to_response(self):
        # Здесь выдаётся обычный токен с музыкальными скоупами,
        # а не X-токен. Если начать выдавать X-токены, то мобильные
        # приложения МТС.Музыки не могут залогинить пользователя (проверено
        # 31.10.2016 - 1.11.2016 в проде).
        # Кроме этого, выдаваемый токен сохраняется в куках на домене
        # music.mts.ru, т.е. не на яндексовом домене. А отдавать X-токен в
        # третьи руки -- ОПАСНО ДЛЯ ЖИЗНИ!
        provider_code = self.provider['code'].lower()
        oauth_credentials = settings.OAUTH_APPLICATIONS_FOR_MUSIC[provider_code]
        self.generate_oauth_token(oauth_credentials)

    def get_task_by_token(self):
        try:
            return self.social_broker.get_task_by_token(
                self.form_values['provider'],
                self.form_values['application'],
                'passport',
                self.form_values['provider_token'],
                provider_token_secret=self.form_values.get('provider_token_secret'),
                scope=self.form_values.get('scope'),
            )
        except social_broker_exceptions.SocialBrokerInvalidTokenError:
            raise InvalidProviderTokenError()
        except social_broker_exceptions.SocialBrokerApplicationUnknownError:
            raise ApplicationInvalidError()
        except social_broker_exceptions.SocialBrokerProviderUnknownError:
            raise ProviderInvalidError()

    def select_account_to_login(self, accounts):
        # Выбираем аккаунт с наибольшим uid
        return sorted(accounts, key=lambda a: a.uid)[-1]

    def respond_success(self):
        if self.track and self.track.process_uuid:
            self.response_values.setdefault('process_uuid', self.track.process_uuid)
        return super(BaseSocialAuthView, self).respond_success()

    def is_suggest_related_account_enabled(self):
        """
        Метод определяет может ли ручка предложить пользователю залогиниться в
        связанный через E-Mail аккаунт, когда не нашлось ни одного аккаунта в
        который можно войти через данный соц. профиль.
        """
        raise NotImplementedError()  # pragma: no cover

    def check_if_challenge_required(self):
        if not settings.CHALLENGE_ON_SOCIAL_AUTH_ENABLED:
            return

        self.setup_antifraud_features()

        business_info = BusinessInfo.from_dict(self.profile.get('business'))
        social_userid = business_info.to_userid() if business_info else self.profile['userid']
        self.antifraud_features.add_social_auth_features(
            social_provider_code=self.provider['code'],
            social_userid=social_userid,
        )

        if self.is_mobile_auth():
            mobile_profile = MobileProfile()
            mobile_profile.setup(self.track, self.account.uid)
        else:
            mobile_profile = None

        self.antifraud_features.is_mobile = self.is_mobile_auth()
        self.antifraud_features.add_lah_cookie_uids(self.request.env.cookies.get('lah'))

        # Нужно для
        # * Ручки челленджа, чтобы знать какому аккаунту показать челлендж
        # * Ручки завершения соц. авторизации, чтобы знать какой аккаунт
        #   авторизовывать.
        self.track.uid = self.account.uid

        try:
            state = self.show_challenge_if_necessary(
                # Разрешаем соц. вход, если единственный доступный челлендж --
                # это CAPTCHA. Делаем так, чтобы фронту не приходилось
                # обрабатывать CAPTCHA особым образом. В будущем планируем
                # сделать CAPTCHA, а не отказом.
                allow_captcha=False,
                allow_new_challenge=True,
                mobile_profile=mobile_profile,
            )
            if state is not None:
                self.state = state
                return state
        except PasswordNotMatchedError:
            self.track.uid = None
            raise AuthNotAllowedError()

    def is_mobile_auth(self):
        return bool(self.track.device_id)


class SocialAuthenticatorMixin(BundleOAuthMixin):
    def generate_oauth_token(self, application, is_x_token=True):
        device_and_track = {}
        if self.track:
            device_and_track.update(self.track_to_oauth_params(self.get_device_params_from_track()))
            device_and_track.update({'passport_track_id': self.track.track_id})

        token_response = self.oauth.token_by_uid(
            application['client_id'],
            application['client_secret'],
            self.account.uid,
            self.request.env.user_ip,
            **device_and_track
        )
        access_token = token_response.get('access_token')
        if not access_token:
            log.error('Failed to get access_token: %s', token_response)
            error = token_response.get('error')
            description = token_response.get('error_description', 'No `access_token` in response found')
            if error == OAUTH_INVALID_GRANT_ERROR and description == OAUTH_FORBIDDEN_ACCOUNT_TYPE_ERROR_DESCRIPTION:
                raise AuthNotAllowedError()
            else:
                raise OAuthPermanentError(description)

        if is_x_token:
            self.response_values['x_token'] = access_token
        # В будущем лучше отказаться от указания в ответе ручки что токен имеет
        # грант X-Токен, поэтому поле token заполняется всегда, а x_token для
        # обратной совместимости.
        self.response_values['token'] = access_token

        if token_response.get('expires_in') is not None:
            self.response_values['token_expires_in'] = token_response.get('expires_in')

    def generate_authorization_code(self, application):
        try:
            authorization_code = self.oauth_authorization_code_by_uid(
                code_strength=OAUTH_CODE_STRENGTH_LONG,
                ttl=settings.AM_CODE_TTL,
                require_activation=False,
                code_challenge=self.track.oauth_code_challenge,
                code_challenge_method=self.track.oauth_code_challenge_method,
                uid=self.account.uid,
                **application
            )
        except (
            OAuthInvalidClientIdError,
            OAuthInvalidClientSecretError,
        ) as e:
            log.error('Failed to generate authorization code: %s' % type(e).__name__)
            raise InternalPermanentError()
        self.response_values['yandex_authorization_code'] = authorization_code

    def statbox_user_authenticated_with_social_profile(self):
        self.statbox.log(
            action='auth',
            provider=self.provider['name'],
            userid=self.profile.get('userid'),
            profile_id=self.profile_id,
            login=self.account.login,
            uid=self.account.uid,
            ip=self.client_ip,
        )

    def social_app_is_third_party(self):
        app_attrs = self.task_data['token']['application_attributes']
        return app_attrs['third_party']

    def fill_response_with_auth_data(self):
        """
        Завершение социальной авторизации: обновление авторизационного профиля,
        обновление данных аккаунта (если он не заблокирован), генерация кук или
        иных авторизационных данных.

        В процессе происходит модификация полей трека.
        """
        self.statbox.bind_context(login=self.account.login, uid=self.account.uid)
        process_env_profile(self.account, track=self.track)
        self.statbox_user_authenticated_with_social_profile()

        self.fill_track_and_response_with_auth_data()

        write_phone_to_log(self.account, self.cookies)
        self.state = RedirectToRetpath()

    def fill_track_and_response_with_auth_data(self):
        raise NotImplementedError()  # pragma: no cover


class FullSocialAuthenticatorMixin(SocialAuthenticatorMixin):
    def set_authorization_to_track(self, old_session_info=None):
        set_authorization_track_fields(
            self.account,
            self.track,
            allow_create_session=self.track.social_output_mode != OUTPUT_MODE_XTOKEN,
            allow_create_token=self.track.social_output_mode == OUTPUT_MODE_XTOKEN,
            old_session_info=old_session_info,
            password_passed=False,
            session_scope=SessionScope.xsession,
        )

        provider_code = self.task_data['profile']['provider']['code']
        self.track.auth_method = settings.AUTH_METHOD_SOCIAL_TEMPLATE % provider_code.lower()

    def set_authentication_cookies_to_response(self, old_session_info):
        extend_session = old_session_info.cookie_status != CookieCheckStatus.Invalid
        multi_session_users = users_from_multisession(old_session_info)

        if (
            extend_session and
            not old_session_info.response.get('allow_more_users', True) and
            self.account.uid not in multi_session_users
        ):
            raise AuthSessionOverflowError()

        try:
            cookies, _, service_guard_container = build_auth_cookies_and_session(
                self.request.env,
                self.track,
                account_type=self.account.type,
                is_yandexoid=self.account.is_yandexoid,
                is_betatester=self.account.is_betatester,
                social_id=self.account.originated_from_profile_id,
                extend_session=extend_session,
                multi_session_users=multi_session_users,
                display_name=self.account.person.display_name,
                logout_datetime=self.account.web_sessions_logout_datetime,
                need_extra_sessguard=True,
                is_child=self.account.is_child,
            )
        except AccountGlobalLogoutError:
            raise BundleAccountGlobalLogoutError()
        except SessionExpiredError:
            raise AuthSessionExpiredError()

        if service_guard_container:
            self.response_values.update({
                'service_guard_container': service_guard_container.pack(),
            })

        self.response_values.update({
            'cookies': cookies,
            'default_uid': self.account.uid,
            'sensitive_fields': ['cookies'],
        })

    def check_social_app_permitted_to_get_auth(self):
        if self.social_app_is_third_party():
            raise ApplicationInvalidError()

    def fill_track_and_response_with_auth_data(self):
        mode = self.track.social_output_mode
        if mode == OUTPUT_MODE_XTOKEN:
            self.set_authorization_to_track()
            self.generate_oauth_token(settings.OAUTH_APPLICATION_AM_XTOKEN)

        elif mode == OUTPUT_MODE_SESSIONID:
            cookie_session_info = self.check_session_cookie(dbfields=[])
            self.set_authorization_to_track(old_session_info=cookie_session_info)
            self.set_authentication_cookies_to_response(old_session_info=cookie_session_info)

        elif mode == OUTPUT_MODE_AUTHORIZATION_CODE:
            self.set_authorization_to_track()
            self.generate_authorization_code(settings.OAUTH_APPLICATION_AM_XTOKEN)

    def check_track_output_mode_compatible(self):
        pass


class ThirdPartySocialAuthenticatorMixin(SocialAuthenticatorMixin):
    def check_social_app_permitted_to_get_auth(self):
        app_attrs = self.task_data['token']['application_attributes']
        if not (
            'related_yandex_client_id' in app_attrs and
            'related_yandex_client_secret' in app_attrs
        ):
            raise ApplicationInvalidError()

    def get_oauth_credentials_from_social_application(self):
        app_attrs = self.task_data['token']['application_attributes']
        return dict(
            client_id=app_attrs['related_yandex_client_id'],
            client_secret=app_attrs['related_yandex_client_secret'],
        )

    def fill_track_and_response_with_auth_data(self):
        mode = self.track.social_output_mode
        if mode == OUTPUT_MODE_XTOKEN:
            oauth_credentials = self.get_oauth_credentials_from_social_application()
            self.generate_oauth_token(oauth_credentials, is_x_token=False)

    def check_track_output_mode_compatible(self):
        """
        Не во всех режимах социальной авторизации можно выписать средство
        аутентикации с ограниченными возможностями (например, сессия даёт
        неограниченные возможности).
        """
        if self.track.social_output_mode != OUTPUT_MODE_XTOKEN:
            raise InvalidTrackStateError()


class RegisterBaseView(
    BaseSocialAuthView,
    BundleFrodoMixin,
    MailSubscriptionsMixin,
):
    def create_account(self, login):
        if self.track and self.track.is_fake_client:
            default_person_info = build_empty_person_info()
        else:
            default_person_info = build_default_person_registration_info(self.client_ip)

        profile = dict(self.profile)
        profile.update(
            firstname=self.firstname,
            lastname=self.lastname,
        )

        default_account_ = default_account(
            login,
            datetime.now(),
            profile,
            default_person_info,
            alias_type='social',
        )

        if self.should_bind_simple_phone(self.profile):
            save_simple_phone = self.build_save_simple_phone(
                account=default_account_,
                phone_number=PhoneNumber.parse(self.profile['phone']),
                is_new_account=True,
            )
            save_simple_phone.submit()
        else:
            save_simple_phone = None

        events = {'action': 'account_register_social', 'consumer': self.consumer}
        with CREATE(default_account_, self.request.env, events) as self.account:
            add_subscription(
                self.account,
                service=get_service(slug='social'),
                login=login,
            )
            self.bind_email_from_profile(self.account)
            self.unsubscribe_from_maillists_if_nessesary()

            if save_simple_phone:
                save_simple_phone.commit()

        if save_simple_phone:
            save_simple_phone.after_commit()

    def should_bind_simple_phone(self, profile):
        return (profile.get('phone') and
                profile['provider']['code'] in settings.SOCIAL_TRUSTED_SIMPLE_PHONE)

    def check_firstname_and_lastname(self):
        """
        Проверяем наличие имени и фамилии в форме или соц. профиле.
        В некоторых случаях имя и/или фамилия не обязательны, это решает брокер.
        """
        firstname_required = not self.provider.get('is_firstname_optional', False)
        lastname_required = not self.provider.get('is_lastname_optional', False)
        if (
            firstname_required and not self.firstname or
            lastname_required and not self.lastname
        ):
            links = self.profile.get('links')
            if links:
                self.response_values['profile_link'] = links[0]
            raise NameRequiredError()

    def bind_email_from_profile(self, account):
        address = self.profile.get('email')
        if not address:
            return
        timestamp = datetime.now()
        email = Email(
            address=address,
            created_at=timestamp,
            confirmed_at=timestamp,
            bound_at=timestamp,
            is_unsafe=True,
        )
        account.emails.add(email)

    def update_display_name(self):
        """
        Генерируем отображаемое имя пользователя.
        """
        if not self.firstname and not self.lastname:
            return

        if self.firstname and self.lastname:
            name = '%s %s' % (self.firstname, self.lastname)
        else:
            name = self.lastname or self.firstname

        display_name = DisplayName(
            name=name,
            provider=self.provider['code'],
            profile_id=self.profile_id,
        )
        with UPDATE(self.account, self.request.env, {'action': 'change_display_name', 'consumer': self.consumer}):
            self.account.person.display_name = display_name
            # PASSP-23241: если True, то на стороне ЧЯ public name вычисляется мимо display name
            # напрямую из firstname и lastname учётной записи
            self.account.person.dont_use_displayname_as_public_name = True

        # перегенерируем с новыми данными
        self.response_values['account'] = self.get_account_short_info(self.account)

    def set_user_default_avatar_async(self):
        """
        Пишем в специальный лог, чтобы потом (в логброкер-клиенте) установить пользователю default avatar на
        основе полученных от SocialBroker данных. Все ошибки пишем в лог и игнорируем.
        """
        avatar_to_upload = get_max_size_avatar_from_profile_fail_safe(self.profile)
        if avatar_to_upload:
            upload_avatar_async(
                uid=self.account.uid,
                avatar_url=avatar_to_upload,
                user_ip=self.client_ip,
            )

    @cached_property
    def firstname(self):
        return self.form_values.get('firstname') or self.profile.get('firstname')

    @cached_property
    def lastname(self):
        return self.form_values.get('lastname') or self.profile.get('lastname')
