# -*- coding: utf-8 -*-
import logging

from passport.backend.api.common.login import build_available_social_login
from passport.backend.api.common.logs import setup_log_prefix
from passport.backend.api.forms.base import DeviceInfoForm
from passport.backend.api.views.bundle.auth.exceptions import AuthNotAllowedError
from passport.backend.api.views.bundle.exceptions import (
    AccountAlreadyRegisteredError,
    AccountInvalidTypeError,
    AccountNotFoundError,
    EulaIsNotAcceptedError,
    InternalTemporaryError,
    InvalidTrackStateError,
)
from passport.backend.api.views.bundle.headers import (
    HEADER_CLIENT_COOKIE,
    HEADER_CLIENT_HOST,
    HEADER_CLIENT_USER_AGENT,
    HEADER_CONSUMER_CLIENT_IP,
)
from passport.backend.api.views.bundle.utils import assert_valid_host
from passport.backend.core.builders.social_api import exceptions as social_api_exceptions
from passport.backend.core.builders.social_api.profiles import convert_task_profile
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.request_id import get_request_id
from passport.backend.core.runner.context_managers import UPDATE
from passport.backend.core.types.account.account import ACCOUNT_TYPE_SOCIAL
from passport.backend.core.types.phone_number.phone_number import PhoneNumber
from passport.backend.utils.string import smart_str
from six.moves.urllib.parse import (
    parse_qsl,
    urlencode,
    urlparse,
    urlunparse,
)

from .base import (
    BaseSocialAuthView,
    FullSocialAuthenticatorMixin,
    OUTPUT_MODE_AUTHORIZATION_CODE,
    OUTPUT_MODE_SESSIONID,
    OUTPUT_MODE_XTOKEN,
    RegisterBaseView,
    SocialAuthenticatorMixin,
    ThirdPartySocialAuthenticatorMixin,
)
from .exceptions import (
    BrokerFailedError,
    ProviderInvalidError,
    UidRejectedError,
)
from .forms import (
    CallbackForm,
    ChooseForm,
    NativeRetpathForm,
    NativeStartForm,
    NativeStartFormV2,
    RegisterByTaskForm,
    RegisterByTokenForm,
    RegisterForm,
    RetpathForm,
    StartForm,
)
from .states import (
    RedirectToBrokerWithoutAuth,
    RedirectToRetpath,
)


log = logging.getLogger('passport.api.view.bundle.auth.social')


class StartView(BaseSocialAuthView):
    """
    Точка начала соц. авторизации.
    Потребитель приходит со всеми нужными данными, создаем редирект в соц. брокер.
    """
    basic_form = StartForm

    def redirect_without_auth(self, output_mode):
        self.state = RedirectToBrokerWithoutAuth()
        self.response_values['track_id'] = self.track_id
        if output_mode == OUTPUT_MODE_AUTHORIZATION_CODE:
            # Заставляем соцпровайдера всегда показывать диалог авторизации,
            # чтобы защититься от незаметного сбора OAuth-токенов злоумышленником
            self.response_values['force_prompt'] = True
        self.statbox.bind(track_id=self.track_id)

    def process_request(self):
        assert_valid_host(self.request.env)

        self._process_retpath_form()
        self._fix_morda_retpath()

        self.process_basic_form()
        self._fix_morda_retpath()

        self.statbox.bind(
            action='submitted',
            type='social',
            retpath=self.form_values['retpath'],
            provider=self.form_values['provider'],
            application=self.form_values['application'],
            origin=self.form_values['origin'],
            ip=self.client_ip,
            host=self.host,
            user_agent=self.user_agent,
            yandexuid=self.cookies.get('yandexuid'),
            broker_consumer=self.form_values['broker_consumer'],
            # Сохраняем старый track_id, чтобы можно было потом при обработке логов соединить "старый" процесс
            # и процесс социальной авторизации
            old_track_id=self.track_id,
            process_uuid=self.form_values.get('process_uuid'),
        )

        service = self.form_values['service']
        if service:
            self.statbox.bind({'from': service.slug})

        code_challenge = self.form_values['code_challenge']
        issue_auth_code = (
            code_challenge and
            self.form_values['provider'] in settings.SOCIAL_PROVIDERS_TO_USE_EXTERNAL_USERAGENT
        )
        output_mode = OUTPUT_MODE_AUTHORIZATION_CODE if issue_auth_code else OUTPUT_MODE_SESSIONID

        log.info('Creating and initializing a track...')
        self.initialize_track(
            code_challenge=code_challenge,
            code_challenge_method=self.form_values['code_challenge_method'],
            social_output_mode=output_mode,
        )

        device_info = self.process_form(DeviceInfoForm, self.all_values)
        self.save_device_params_to_track(device_info)

        # В контексте мультиавторизации ВСЕГДА дописываем нового пользователя
        # Это значит, что отправим в брокер с retpath на наш callback
        # TODO: поменять этот код, т.к. выглядит странно, если авторизация есть
        self.redirect_without_auth(output_mode=output_mode)

        self.statbox.log(state=self.state.state)

    def respond_error(self, exception):
        self._fill_response_with_retpath()
        return super(StartView, self).respond_error(exception)

    def _process_retpath_form(self):
        values = self.process_form(RetpathForm(), self.all_values)
        self.form_values.update(values)


class SocialSuggestEnabledMixin(object):
    def is_suggest_related_account_enabled(self):
        if not settings.SOCIAL_SUGGEST_RELATED_ACCOUNT_ENABLED:
            return False
        if not self.track.social_broker_consumer:
            return False
        return any(e.match(self.track.social_broker_consumer) for e in settings.SOCIAL_SUGGEST_RELATED_ACCOUNT_CONSUMERS_RE_LIST)


class NativeStartView(
    SocialSuggestEnabledMixin,
    FullSocialAuthenticatorMixin,
    BaseSocialAuthView,
):
    basic_form = NativeStartForm
    antifraud_auth_type = 'social_native_start'

    def process_request(self):
        self._process_retpath_form()
        self._fix_morda_retpath()
        self._fill_response_with_retpath()

        self.process_basic_form()
        self._fix_morda_retpath()

        self.initialize_track(OUTPUT_MODE_XTOKEN)

        device_info = self.process_form(DeviceInfoForm, self.all_values)
        self.save_device_params_to_track(device_info)

        self.native_start_submitted_to_statbox()

        self.fill_response_with_track_fields()

        task_data = self.get_task_by_token()

        self.statbox.bind(action='callback_end')

        with self.track_transaction.rollback_on_error():
            self.track.social_task_data = task_data
            self.track.social_task_id = task_data['task_id']
            self.load_task_data()

        self.check_social_app_permitted_to_get_auth()
        self.load_accounts_and_select_state()
        if self.account:
            setup_log_prefix(self.account)

        self.statbox_user_come_back_with_task()

        should_authorize = len(self.accounts) == 1
        if should_authorize:
            with self.track_transaction.commit_on_error():
                redirect_state = (
                    self.check_user_policies() or
                    self.check_if_challenge_required()
                )
                if redirect_state is not None:
                    return

            with self.track_transaction.rollback_on_error():
                self.bind_profile_to_enabled_account()
                self.track.surface = 'native_social'
                self.fill_response_with_auth_data()
                self.update_subscriptions_and_statistics()
                self.track.is_successful_completed = True

    def native_start_submitted_to_statbox(self):
        self.statbox.log(
            action='submitted',
            type='social_native',
            retpath=self.form_values['retpath'],
            provider=self.form_values['provider'],
            application=self.form_values['application'],
            origin=self.form_values['origin'],
            ip=self.client_ip,
            host=self.host,
            user_agent=self.user_agent,
            track_id=self.track_id,
            yandexuid=self.cookies.get('yandexuid'),
        )

    def _process_retpath_form(self):
        values = self.process_form(NativeRetpathForm(), self.all_values)
        self.form_values.update(values)


class ThirdPartyNativeStartView(ThirdPartySocialAuthenticatorMixin, NativeStartView):
    basic_form = NativeStartFormV2

    def is_suggest_related_account_enabled(self):
        return False

    def check_if_challenge_required(self):
        """
        Сейчас в процессе челленджа выписывается сессия, но в third party-ручках
        нельзя выписывать сессии и X-токены.
        """


class CallbackView(
    SocialSuggestEnabledMixin,
    FullSocialAuthenticatorMixin,
    BaseSocialAuthView,
):
    """
    Пользователь вернулся из брокера с каким-то status, track_id и task_id.
    Идем в social.api, берем профили, фильтруем, определяемся с действием и отдаем фронтенду все необходимое.
    """
    require_track = True
    basic_form = CallbackForm
    antifraud_auth_type = 'social_callback'

    def should_authorize(self):
        return len(self.accounts) == 1

    def check_track(self):
        if self.track.social_task_id and self.track.social_task_id != self.form_values['task_id']:
            log.debug(
                'Track task id (%s) and form task id (%s) should be equal' %
                (self.track.social_task_id, self.form_values['task_id']),
            )
            raise InvalidTrackStateError()

    def process_request(self):
        self.process_basic_form()

        self.read_track()
        self.fill_response_with_track_fields()
        self.check_host_if_required()

        self.check_track()
        self.check_track_output_mode_compatible()

        self.statbox.log(
            action='callback_begin',
            broker_status=self.form_values['status'],
            task_id=self.form_values['task_id'],
        )

        if self.form_values['status'] != 'ok':
            retpath = self._build_failure_retpath()
            if retpath:
                self.response_values.update({'retpath': retpath})
            raise BrokerFailedError()

        self.statbox.bind(action='callback_end')

        with self.track_transaction.rollback_on_error():
            self.track.social_task_id = self.form_values['task_id']
            self.load_task_data()

        self.check_social_app_permitted_to_get_auth()
        self.load_accounts_and_select_state(suggest_uid=self.form_values['uid'])
        if not self.form_values['uid'] and self.account:
            setup_log_prefix(self.account)

        self.statbox_user_come_back_with_task()

        if self.should_authorize():
            with self.track_transaction.commit_on_error():
                redirect_state = (
                    self.check_user_policies() or
                    self.check_if_challenge_required()
                )
                if redirect_state is not None:
                    return

            with self.track_transaction.rollback_on_error():
                self.bind_profile_to_enabled_account()
                self.fill_response_with_auth_data()
                self.update_subscriptions_and_statistics()
                self.track.is_successful_completed = True

    def _build_failure_retpath(self):
        if not self.track.retpath:
            return

        args = [('status', 'error')]
        error_code = self.form_values.get('code')
        if error_code:
            args.append(('code', error_code))
        args.append(('request_id', get_request_id()))

        if self.track.social_place == 'fragment':
            # fragment
            url_bit_index = 5
        else:
            # query
            url_bit_index = 4
        parsed_url = list(urlparse(smart_str(self.track.retpath)))
        old_args = parsed_url[url_bit_index]
        old_args = parse_qsl(old_args)
        args = old_args + args
        args = urlencode(args)
        parsed_url[url_bit_index] = args
        return urlunparse(parsed_url)


class SecureCallbackView(CallbackView):
    required_headers = CallbackView.required_headers + [
        HEADER_CLIENT_HOST,
        HEADER_CLIENT_COOKIE,
    ]

    def should_authorize(self):
        return True

    def check_track(self):
        if not self.track.uid:
            raise InvalidTrackStateError()

    def load_accounts_and_select_state(self, **kwargs):
        self.accounts = [
            account
            for account in self.get_accounts_for_profile(log_to_statbox=True)
            if account.uid == int(self.track.uid)
        ]

        if not self.accounts:
            # Не нашли аккаунта для данного профиля с уидом из трека
            raise UidRejectedError()

        # Авторизуем пользователя.
        self.account = self.accounts[0]
        self.state = RedirectToRetpath()
        setup_log_prefix(self.account)


class ChooseView(FullSocialAuthenticatorMixin, BaseSocialAuthView):
    """
    Если в результате вызова /callback оказалось несколько доступных аккаунтов,
    то пользователю отрисовалась страница выбора, и сюда приходит UID одного из них.
    Проверим пригодность этого UID для авторизации и авторизуем.
    """
    require_track = True
    basic_form = ChooseForm
    antifraud_auth_type = 'social_choose'

    def process_request(self):
        self.process_basic_form()
        self.read_track()
        self.fill_response_with_track_fields()
        self.check_host_if_required()
        self.check_track_output_mode_compatible()
        self.load_task_data_from_track()
        self.check_social_app_permitted_to_get_auth()

        # Еще раз получим список доступных профилей, выберем для них аккаунты.
        self.accounts = self.get_accounts_for_profile()
        accounts_by_uid = {account.uid: account for account in self.accounts}
        uid = self.form_values['uid']
        if uid not in accounts_by_uid:
            # Эта ситуация может произойти в результате подмены uid в запросе или в результате
            # конкурирующих действий пользователя.
            raise UidRejectedError()

        self.account = accounts_by_uid[uid]

        with self.track_transaction.commit_on_error():
            redirect_state = (
                self.check_user_policies() or
                self.check_if_challenge_required()
            )
            if redirect_state is not None:
                return

        with self.track_transaction.rollback_on_error():
            self.bind_profile_to_enabled_account()
            self.fill_response_with_auth_data()
            self.update_subscriptions_and_statistics()
            self.track.is_successful_completed = True


class ThirdPartyChooseView(ThirdPartySocialAuthenticatorMixin, ChooseView):
    def check_if_challenge_required(self):
        """
        Сейчас в процессе челленджа выписывается сессия, но в third party-ручках
        нельзя выписывать сессии и X-токены.
        """


class RegisterInfoView(BaseSocialAuthView):
    """
    Если в процессе авторизации аккаунт оказался заблокированным и у пользователя нет
    незаблокированных аккаунтов, то отправляем его на регистрацию.
    Технически это выглядит так:
    пользователь приходит в фронт на GET /register, зная только track_id,
    фронт идет в эту ручку, получает все необходимые данные и отрисовывает страницу регистрации,
    дальше все как при обычном status=register.
    """
    require_track = True

    def process_request(self):
        self.read_track()
        self.fill_response_with_track_fields()
        # Если social_task_data не заполнен - значит не была произведена загрузка данных из
        # SocialApi на /callback, тогда делать тут нам нечего.
        self.load_task_data_from_track()


class RegisterView(FullSocialAuthenticatorMixin, RegisterBaseView):
    """
    Если в результате вызова /callback не нашли доступных аккаунтов, то просим у пользователя подтверждения
    и создаем аккаунт, далее авторизуем.
    Также выполняем проверку check_spammer.
    """
    require_track = True
    basic_form = RegisterForm

    def process_request(self):
        self.process_basic_form()
        self.read_track()
        self.fill_response_with_track_fields()
        self.check_host_if_required()

        if self.track.is_successful_completed and not self.track.social_register_uid:
            log.debug('Completed social registration track should contain social register uid')
            raise InvalidTrackStateError()

        self.check_track_output_mode_compatible()
        self.load_task_data()
        self.check_social_app_permitted_to_get_auth()

        if not self.form_values['eula_accepted']:
            raise EulaIsNotAcceptedError()

        if not self.track.is_captcha_ignored:
            with self.track_transaction.rollback_on_error():
                self.is_socialreg_captcha_required()
                self.check_track_for_captcha(log_fail_to_statbox=False)

        if not self.track.is_successful_completed:
            self.register_social_account()
        else:
            self.login_to_registered_social_account()

        self.response_values['has_enabled_accounts'] = True

    def register_social_account(self):
        # Эта проверка нужна в таком случае:
        # Человек в двух вкладках начал соц авторизацию, дошел до подтверждения ПС,
        # затем в обоих вкладках нажал на "продолжить" и мы попадаем сюда, будет создано два аккаунта.
        # Для предотвращения этого идем в SocialApi и проверяем текущее состояние дел.
        # Здесь тоже возможны логические гонки, но маловероятны и в таких масштабах не страшны.
        self.accounts = self.get_accounts_for_profile(
            exclude_disabled=True,
            need_emails=False,
            need_phones=False,
        )
        if self.accounts:
            raise AccountAlreadyRegisteredError()

        self.check_firstname_and_lastname()

        login = build_available_social_login(settings.SOCIAL_LOGIN_GENERATION_RETRIES, self.request.env)

        self.create_account(login)

        self.frodo_check_spammer(
            action='admsocialreg',
            increment_counters=not self.track.is_fake_client,
            service=self.consumer,
            social_provider=self.provider['code'],
            firstname=self.account.person.firstname,
            lastname=self.account.person.lastname,
        )

        self.bind_profile_to_enabled_account()
        self.update_display_name()
        self.set_user_default_avatar_async()

        self.statbox.log(
            action='account_created',
            captcha_generation_number=self.track.captcha_generate_count.get(default=0),
            captcha_key=self.track.captcha_key,
            city=self.account.person.city or None,
            country=self.account.person.country or None,
            is_voice_generated=bool(self.track.voice_captcha_type),
            karma=self.account.karma.value,
            language=self.track.language,
            login=self.account.normalized_login,
            provider=self.provider.get('name'),
            retpath=self.track.retpath,
            uid=self.account.uid,
            user_agent=self.request.env.user_agent,
            userid=self.profile.get('userid'),
        )

        with self.track_transaction.rollback_on_error():
            self.fill_response_with_auth_data()
            self.update_subscriptions_and_statistics()
            self.track.is_successful_completed = True
            self.track.social_register_uid = self.account.uid

    def login_to_registered_social_account(self):
        if not (
            self.track.uid and
            self.track.social_register_uid and
            int(self.track.uid) == int(self.track.social_register_uid)
        ):
            log.debug(
                'Track uid (%s) and social register uid (%s) should be equal' %
                (self.track.uid, self.track.social_register_uid),
            )
            raise InvalidTrackStateError()

        try:
            self.get_account_from_track()
        except AccountNotFoundError:
            log.debug('New social account not found: %s' % self.track.uid)
            # Скорее всего данные о новом аккаунте ещё не достигли реплики,
            # нужно попросить фронтенд повторить запрос.
            raise InternalTemporaryError()

        if not self.account.person.display_name.profile_id:
            log.debug('New social account does not have profile id: %s' % self.track.uid)
            # Скорее всего данные о новом аккаунте ещё не достигли реплики,
            # нужно попросить фронтенд повторить запрос.
            raise InternalTemporaryError()
        self.profile_id = self.account.person.display_name.profile_id

        with self.track_transaction.rollback_on_error():
            self.fill_response_with_auth_data()

        self.response_values.update(
            account=self.get_account_short_info(self.account),
            profile_id=self.account.person.display_name.profile_id,
        )

class ThirdPartyRegisterView(ThirdPartySocialAuthenticatorMixin, RegisterView):
    pass


class RegisterByTaskView(SocialAuthenticatorMixin, RegisterBaseView):
    """
    Отдельная независимая ручка, которая по брокерному task_id создаст аккаунт
    или найдет существующий, и для него выпишет OAuth токен яндексовый.

    Ручка не проверяет и не выписывает куки, поэтому HOST можно не
    проверять на действительность.
    """
    required_grants = ['auth_social.register_by_task_id']
    require_track = False
    basic_form = RegisterByTaskForm

    def process_request(self):
        self.process_basic_form()

        self.load_task_data(use_track=False, allowed_providers=settings.REGISTER_BY_TASK_PROVIDERS)

        self.accounts = self.get_accounts_for_profile(
            exclude_disabled=True,
            need_emails=False,
        )
        if self.accounts:
            self.account = self.select_account_to_login(self.accounts)
            setup_log_prefix(self.account)
            if self.account.type != ACCOUNT_TYPE_SOCIAL or self.account.have_password:
                raise AccountInvalidTypeError()

            phone_number = self.profile.get('phone')
            if (phone_number and
                not self.account.phones.by_number(phone_number) and
                    self.should_bind_simple_phone(self.profile)):
                # Т.к. раньше в момент регистрации к социальщикам
                # не привязывался телефон, допривяжем его если нужно.
                save_simple_phone = self.build_save_simple_phone(
                    phone_number=PhoneNumber.parse(phone_number),
                )
                save_simple_phone.submit()
                with UPDATE(
                    self.account,
                    self.request.env,
                    {'action': 'bind_phone_to_social', 'consumer': self.consumer},
                ):
                    save_simple_phone.commit()
                save_simple_phone.after_commit()
        else:
            login = build_available_social_login(settings.SOCIAL_LOGIN_GENERATION_RETRIES, self.request.env)
            self.create_account(login)

        self.frodo_check_spammer(
            action='admsocialreg',
            increment_counters=True,
            service=self.consumer,
            social_provider=self.provider['code'],
            firstname=self.account.person.firstname,
            lastname=self.account.person.lastname,
        )

        self.bind_profile_to_enabled_account(task_id=self.request_values['task_id'])
        self.update_display_name()

        self.statbox.log(
            action='account_created',
            city=self.account.person.city or None,
            country=self.account.person.country or None,
            karma=self.account.karma.value,
            login=self.account.normalized_login,
            provider=self.provider.get('name'),
            uid=self.account.uid,
            user_agent=self.request.env.user_agent,
            userid=self.profile.get('userid'),
        )

        self.fill_response_with_auth_data()

        # Удаляем брокерный таск, чтобы предотвратить повторные вызовы ручки.
        try:
            self.social_api.delete_task(self.request_values['task_id'])
        except social_api_exceptions.BaseSocialApiError:
            # Если не получилось - не падаем
            log.exception('Failed to delete task, ignoring')

    def fill_track_and_response_with_auth_data(self):
        self.set_limited_oauth_token_for_music_to_response()


class RegisterByTokenView(SocialAuthenticatorMixin, RegisterBaseView):
    """
    Отдельная независимая ручка, которая по токену mts музыки создаст (кажется, что он уже гарантированно должнен быть) аккаунт
    или найдет существующий, и для него выпишет OAuth токен яндексовый (яндекс музыки).

    Ручка не проверяет и не выписывает куки, поэтому HOST можно не
    проверять на действительность.
    """
    required_grants = ['auth_social.register_by_token']
    require_track = False
    basic_form = RegisterByTokenForm
    required_headers = [
        HEADER_CLIENT_USER_AGENT,
        HEADER_CONSUMER_CLIENT_IP,
    ]

    def process_request(self):
        self.process_basic_form()

        if self.form_values['provider'].lower() not in settings.REGISTER_BY_TOKEN_PROVIDERS:
            raise ProviderInvalidError()
        self.task_data = self.get_task_by_token()
        self.profile = convert_task_profile(self.task_data['profile'])
        self.provider = self.profile['provider']

        self.accounts = self.get_accounts_for_profile(exclude_disabled=True)
        if self.accounts:
            self.account = self.select_account_to_login(self.accounts)
            setup_log_prefix(self.account)

            if self.account.type != ACCOUNT_TYPE_SOCIAL or self.account.have_password:
                raise AccountInvalidTypeError()

            self.bind_profile_to_enabled_account(task_id=self.task_data['task_id'])
        else:
            login = build_available_social_login(settings.SOCIAL_LOGIN_GENERATION_RETRIES, self.request.env)
            self.create_account(login)

            self.frodo_check_spammer(
                action='admsocialreg',
                increment_counters=True,
                service=self.consumer,
                social_provider=self.provider['code'],
                firstname=self.account.person.firstname,
                lastname=self.account.person.lastname,
            )

            self.bind_profile_to_enabled_account(task_id=self.task_data['task_id'])
            self.update_display_name()

            self.statbox.log(
                action='account_created',
                city=self.account.person.city or None,
                country=self.account.person.country or None,
                karma=self.account.karma.value,
                login=self.account.normalized_login,
                provider=self.provider.get('name'),
                uid=self.account.uid,
                user_agent=self.request.env.user_agent,
                userid=self.profile.get('userid'),
            )

        self.set_limited_oauth_token_for_music_to_response()

    def respond_success(self):
        response_values = {
            'access_token': self.response_values['token'],
        }
        if self.response_values.get('token_expires_in') is not None:
            response_values.update(expires_in=self.response_values['token_expires_in'])
        self.response_values = response_values
        self.state = None
        return super(RegisterByTokenView, self).respond_success()

    def respond_error(self, exception):
        self.response_values.clear()
        return super(RegisterByTokenView, self).respond_error(exception)


class IssueCredentialsView(
    FullSocialAuthenticatorMixin,
    BaseSocialAuthView,
):
    require_track = True
    required_grants = ['auth_social.issue_credentials']

    def process_request(self):
        self.read_track()
        self.fill_response_with_track_fields()

        self.check_retpath_available()
        self.check_uid_available()
        self.check_host_if_required()
        self.check_track_output_mode_compatible()
        self.load_task_data_from_track()
        self.check_social_app_permitted_to_get_auth()
        self.load_accounts_and_select_state(suggest_uid=int(self.track.uid))

        if not (self.account and self.account.uid == int(self.track.uid)):
            raise AuthNotAllowedError()

        if not self.track.allow_authorization:
            with self.track_transaction.commit_on_error():
                redirect_state = (
                    self.check_user_policies() or
                    self.check_if_challenge_required()
                )
                if redirect_state is not None:
                    return

        with self.track_transaction.rollback_on_error():
            self.bind_profile_to_enabled_account()
            self.fill_response_with_auth_data()
            self.update_subscriptions_and_statistics()
            self.track.is_successful_completed = True

    def check_retpath_available(self):
        if not self.track.retpath:
            log.debug('Retpath not found')
            raise InvalidTrackStateError()

    def check_uid_available(self):
        if not self.track.uid:
            log.debug('Uid not found')
            raise InvalidTrackStateError()

    def is_suggest_related_account_enabled(self):
        return False


__all__ = globals().keys()
