# -*- coding: utf-8 -*-

from passport.backend.core import validators
from passport.backend.core.conf import settings
from passport.backend.core.utils.pkce import CODE_CHALLENGE_METHOD_SHA256


class RetpathForm(validators.Schema):
    def __init__(self):
        super(RetpathForm, self).__init__()

        # Если объявлять поле на форме, то неизвестно, какие настройки
        # туда проставится на самом деле. Это зависит от порядка импортов и инициализации конфигов,
        # что плохо, да и тестировать тяжело.
        self.add_retpath_validator()

    def add_retpath_validator(self):
        self.add_field(
            'retpath',
            validators.RetPath(
                not_empty=True,
                additional_allowed_schemes=settings.ALLOWED_SOCIAL_RETPATH_SCHEMES,
                additional_allowed_scheme_prefixes=settings.ALLOWED_SOCIAL_RETPATH_SCHEME_PREFIXES,
            ),
        )


class StartForm(RetpathForm):
    """
    Форма начала социальной авторизации. Принимает все параметры,
    необходимые социальному брокеру.
    """
    return_brief_profile = validators.StringBool(not_empty=True, if_missing=None, strip=True)
    place = validators.Place(if_missing=None)
    service = validators.Service(if_missing=None, ignore_unknown_service=True)

    code_challenge = validators.String(not_empty=True, if_missing=None, strip=True)
    code_challenge_method = validators.OneOf(
        [CODE_CHALLENGE_METHOD_SHA256],
        if_missing=None,
    )

    origin = validators.String(not_empty=True, if_missing=None, strip=True)
    provider = validators.String(not_empty=True, if_missing=None, strip=True)
    application = validators.String(not_empty=True, if_missing=None, strip=True)
    broker_consumer = validators.String(not_empty=True, if_missing=None, strip=True)

    chained_validators = [
        validators.RequireSet([('code_challenge_method', 'code_challenge')], allow_empty=True),
    ]

    # параметр для провязки с логами фронта
    process_uuid = validators.String(if_missing=None, strip=True)

    def __init__(self):
        super(StartForm, self).__init__()

        # Если объявлять поле на форме, то неизвестно, какие настройки
        # туда проставится на самом деле. Это зависит от порядка импортов и инициализации конфигов,
        # что плохо, да и тестировать тяжело.
        self.add_field(
            'retpath',
            validators.RetPath(
                not_empty=True,
                additional_allowed_schemes=settings.ALLOWED_SOCIAL_RETPATH_SCHEMES,
                additional_allowed_scheme_prefixes=settings.ALLOWED_SOCIAL_RETPATH_SCHEME_PREFIXES,
            ),
        )


class NativeRetpathForm(validators.Schema):
    def __init__(self):
        super(NativeRetpathForm, self).__init__()

        # Если объявлять поле на форме, то неизвестно, какие настройки
        # туда проставится на самом деле. Это зависит от порядка импортов и инициализации конфигов,
        # что плохо, да и тестировать тяжело.
        self.add_retpath_validator()

    def add_retpath_validator(self):
        self.add_field(
            'retpath',
            validators.RetPath(
                not_empty=True,
                additional_allowed_schemes=settings.ALLOWED_SOCIAL_RETPATH_SCHEMES,
            ),
        )


class NativeStartForm(NativeRetpathForm):
    """
    Форма начала социальной авторизации. Принимает все параметры,
    необходимые социальному брокеру.
    """
    place = validators.Place(if_missing=None)

    provider = validators.String(not_empty=True, if_missing=None, strip=True)
    application = validators.String(not_empty=True, if_missing=None, strip=True)

    # эти поля нужны только для записи в логи статистики
    broker_consumer = validators.String(not_empty=True, strip=True)
    origin = validators.String(not_empty=True, if_missing=None, strip=True)

    provider_token = validators.String(not_empty=True, strip=True)
    provider_token_secret = validators.String(not_empty=True, if_missing=None, strip=True)
    scope = validators.String(not_empty=True, if_missing=None, strip=True)

    process_uuid = validators.String(if_missing=None, strip=True)

    chained_validators = [validators.RequireIfMissing('application', missing='provider')]


class NativeStartFormV2(NativeStartForm):
    # Для нативной авторизации правильно требовать и идентификатор социального
    # провайдера, и client_id, т.к. оба они известны приложению, передающему
    # токен.
    provider = validators.String(not_empty=True, strip=True)
    application = validators.String(not_empty=True, strip=True)

    # Потребитель в данном случае избыточен и не надёжен. Лучше
    # идентифицировать потребителя по паре (provider, application) и
    # действительному токену.
    broker_consumer = None

    chained_validators = []


class CallbackForm(validators.Schema):
    """
    Форма для ручки callback, куда пользователь приносит task_id брокера.
    """
    task_id = validators.HexString(not_empty=False, if_missing=None, strip=True)
    status = validators.String(not_empty=True, strip=True)
    uid = validators.Uid(not_empty=False, if_missing=None)
    code = validators.String(if_missing=None, strip=True)

    # если status=='ok' - требуем наличия task_id
    chained_validators = [validators.RequireIfEquals(['task_id'], 'status', 'ok')]


class BaseRegisterForm(validators.Schema):
    unsubscribe_from_maillists = validators.StringBool(if_missing=False)
    origin = validators.String(if_missing=None)
    app_id = validators.String(if_missing=None)


class RegisterForm(BaseRegisterForm):
    """
    Форма для ручки create_account, пользователь должен согласиться с условиями EULA.
    """
    eula_accepted = validators.StringBool(not_empty=True, strip=True)
    firstname = validators.AntiFraudFirstName(if_missing=None, not_empty=True)
    lastname = validators.AntiFraudLastName(if_missing=None, not_empty=True)


class RegisterByTaskForm(BaseRegisterForm):
    """
    Форма для ручки register_by_task.
    """
    task_id = validators.String(not_empty=True, strip=True)


class ChooseForm(validators.Schema):
    """
    Форма для ручки choose, пользователь должен выбрать один UID.
    """
    uid = validators.Uid()


class RegisterByTokenForm(BaseRegisterForm):
    """
    Форма для ручки register_by_token.
    """
    provider_token = validators.String(not_empty=True, strip=True)
    # Для нативной авторизации правильно требовать и идентификатор социального
    # провайдера, и client_id, т.к. оба они известны приложению, передающему
    # токен.
    provider = validators.String(not_empty=True, strip=True)
    application = validators.String(not_empty=True, strip=True)
