# -*- coding: utf-8 -*-
from base64 import b64decode
from datetime import datetime
import logging
import zlib

from lxml.etree import XMLSyntaxError
from onelogin.saml2.auth import OneLogin_Saml2_Auth
from onelogin.saml2.constants import OneLogin_Saml2_Constants
from onelogin.saml2.errors import (
    OneLogin_Saml2_Error,
    OneLogin_Saml2_ValidationError,
)
from onelogin.saml2.logout_request import OneLogin_Saml2_Logout_Request
from onelogin.saml2.utils import OneLogin_Saml2_Utils
from onelogin.saml2.xmlparser import fromstring
from passport.backend.api.common.authorization import set_authorization_track_fields
from passport.backend.api.views.bundle.auth.base import BundleBaseAuthorizationMixin
from passport.backend.api.views.bundle.auth.sso.exceptions import (
    SamlEmailExistError,
    SamlEmailNoInResponseError,
    SamlEmailUnsupportableDomainError,
    SamlInvalidSignatureError,
    SamlRegistrationNotSupportedError,
    SamlRequestInvalidError,
    SamlResponseExternalError,
    SamlResponseInvalidError,
    SamlSettingsError,
)
from passport.backend.api.views.bundle.auth.sso.forms import (
    CommitForm,
    LogoutFederalForm,
    SubmitForm,
)
from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.exceptions import (
    AccountNotFoundError,
    DomainNotFoundError,
    EmailExistError,
    EmailUnsupportableDomainError,
)
from passport.backend.api.views.bundle.headers import (
    HEADER_CLIENT_HOST,
    HEADER_CLIENT_USER_AGENT,
    HEADER_CONSUMER_CLIENT_IP,
)
from passport.backend.api.views.bundle.mixins import (
    BundleAccountGetterMixin,
    BundleAccountResponseRendererMixin,
    BundleCacheResponseToTrackMixin,
    BundleFederalMixin,
)
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.helpers import trim_message
from passport.backend.core.logging_utils.loggers.statbox import StatboxLogger
from passport.backend.core.models.domain import Domain
from passport.backend.core.runner.context_managers import UPDATE
from passport.backend.core.utils.decorators import cached_property


log = logging.getLogger(__name__)


class BaseSSOLoginView(
    BaseBundleView,
    BundleFederalMixin,
):
    required_headers = (
        HEADER_CLIENT_HOST,
        HEADER_CLIENT_USER_AGENT,
        HEADER_CONSUMER_CLIENT_IP,
    )
    required_grants = ['auth_by_sso.base']
    action = None
    domain = None  # от какого IdP ответ. Объект Domain

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode='auth_by_sso',
            action=self.action,
            ip=self.client_ip,
            consumer=self.consumer,
            user_agent=self.user_agent,
        )

    def get_issuer_from_saml_request_or_response(self, logout=False):
        is_response = self.form_values.get('saml_response') is not None
        field = 'saml_response' if is_response else 'saml_request'
        exception = SamlResponseInvalidError if is_response else SamlRequestInvalidError

        def decoder():
            decoded = b64decode(self.form_values[field])
            if not is_response:
                # We try to inflate
                try:
                    inflated = zlib.decompress(decoded, -15)
                    return inflated
                except Exception:
                    pass
            return decoded

        issuer = None
        try:
            saml_xml = fromstring(decoder())
            xpath = '/samlp:{}{}/saml:Issuer'.format('Logout' if logout else '', 'Response' if is_response else 'Request')
            issuer_node = saml_xml.xpath(xpath, namespaces=OneLogin_Saml2_Constants.NSMAP)
            if issuer_node:
                issuer = issuer_node[0].text
        except (TypeError, XMLSyntaxError):
            raise exception('Invalid base64 or xml')
        if not issuer:
            raise exception('SAML{} does not contain Issuer field'.format('Response' if is_response else 'Request'))
        return issuer, saml_xml

    def set_domain(self, domain=None, domain_id=None):
        if not domain and not domain_id:
            raise DomainNotFoundError()
        # проверяем что домен этого IdP обслуживаем
        if domain:
            domain_info = self.blackbox.hosted_domains(domain=domain)
        else:
            domain_info = self.blackbox.hosted_domains(domain_id=domain_id)
        if not domain_info['hosted_domains']:
            raise DomainNotFoundError()
        self.domain = Domain().parse(domain_info)
        if not self.domain.is_enabled:
            raise DomainNotFoundError()


class SubmitToSSOLoginView(BaseSSOLoginView):
    basic_form = SubmitForm
    require_track = True
    action = 'create_sso_samlrequest'

    def process_request(self):
        self.process_basic_form()
        self.read_track()
        if not self.track.user_entered_login or '@' not in self.track.user_entered_login:
            raise DomainNotFoundError('Domain is not found in track')

        idp_domain = self.track.user_entered_login.rsplit('@', 1)[1]
        self.statbox.bind(
            track_id=self.track_id,
            saml_idp_domain=idp_domain,
        )
        self.set_domain(domain=idp_domain)
        # Патчим чтоб уникальный id был равен нашему треку для простоты работы
        OneLogin_Saml2_Utils.generate_unique_id = staticmethod(lambda: 'track_' + self.track_id)
        samlresponse_receiver_url = settings.SAMLRESPONSE_RECEIVER_URL_TEMPLATE % self.request.env.host
        # выбор конфига в зависимости от IdP пользователя
        saml_settings = self.get_saml_settings(domain_id=self.domain.id, only_enabled=True)
        if saml_settings is None:
            raise DomainNotFoundError('Domain is not available to login: %s' % self.domain.domain)
        # урл возврата ответа проставляем в конфиг на - /submit с нужным tld (env.host)
        saml_settings['sp']['assertionConsumerService']['url'] = samlresponse_receiver_url
        try:
            auth = OneLogin_Saml2_Auth({}, old_settings=saml_settings)
            # тут надо в return_to передать какой-то валидный урл или сломается. вариант 2 - строкой выше передать урл текущий
            redirect_url = auth.login(
                force_authn=settings.SAML_SSO_FORCE_AUTHN,
                return_to=samlresponse_receiver_url,
            )
        except OneLogin_Saml2_Error as e:
            log.debug('SAML error: %s %s' % (e.code, e.message))
            self.statbox.log(status='error', error=trim_message('%s %s' % (e.code, e.message)))
            raise SamlSettingsError()

        with self.track_manager.transaction(self.track_id).rollback_on_error() as track:
            track.domain = idp_domain
            if self.form_values['code_challenge']:
                track.oauth_code_challenge = self.form_values['code_challenge']
                track.oauth_code_challenge_method = self.form_values['code_challenge_method']

        self.statbox.log(status='success')
        self.response_values.update(redirect_to=redirect_url)


def prepare_flask_request_to(request, get_data, post_data, path, lowercase_urlencoding=False):
    return {
        'https': 'on',  # тут важно что ответ с samlresponse всегда прилетает на https на фронте
        'http_host': request.env.host,
        'script_name': path,
        'get_data': get_data,
        'post_data': post_data,
        # True if using ADFS as IdP, https://github.com/onelogin/python-saml/pull/144
        'lowercase_urlencoding': lowercase_urlencoding,
        'query_string': request.query_string,
    }


class CommitToSSOLoginView(BundleBaseAuthorizationMixin,
                           BundleAccountGetterMixin,
                           BundleAccountResponseRendererMixin,
                           BundleCacheResponseToTrackMixin,
                           BaseSSOLoginView):
    basic_form = CommitForm
    action = 'processing_sso_samlresponse'

    auth = None  # SP SAML instance
    federal_alias_value = None  # id домена обслуживаемого IdP / уникальный идентификатор пользователя в пределах этого домена
    saml_settings = None

    @staticmethod
    def get_track_id(response):
        in_response_to = response.get('InResponseTo')
        if in_response_to.startswith('track_'):
            return in_response_to[len('track_'):]
        raise SamlResponseInvalidError('Invalid InResponseTo field format')

    def process_saml_response(self):
        issuer, saml_response_xml = self.get_issuer_from_saml_request_or_response()
        self.track_id = self.get_track_id(saml_response_xml)
        self.read_track()
        domain = self.track.domain
        self.set_domain(domain=domain)
        # выбор конфига в зависимости от содержимого saml response (какой домен прислал ответ)
        self.saml_settings = self.get_saml_settings(domain_id=self.domain.id, only_enabled=True)
        if self.saml_settings is None:
            raise DomainNotFoundError('Domain is not available to login: %s' % self.domain.domain)
        request = prepare_flask_request_to(
            request=self.request,
            get_data={},
            post_data=dict(
                SAMLResponse=self.form_values['saml_response'],
                RelayState=self.form_values['relay_state'],
            ),
            path=settings.SAMLRESPONSE_RECEIVER_URL_PATH,
            lowercase_urlencoding=self.saml_settings['idp'].get('lowercase_urlencoding', False)
        )
        try:
            self.auth = OneLogin_Saml2_Auth(request, old_settings=self.saml_settings)
            self.auth.process_response()
        except OneLogin_Saml2_Error as e:
            log.debug('SAML error: %s %s' % (e.code, e.message))
            self.statbox.log(status='error', error=trim_message('%s %s' % (e.code, e.message)))
            raise SamlSettingsError()
        except OneLogin_Saml2_ValidationError as e:
            log.debug('SAML Response error: %s %s' % (e.code, e.message))
            self.statbox.log(status='error', error=trim_message('%s %s' % (e.code, e.message)))
            raise SamlResponseInvalidError()

        errors = self.auth.get_errors()
        self.statbox.bind(
            track_id=self.track_id,
            saml_idp_domain=self.domain.domain,
            name_id=self.auth.get_nameid(),
        )
        if len(errors) != 0 or not self.auth.is_authenticated():
            last_reason = self.auth.get_last_error_reason()
            log.debug('Not authorized, errors: [%s], reason: %s' % (';'.join(errors), last_reason))
            self.statbox.log(status='error', error=trim_message('Not authorized: [%s]' % ';'.join(errors)))
            if last_reason and last_reason.startswith('The status code of the Response was not Success, was '):
                # в случае этих ошибок в adfs event log осталась запись почему adfs отдал ошибку, пусть сначала попробуют решить на своей стороне
                raise SamlResponseExternalError()
            elif last_reason and last_reason.startswith('Signature validation failed.'):
                # скорее всего неверный сертификат подписи сообщений добавили в настройки
                raise SamlInvalidSignatureError()
            raise SamlResponseInvalidError()

        if not self.auth.get_nameid():
            # по идее не достижим, но нужен как минимум для тестирования, где мы мокаем все валидации ответа в библиотеке и можем случайно пропустить пустой идентификатор дальше
            raise SamlResponseInvalidError('No Name_ID on SamlResponse')
        log.debug('SAML Response contains claims: {}'.format(','.join(self.auth.get_attributes().keys())))
        self.federal_alias_value = self.make_alias(self.auth.get_nameid(), self.domain)

    def _register_federal(self):
        email = self.get_attribute_from_response('User.EmailAddress')
        if not email:
            raise SamlEmailNoInResponseError('No User.EmailAddress claim in SamlResponse')

        try:
            self.register_federal(
                domain=self.domain,
                name_id=self.auth.get_nameid(),
                email=email,
                firstname=self.get_attribute_from_response('User.Firstname'),
                lastname=self.get_attribute_from_response('User.Surname'),
            )
        except EmailExistError:
            raise SamlEmailExistError('User with same email already exist or email is prohibited')
        except EmailUnsupportableDomainError:
            raise SamlEmailUnsupportableDomainError('Mail Domain is different from the domain of the Identity Provider')

    def get_attribute_from_response(self, attr_name):
        value = self.auth.get_attribute(attr_name)
        if value:
            return value[0]

    def process_request(self):
        self.process_basic_form()
        self.process_saml_response()

        availability_info = self.blackbox.userinfo(
            login=self.federal_alias_value.lower(),
            sid='federal',
        )
        is_user_exist = availability_info['uid'] is not None

        if is_user_exist:
            self.parse_account_with_domains(availability_info)
        elif self.saml_settings['idp'].get('disable_jit_provisioning', False):
            log.debug('Settings prohibit registration at the entrance in this domain')
            raise SamlRegistrationNotSupportedError()
        else:
            self._register_federal()

        with self.track_manager.transaction(self.track_id).commit_on_error() as self.track:
            self.track.auth_method = settings.AUTH_METHOD_SAML_SSO
            if not is_user_exist:
                self.track.is_successful_registered = True

            # Большинство полей проставляем тут, а allow_authorization - только если не требуется дорегистрация
            set_authorization_track_fields(
                self.account,
                self.track,
                allow_create_session=False,
                allow_create_token=False,
                password_passed=False,
            )

            redirect_state = self.check_user_policies()
            if redirect_state is not None:
                self.state = redirect_state
                self.response_values.update(track_id=self.track_id)
                self.fill_response_with_account(personal_data_required=False)
                return

            self.track.allow_authorization = True

        self.response_values.update(track_id=self.track_id)
        log.debug('Successfully authorized: [%s]' % self.federal_alias_value)
        self.statbox.log(uid=self.account.uid, status='success')


class LogoutSSOView(BaseSSOLoginView,
                    BundleAccountGetterMixin):
    basic_form = LogoutFederalForm
    action = 'logout_sso_idp_initiated'
    federal_alias_value = None  # id домена обслуживаемого IdP / уникальный идентификатор пользователя в пределах этого домена

    def parse_name_id(self, name_id, domain):
        # Пока по примеру ADFS ожидаем что name_id это будет login или login@domain
        # Если что-то придет другое, необходимо будет пересмотреть это место, но мы пока не представляем что
        if name_id.endswith('@' + domain.domain):
            name_id = name_id.rsplit('@', 1)[0]
        return '%s/%s' % (domain.id, name_id)

    def glogout_user(self):
        events = dict(action='account_glogout_federal', consumer=self.consumer)
        with UPDATE(self.account, self.request.env, events):
            self.account.global_logout_datetime = datetime.now()

        self.statbox.log(uid=self.account.uid, status='success')

    def process_request(self):
        self.process_basic_form()

        issuer, saml_response_xml = self.get_issuer_from_saml_request_or_response(logout=True)
        saml_settings = self.get_saml_settings(entity_id=issuer, only_enabled=True)

        if saml_settings is None:
            raise DomainNotFoundError('SAML SSO config not found for issuer: %s' % issuer)
        request = prepare_flask_request_to(
            request=self.request,
            get_data=dict(
                SAMLRequest=self.form_values['saml_request'],
            ),
            post_data={},
            path=settings.SAMLREQUEST_LOGOUT_RECEIVER_URL_PATH,
            lowercase_urlencoding=saml_settings['idp'].get('lowercase_urlencoding', False),
        )
        try:
            auth = OneLogin_Saml2_Auth(request, old_settings=saml_settings)
            redirect_url = auth.process_slo(keep_local_session=True)  # сами разглогаутим ниже
        except OneLogin_Saml2_Error as e:
            log.debug('SAML error: %s %s' % (e.code, e.message))
            self.statbox.log(status='error', error=trim_message('%s %s' % (e.code, e.message)))
            raise SamlSettingsError()
        except OneLogin_Saml2_ValidationError as e:
            log.debug('SAML Response error: %s %s' % (e.code, e.message))
            self.statbox.log(status='error', error=trim_message('%s %s' % (e.code, e.message)))
            raise SamlRequestInvalidError()

        errors = auth.get_errors()
        if len(errors) != 0:
            log.debug('Logout request failed, errors: [%s], reason: %s' % (';'.join(errors), auth.get_last_error_reason()))
            self.statbox.log(status='error', error=trim_message('Logout request failed: [%s]' % ';'.join(errors)))
            # TODO вообще тут вместо ошибок наших надо формировать правильный ответ idp, но библиотека умеет только ответ ОК
            raise SamlRequestInvalidError()

        name_id = OneLogin_Saml2_Logout_Request.get_nameid(saml_response_xml)

        # необходимо найти домен, который обслуживается этим idp, и проверить есть ли пользователь на нем
        domain_ids = saml_settings['idp']['domain_ids']
        for domain_id in domain_ids:
            domain_info = self.blackbox.hosted_domains(domain_id=domain_id)
            if not domain_info['hosted_domains']:
                continue
            domain = Domain().parse(domain_info)
            federal_alias_value = self.parse_name_id(name_id, domain)
            availability_info = self.blackbox.userinfo(
                login=federal_alias_value,
                sid='federal',
            )

            if availability_info['uid'] is not None:
                self.parse_account(availability_info)
            else:
                continue

            self.glogout_user()

        if not self.account:
            self.statbox.log(status='error', error=trim_message('Logout request failed: user %s not found on domain ids [%s]' % (name_id, ' ,'.join(map(str, domain_ids)))))
            raise AccountNotFoundError()

        self.response_values.update(redirect_to=redirect_url)
