# -*- coding: utf-8 -*-
import logging

from passport.backend.api.common import set_authorization_track_fields
from passport.backend.api.views.bundle.auth.sso.forms import CompleteFederalForm
from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.exceptions import (
    ActionNotRequiredError,
    EulaIsNotAcceptedError,
)
from passport.backend.api.views.bundle.headers import (
    HEADER_CLIENT_USER_AGENT,
    HEADER_CONSUMER_CLIENT_IP,
)
from passport.backend.api.views.bundle.mixins import BundleAccountGetterMixin
from passport.backend.core.runner.context_managers import UPDATE


log = logging.getLogger(__name__)


class CompleteFederalView(BaseBundleView, BundleAccountGetterMixin):
    required_grants = ['auth_by_sso.base']
    require_track = True
    required_headers = (
        HEADER_CONSUMER_CLIENT_IP,
        HEADER_CLIENT_USER_AGENT,
    )
    basic_form = CompleteFederalForm

    def process_request(self):
        self.read_track()
        self.get_account_from_track(check_disabled_on_deletion=True)
        if not self.account.is_federal or self.account.is_complete_federal:
            raise ActionNotRequiredError()

        self.process_basic_form()

        if not self.form_values['eula_accepted']:
            raise EulaIsNotAcceptedError()

        with UPDATE(
            self.account,
            self.request.env,
            {
                'action': 'complete_federal',
                'consumer': self.consumer,
            },
        ):
            self.account.is_pdd_agreement_accepted = self.form_values['eula_accepted']

        with self.track_manager.transaction(self.track_id).rollback_on_error() as track:
            set_authorization_track_fields(
                self.account,
                track,
                allow_create_session=True,
                allow_create_token=False,
                password_passed=False,
            )
