# -*- coding: utf-8 -*-
import logging

from defusedxml.minidom import parseString
from onelogin.saml2.errors import OneLogin_Saml2_Error
from onelogin.saml2.settings import OneLogin_Saml2_Settings
from passport.backend.api.common.format_response import XmlLoggedResponse
from passport.backend.api.views.bundle.auth.sso.exceptions import SamlSettingsError
from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.core.conf import settings
from passport.backend.utils.string import smart_text
from six.moves.urllib.parse import urljoin


log = logging.getLogger(__name__)


class MetadataGeneratorSuccessfulResponse(XmlLoggedResponse):
    def __init__(self, response):
        super(MetadataGeneratorSuccessfulResponse, self).__init__(
            response=response,
            format_=smart_text,
            status=200,
        )


class MetadataGeneratorView(BaseBundleView):

    def process_request(self):
        saml_settings = settings.get_yandex_sp_config()  # оставляет пустыми значения IdP
        try:
            saml_settings_obj = OneLogin_Saml2_Settings(saml_settings, sp_validation_only=True)
            metadata_xml = saml_settings_obj.get_sp_metadata()
        except OneLogin_Saml2_Error as e:
            log.debug('SAML error: %s %s' % (e.code, e.message))
            raise SamlSettingsError()

        try:
            xml = parseString(metadata_xml.encode('utf-8'), forbid_dtd=True, forbid_entities=True, forbid_external=True)
        except Exception as e:
            raise Exception('Error parsing metadata. ' + e.message)

        entity_descriptor = xml.getElementsByTagName('md:EntityDescriptor')[0]
        sp_sso_descriptor = entity_descriptor.getElementsByTagName('md:SPSSODescriptor')[0]
        assertion_consumer_service = sp_sso_descriptor.getElementsByTagName('md:AssertionConsumerService')[0]

        # нода которую будем размножать
        acs_node = xml.createElement('md:AssertionConsumerService')
        acs_node.setAttribute('Binding', assertion_consumer_service.getAttribute('Binding'))

        # удалим старую ноду, ее неудобно переписывать в цикле
        sp_sso_descriptor.removeChild(assertion_consumer_service)

        # Дополнить урлы авторизации
        template = urljoin(settings.PASSPORT_BASE_URL_TEMPLATE, settings.SAMLRESPONSE_RECEIVER_URL_PATH)
        for index, tld in enumerate(settings.PASSPORT_TLDS):
            acs_node.setAttribute('Location', template % {'tld': tld})
            acs_node.setAttribute('index', str(index))
            sp_sso_descriptor.appendChild(acs_node.cloneNode(True))

        self.response_values.update(metadata=xml.toprettyxml())

    def respond_success(self):
        return MetadataGeneratorSuccessfulResponse(self.response_values['metadata'])
