# -*- coding: utf-8 -*-
from collections import namedtuple

from passport.backend.api.common.account import get_allowed_auth_methods
from passport.backend.api.common.authorization import build_cookie_lah
from passport.backend.api.common.social_api import is_social_auth_available
from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.exceptions import AccountNotFoundError
from passport.backend.api.views.bundle.headers import (
    HEADER_CLIENT_COOKIE,
    HEADER_CLIENT_HOST,
    HEADER_CONSUMER_CLIENT_IP,
)
from passport.backend.api.views.bundle.mixins import (
    BundleAccountGetterMixin,
    BundleAccountResponseRendererMixin,
)
from passport.backend.core.builders.blackbox.utils import add_phone_arguments
from passport.backend.core.conf import settings
from passport.backend.core.cookies.cookie_lah import try_parse_cookie_lah
from passport.backend.core.utils.blackbox import get_many_accounts_by_uids

from .exceptions import AccountNotInSuggestError
from .forms import (
    DeleteAccountFromSuggestForm,
    GetSuggestedInputLoginForm,
)


BASIC_GRANT = 'account_suggest.read'
EXTENDED_GRANT = 'account_suggest.edit'


AccountInfo = namedtuple('AccountInfo', 'account social_profiles history_item')


class BaseAccountSuggestView(BaseBundleView):
    required_headers = (
        HEADER_CLIENT_HOST,
        HEADER_CLIENT_COOKIE,
        HEADER_CONSUMER_CLIENT_IP,
    )

    def _parse_cookie_lah(self):
        return try_parse_cookie_lah(self.cookies.get('lah'))

    def _parse_and_clean_cookie_lah(self):
        auth_history_container = self._parse_cookie_lah()

        uids = [item.uid for item in auth_history_container]
        accounts, unknown_uids = get_many_accounts_by_uids(
            uids,
            self.blackbox,
            userinfo_args=add_phone_arguments(emails=True),
        )
        for uid in unknown_uids:
            auth_history_container.remove(uid)

        account_mapping = {account.uid: account for account in accounts}
        return account_mapping, auth_history_container

    def _build_cookies(self, auth_history_container):
        return [
            build_cookie_lah(
                self.request.env,
                auth_history_container,
            ),
        ]


class DeleteAccountFromSuggestView(BaseAccountSuggestView):
    required_grants = [BASIC_GRANT, EXTENDED_GRANT]
    basic_form = DeleteAccountFromSuggestForm

    def process_request(self):
        self.process_basic_form()
        _, auth_history_container = self._parse_and_clean_cookie_lah()
        auth_history_container.remove(uid=self.form_values['uid'])
        self.response_values.update(
            cookies=self._build_cookies(auth_history_container),
        )


class GetSuggestView(BaseAccountSuggestView, BundleAccountResponseRendererMixin):
    required_grants = [BASIC_GRANT]

    def _account_info_to_response(self, account_info):
        allowed_auth_methods = get_allowed_auth_methods(
            account=account_info.account,
            social_profiles=account_info.social_profiles,
            user_ip=self.client_ip,
            cookies=self.cookies,
        )
        account_response = self.account_to_response(
            account=account_info.account,
            personal_data_required=False,
            account_info_required=False,
            display_name_required=True,
        )
        account_response.update(
            primary_alias_type=self.get_alias_type(account=account_info.account),
        )
        if account_info.account.emails.default:
            account_response.update(
                default_email=account_info.account.emails.default.address,
            )
        return {
            'account': account_response,
            'allowed_auth_methods': allowed_auth_methods,
            'preferred_auth_method': self._get_preferred_auth_method(account_info, allowed_auth_methods),
        }

    def _get_preferred_auth_method(self, account_info, allowed_auth_methods):
        last_used_method = settings.AUTH_METHOD_FROM_ID.get(account_info.history_item.method)
        if last_used_method in allowed_auth_methods:
            return last_used_method
        elif allowed_auth_methods:
            return allowed_auth_methods[0]

    def _make_account_infos(self, account_mapping, auth_history_container):
        profile_mapping = {}
        uids = [
            uid for uid, account in account_mapping.items()
            if is_social_auth_available(account)
        ]
        if uids and not settings.IS_INTRANET:
            # TODO: если кука сильно разрастётся, придётся ходить в social_api чанками
            profiles = self.social_api.get_profiles(uids=uids, allow_auth=True)
            for profile in profiles:
                if profile['provider_code'] in settings.AUTH_ALLOWED_PROVIDERS:
                    profile_mapping.setdefault(profile['uid'], []).append(profile)

        result = []
        # Упорядочим записи, чтобы первыми шли самые свежие из тех, для кого известен метод
        ordered_history_items = sorted(
            auth_history_container,
            key=lambda item: (item.method != settings.AUTH_METHOD_TO_ID[settings.AUTH_METHOD_UNKNOWN], item.timestamp),
            reverse=True,
        )
        for history_item in ordered_history_items:
            uid = history_item.uid
            result.append(AccountInfo(
                account=account_mapping[uid],
                social_profiles=profile_mapping.get(uid, []),
                history_item=history_item,
            ))
        return result

    def process_request(self):
        account_mapping, auth_history_container = self._parse_and_clean_cookie_lah()
        self.response_values.update(
            accounts=[
                self._account_info_to_response(account_info)
                for account_info in self._make_account_infos(account_mapping, auth_history_container)
            ],
            cookies=self._build_cookies(auth_history_container),
        )


class GetSuggestedInputLogin(BaseAccountSuggestView, BundleAccountGetterMixin):
    required_grants = [BASIC_GRANT]
    basic_form = GetSuggestedInputLoginForm

    def process_request(self):
        self.process_basic_form()
        input_login = None

        auth_history_container = self._parse_cookie_lah()
        auth_history_item = auth_history_container.by_uid(self.form_values['uid'])
        if not auth_history_item:
            raise AccountNotInSuggestError()

        if auth_history_item.input_login:
            try:
                self.get_account_by_login(login=auth_history_item.input_login, enabled_required=False)
                if self.account.uid == auth_history_item.uid:
                    input_login = auth_history_item.input_login
            except AccountNotFoundError:
                pass

        self.response_values.update(
            input_login=input_login,
        )


class CheckSuggestAvailability(BaseAccountSuggestView):
    required_grants = [BASIC_GRANT]

    def process_request(self):
        auth_history_container = self._parse_cookie_lah()
        is_suggest_available = bool(auth_history_container)
        # Намеренно не делаем более сложных проверок (наличия аккаунта, проверки глогаута, etc),
        # чтобы ручка отвечала быстро
        self.response_values.update(
            is_suggest_available=is_suggest_available,
        )
