# -*- coding: utf-8 -*-
from datetime import datetime
import logging

from passport.backend.api.common.account import is_auth_by_sms_secure_enough_for_account
from passport.backend.api.common.phone import get_accounts_with_actual_secure_phone
from passport.backend.api.views.bundle.auth.suggest_by_phone.forms import SuggestAccountsByPhoneCheckAvailabilityForm
from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.exceptions import InvalidTrackStateError
from passport.backend.api.views.bundle.headers import HEADER_CONSUMER_CLIENT_IP
from passport.backend.api.views.bundle.mixins import (
    BundleAccountPropertiesMixin,
    BundlePhoneMixin,
)
from passport.backend.api.yasms.utils import get_many_accounts_with_phones_by_uids
from passport.backend.core.builders.antifraud import (
    BaseAntifraudApiError,
    get_antifraud_api,
)
from passport.backend.core.builders.historydb_api import get_historydb_api
from passport.backend.core.conf import settings
from passport.backend.core.models.account import ACCOUNT_DISABLED_ON_DELETION
from passport.backend.core.types.account.account import (
    ACCOUNT_TYPE_LITE,
    ACCOUNT_TYPE_NEOPHONISH,
    ACCOUNT_TYPE_NORMAL,
    ACCOUNT_TYPE_PDD,
    ACCOUNT_TYPE_SOCIAL,
)
from passport.backend.core.types.phone_number.phone_number import (
    get_alt_phone_numbers_of_phone_number,
    parse_phone_number,
)
from passport.backend.utils.time import unixtime_to_datetime


log = logging.getLogger('passport.api.view.bundle.auth.suggest_by_phone')


AUTH_FLOW_INSTANT = 'instant'
AUTH_FLOW_FULL = 'full'
REGISTER_FLOW_PORTAL = 'portal'
REGISTER_FLOW_NEOPHONISH = 'neophonish'


class BaseSuggestAccountsByPhoneView(BaseBundleView, BundleAccountPropertiesMixin):
    required_headers = (
        HEADER_CONSUMER_CLIENT_IP,
    )

    required_grants = ['auth.suggest_by_phone']

    def get_suitable_accounts_by_phone(self, phone_number):
        allowed_account_types = {
            ACCOUNT_TYPE_LITE,
            ACCOUNT_TYPE_NEOPHONISH,
            ACCOUNT_TYPE_NORMAL,
            ACCOUNT_TYPE_PDD,
            ACCOUNT_TYPE_SOCIAL,
        }

        all_phone_numbers = set([phone_number] + get_alt_phone_numbers_of_phone_number(phone_number))

        bindings = self.blackbox.phone_bindings(
            phone_numbers=[p.e164 for p in all_phone_numbers],
            need_history=False,
            need_unbound=False,
        )

        uids = {binding['uid'] for binding in bindings}

        accounts, _ = get_many_accounts_with_phones_by_uids(
            uids=uids,
            blackbox_builder=self.blackbox,
            userinfo_args=dict(
                get_family_info=True,
            ),
        )

        suitable_accounts = []
        for account in accounts:
            if account.type not in allowed_account_types:
                continue
            if (
                account.disabled_status == ACCOUNT_DISABLED_ON_DELETION and
                not self.can_restore_disabled_account(account)
            ):
                continue
            if not (account.phones.secure and account.phones.secure.number in all_phone_numbers):
                continue
            suitable_accounts.append(account)

        return suitable_accounts

    @staticmethod
    def get_uid_to_lastauth_mapping(accounts):
        return get_historydb_api().lastauth_bulk(
            uids=[account.uid for account in accounts],
        )

    def filter_accounts(self, accounts, uid_to_lastauth_mapping):
        filtered_accounts = []

        # Сначала отфильтруем по lastauth
        for account in accounts:
            # Убираем детские аккаунт из результатирующей выборки
            if account.is_child:
                continue
            lastauth = uid_to_lastauth_mapping.get(account.uid)
            if (
                lastauth is None or
                datetime.now() - unixtime_to_datetime(lastauth) > settings.MAX_LASTAUTH_AGE_FOR_AUTH_SUGGEST
            ):
                if account.uid not in settings.IGNORE_POSSIBLE_PHONE_OWNER_CHANGE_FOR_UIDS:
                    log.debug('Account %s not shown due to old lastauth', account.uid)
                    continue
            filtered_accounts.append(account)

        # Затем - проверим, не сменился ли владелец номера с момента привязки номера
        if filtered_accounts and settings.USE_PHONE_SQUATTER:
            accounts_with_actual_secure_phone = get_accounts_with_actual_secure_phone(
                accounts=filtered_accounts,
                request_id=self.request.env.request_id,
            )

            all_uids = [account.uid for account in filtered_accounts]
            uids_with_actual_secure_phone = [account.uid for account in accounts_with_actual_secure_phone]
            uids_with_not_actual_secure_phone = set(all_uids) - set(uids_with_actual_secure_phone)

            if uids_with_not_actual_secure_phone:
                uids_string = ', '.join([str(uid) for uid in uids_with_not_actual_secure_phone])
                if settings.PHONE_SQUATTER_DRY_RUN:
                    log.debug('Accounts %s seem not to own secure phone anymore. Still showing them due to dry run', uids_string)
                else:
                    log.debug('Accounts %s seem not to own secure phone anymore. Not showing them', uids_string)
                    filtered_accounts = accounts_with_actual_secure_phone

        log.debug('Going to suggest accounts: %s', ', '.join([str(account.uid) for account in filtered_accounts]) or '-')
        return filtered_accounts


class SuggestAccountsByPhoneCheckAvailabilityView(BaseSuggestAccountsByPhoneView):
    basic_form = SuggestAccountsByPhoneCheckAvailabilityForm

    def process_request(self):
        self.process_basic_form()

        phone = parse_phone_number(self.form_values['phone_number'])
        suitable_accounts = self.get_suitable_accounts_by_phone(phone)

        uid_to_lastauth_mapping = self.get_uid_to_lastauth_mapping(suitable_accounts)
        filtered_accounts = self.filter_accounts(suitable_accounts, uid_to_lastauth_mapping=uid_to_lastauth_mapping)

        self.response_values.update(
            is_suggest_available=bool(filtered_accounts),
        )


class SuggestAccountsByPhoneListView(BaseSuggestAccountsByPhoneView, BundlePhoneMixin):
    require_track = True

    def check_track_state(self):
        if not (
            self.track.phone_confirmation_phone_number and
            self.is_phone_confirmed_in_track(allow_by_flash_call=True)
        ):
            raise InvalidTrackStateError()

    def accounts_to_response(self, accounts, uid_to_card_presence_mapping, uid_to_lastauth_mapping):
        response = []
        for account in accounts:
            login = account.human_readable_login
            if account.is_neophonish:
                login = ''  # по неофонишному алиасу ЧЯ акк не найдёт
            account_dict = {
                'uid': account.uid,
                'login': login,
                'primary_alias_type': account.type,
                'has_plus': bool(account.plus.has_plus),
                'has_family': account.has_family,
                'has_bank_card': uid_to_card_presence_mapping.get(account.uid, False),
            }
            if account.person and account.person.default_avatar:
                account_dict.update(
                    default_avatar=account.person.default_avatar,
                    avatar_url=settings.GET_AVATAR_URL % (
                        account.person.default_avatar,
                        self.track.avatar_size or 'normal',
                    ),
                )

            auth_flows = []
            if (
                account.is_enabled and
                is_auth_by_sms_secure_enough_for_account(account)
                # TODO: учитывать также флаг от операторов связи "у номера сменился владелец" (PASSP-36295)
            ):
                auth_flows.append(AUTH_FLOW_INSTANT)
            if login:
                auth_flows.append(AUTH_FLOW_FULL)
            account_dict.update(
                display_name=account.person.display_name.as_dict(),
                allowed_auth_flows=auth_flows,
            )

            response.append(account_dict)

        response = sorted(
            response,
            key=lambda d: (
                # Сначала выводим имеющих и Плюс, и карту. Затем имеющих только Плюс. И так далее. Наконец,
                # сортируем по последней активности.
                d['has_plus'] and d['has_bank_card'],
                d['has_plus'],
                d['has_bank_card'],
                d['has_family'],
                uid_to_lastauth_mapping[d['uid']] or 0,
            ),
            reverse=True,
        )
        self.response_values['accounts'] = response

    def process_request(self):
        self.read_track()
        self.check_track_state()

        phone = parse_phone_number(self.track.phone_confirmation_phone_number)
        suitable_accounts = self.get_suitable_accounts_by_phone(phone)

        uid_to_lastauth_mapping = self.get_uid_to_lastauth_mapping(suitable_accounts)
        filtered_accounts = self.filter_accounts(suitable_accounts, uid_to_lastauth_mapping=uid_to_lastauth_mapping)

        try:
            uid_to_card_presence_mapping = get_antifraud_api().uid_cards(
                uids=[account.uid for account in filtered_accounts],
            ).uids_with_cards
        except BaseAntifraudApiError:
            uid_to_card_presence_mapping = {}

        self.accounts_to_response(
            accounts=filtered_accounts,
            uid_to_card_presence_mapping=uid_to_card_presence_mapping,
            uid_to_lastauth_mapping=uid_to_lastauth_mapping,
        )

        allowed_registration_flows = []
        if settings.ALLOW_REGISTRATION:
            # Нового неофониша предлагаем зарегистрировать, только если старых не нашлось
            if not any([acc.is_neophonish for acc in filtered_accounts]):
                allowed_registration_flows.append(REGISTER_FLOW_NEOPHONISH)
            allowed_registration_flows.append(REGISTER_FLOW_PORTAL)
        self.response_values.update(
            allowed_registration_flows=allowed_registration_flows,
        )
