# -*- coding: utf-8 -*-
from datetime import datetime
import logging

from passport.backend.api.views.bundle.auth.third_step.forms import (
    FreezeAskingDataForm,
    MissingDataToAskForm,
)
from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.exceptions import (
    AccountUidMismatchError,
    InvalidTrackStateError,
)
from passport.backend.api.views.bundle.mixins import (
    BundleAccountGetterMixin,
    BundleAccountResponseRendererMixin,
    BundlePhoneMixin,
)
from passport.backend.core.conf import settings
from passport.backend.core.runner.context_managers import UPDATE
from passport.backend.core.types.login.login import is_test_yandex_login


ASK_ADDITIONAL_DATA_GRANT = 'account.ask_additional_data'
FREEZE_ADDITIONAL_DATA_ASKING_GRANT = 'account.freeze_additional_data_asking'

STARTING_POINT = 'password'

STATE_PHONE = 'phone'
STATE_EMAIL = 'email'
STATE_AVATAR = 'avatar'
STATE_PASSWORD = 'password'

STATE_TRANSITION_GRAPH = {
    STATE_PHONE: STATE_EMAIL,
    STATE_EMAIL: STATE_AVATAR,
    STATE_AVATAR: STATE_PASSWORD,
    STATE_PASSWORD: STATE_PHONE,
}

log = logging.getLogger('passport.backend.api.views.bundle.auth.third_step')


class AskAdditionalDataView(BundleAccountGetterMixin,
                            BundlePhoneMixin,
                            BundleAccountResponseRendererMixin,
                            BaseBundleView):
    """
    Для повышения безопасности аккаунта пытаемся
    заполучить недостающие данные у пользователя
    """
    required_grants = [ASK_ADDITIONAL_DATA_GRANT]

    basic_form = MissingDataToAskForm

    def ask_for_phone_data(self):
        if (
            not self.account.have_password or
            self.account.totp_secret.is_set or
            (self.account.phones.secure and not self.account.phones.secure.need_admission)
        ):
            return None, {}

        state = STATE_PHONE
        data = {}

        phones = self.account.phones.all().values()
        to_confirm = []
        to_secure = []
        for phone in phones:
            if phone.bound:
                to_secure.append({
                    'id': phone.id,
                    'number': phone.number.e164,
                })
            else:
                to_confirm.append({
                    'id': phone.id,
                    'number': phone.number.e164,
                })

        if self.account.phones.secure:
            data['action'] = 'admit'
            data['data'] = [{
                'id': self.account.phones.secure.id,
                'number': self.account.phones.secure.number.e164,
            }]
        elif to_secure:
            data['action'] = 'secure'
            data['data'] = to_secure
        elif to_confirm:
            data['action'] = 'confirm'
            data['data'] = to_confirm
        else:
            data['action'] = 'add'

        return state, data

    def ask_for_email_data(self):
        if (
            not self.account.have_password or
            self.account.totp_secret.is_set or
            self.account.emails.suitable_for_restore
        ):
            return None, {}

        state = STATE_EMAIL
        data = {}

        to_restore = []
        to_confirm = []

        for email in self.account.emails.external:
            if email.is_confirmed and not email.is_rpop and not email.is_silent and email.is_unsafe:
                to_restore.append(email.address)
            elif not email.is_confirmed:
                to_confirm.append(email.address)

        if to_restore:
            data['action'] = 'restore'
            data['addresses'] = to_restore
        elif to_confirm:
            data['action'] = 'confirm'
            data['addresses'] = to_confirm
        else:
            data['action'] = 'add'

        return state, data

    def ask_for_avatar_data(self):
        if not self.account.person.is_avatar_empty:
            return None, {}

        state = STATE_AVATAR
        data = {
            'action': 'add',
        }

        return state, data

    def ask_for_password_data(self):
        if not self.account.password.is_set or not self.account.password.is_weak:
            return None, {}
        if self.account.is_pdd:
            # Некоторые ПДД-пользователи не имеют права изменять свой пароль
            response = self.blackbox.hosted_domains(domain=self.account.domain.domain)
            self.account.parse(response)
            if not self.account.domain.can_users_change_password:
                return None, {}

        state = STATE_PASSWORD
        data = {
            'action': 'weak',
        }

        return state, data

    def get_state_and_data(self, ask_next):
        state = None
        data = {}

        full_circle = len(STATE_TRANSITION_GRAPH)
        for _ in range(full_circle):
            if ask_next == STATE_PHONE:
                state, data = self.ask_for_phone_data()
            elif ask_next == STATE_EMAIL:
                state, data = self.ask_for_email_data()
            elif ask_next == STATE_AVATAR:
                state, data = self.ask_for_avatar_data()
            elif ask_next == STATE_PASSWORD:
                state, data = self.ask_for_password_data()

            if state:
                break
            ask_next = STATE_TRANSITION_GRAPH[ask_next]

        return state, data

    def process_request(self):
        self.process_basic_form()
        self.read_or_create_track('authorize')
        self.get_account_from_session_or_oauth_token(
            multisession_uid=self.form_values['uid'],
            need_phones=True,
            emails=True,
            email_attributes='all',
            attributes=settings.BLACKBOX_ATTRIBUTES + (
                'account.additional_data_asked',
                'account.additional_data_ask_next_datetime',
            ),
        )

        if self.track.uid and int(self.track.uid) != self.account.uid:
            raise AccountUidMismatchError(known_uids=[self.account.uid])

        ask_next_dt = self.account.additional_data_ask_next_datetime
        if ask_next_dt and datetime.now() < ask_next_dt:
            return

        last_data_asked = self.account.additional_data_asked
        if last_data_asked:
            ask_next = STATE_TRANSITION_GRAPH.get(last_data_asked, STARTING_POINT)
        else:
            ask_next = STARTING_POINT

        state, data = self.get_state_and_data(ask_next)

        if state:
            with self.track_manager.transaction(self.track_id).rollback_on_error() as track:
                track.additional_data_asked = state
                self.set_uid_to_track(self.account.uid, track)

            self.response_values.update(
                track_id=self.track_id,
                state=state,
                **data
            )


class FreezeAdditionalDataAskingView(BundleAccountGetterMixin, BaseBundleView):
    """
    Не спрашиваем данные в течение некоторого срока после того, как пользователь
        * либо отменил действие,
        * либо выполнил его (по умолчанию).
    Если пользователь проигнорировал окно дозапроса данных (закрыл его) - продолжаем спрашивать.
    """
    required_grants = [FREEZE_ADDITIONAL_DATA_ASKING_GRANT]
    require_track = True

    basic_form = FreezeAskingDataForm

    def process_request(self):
        self.process_basic_form()
        self.read_track()
        if not self.track.additional_data_asked:
            raise InvalidTrackStateError()

        self.get_account_from_session_or_oauth_token(
            multisession_uid=self.form_values['uid'],
            attributes=settings.BLACKBOX_ATTRIBUTES + (
                'account.additional_data_asked',
                'account.additional_data_ask_next_datetime',
            ),
        )

        if settings.IS_TEST and is_test_yandex_login(self.account.login):
            period_key = 'short'
        else:
            period_key = 'long'

        period = settings.ADDITIONAL_DATA_FREEZE_PERIOD_AFTER_COMPLETION[period_key]
        if self.form_values['user_declined']:
            period = settings.ADDITIONAL_DATA_FREEZE_PERIOD_AFTER_DECLINE[period_key]

        events = {
            'action': 'additional_data_asked',
        }

        with UPDATE(self.account, self.request.env, events):
            self.account.additional_data_asked = self.track.additional_data_asked
            self.account.additional_data_ask_next_datetime = datetime.now() + period
