# -*- coding: utf-8 -*-
import logging

from passport.backend.api.common.authorization import SessionScope
from passport.backend.api.common.common import extract_tld
from passport.backend.api.views.bundle.auth.base import BundleBaseAuthorizationMixin
from passport.backend.api.views.bundle.auth.token.forms import SubmitForm
from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.constants import X_TOKEN_OAUTH_SCOPE
from passport.backend.api.views.bundle.exceptions import AccountInvalidTypeError
from passport.backend.api.views.bundle.headers import (
    HEADER_CONSUMER_AUTHORIZATION,
    HEADER_CONSUMER_CLIENT_IP,
)
from passport.backend.api.views.bundle.mixins import BundleAccountGetterMixin
from passport.backend.core import authtypes
from passport.backend.core.builders.staff import (
    get_staff,
    StaffEntityNotFoundError,
    StaffTemporaryError,
)
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.loggers.statbox import StatboxLogger
from passport.backend.core.portallib import (
    is_yandex_ip,
    is_yandex_server_ip,
)
from passport.backend.core.utils.decorators import cached_property
from six.moves.urllib.parse import urlparse


log = logging.getLogger('passport.api.view.bundle.auth.token')


BROWSER_SCOPE = 'yadisk:browser_sync'


class SubmitView(BundleAccountGetterMixin,
                 BundleBaseAuthorizationMixin,
                 BaseBundleView):

    required_headers = [
        HEADER_CONSUMER_AUTHORIZATION,
        HEADER_CONSUMER_CLIENT_IP,
    ]
    required_grants = ['auth_by_token.base']

    basic_form = SubmitForm

    track_type = 'authorize'

    required_scope = [X_TOKEN_OAUTH_SCOPE, BROWSER_SCOPE]

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode='any_auth',
            type='x-token',
            ip=self.client_ip,
            consumer=self.consumer,
        )

    def get_passport_host(self, retpath):
        parsed = urlparse(retpath)
        if not parsed.hostname:
            return
        
        foreign_domain = extract_tld(parsed.hostname, settings.FOREIGN_DOMAINS_WITH_PASSPORT)
        if foreign_domain:
            return settings.FOREIGN_PASSPORT_BASE_URL_TEMPLATE % {'foreign_domain': foreign_domain}

        tld = extract_tld(parsed.hostname, settings.PASSPORT_TLDS)
        if tld:
            return settings.PASSPORT_BASE_URL_TEMPLATE % {'tld': tld}

    def process_request(self):
        self.process_basic_form()
        self.get_account_by_oauth_token(
            required_scope=self.required_scope,
        )
        self.statbox.bind_context(
            client_id=self.oauth_info['client_id'],
            scope=', '.join(self.oauth_info['scope']),
        )
        self.assert_allowed_to_get_cookie()
        if (
            settings.IS_INTRANET and
            not (
                is_yandex_ip(str(self.client_ip)) or
                is_yandex_server_ip(str(self.client_ip))
            )
        ):
            # В ятиме запрещаем роботам обменивать токен на куку из внешней сети
            try:
                staff_info = get_staff().get_user_info(
                    uid=self.account.uid,
                    fields=['official.is_robot'],
                    oauth_token=settings.STAFF_TOKEN,
                )
                if staff_info['official']['is_robot']:
                    log.debug('User %s is a robot, according to Staff', self.account.uid)
                    raise AccountInvalidTypeError()
            except StaffEntityNotFoundError:
                log.debug('User %s not found in Staff', self.account.uid)
                raise AccountInvalidTypeError()
            except StaffTemporaryError:
                # считаем, что таймаутящийся Стафф - не повод запрещать авторизацию
                pass

        self.create_track(self.track_type)
        self.statbox.bind_context(track_id=self.track_id)

        with self.track_transaction.commit_on_error():
            self.fill_track_with_account_data(
                allow_create_session=True,
                auth_source=authtypes.AUTH_SOURCE_XTOKEN,
                session_scope=SessionScope.xsession,
            )
            self.track.retpath = self.form_values['retpath']
            self.track.login_id = self.login_id

        self.response_values.update(
            track_id=self.track_id,
            passport_host=self.get_passport_host(self.track.retpath),
        )

        self.statbox.log(action='completed')
