# -*- coding: utf-8 -*-
import logging

from passport.backend.api.views.bundle.autofill.exceptions import PartnerIdNotMatchedError
from passport.backend.api.views.bundle.autofill.forms import ValidatePartnerForm
from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.core.turboapp_partners import (
    get_partner_id_by_host,
    get_partner_id_by_url,
)
from passport.backend.utils.common import url_to_ascii
from passport.backend.utils.string import smart_text
from six.moves.urllib.parse import (
    urlsplit,
    urlunsplit,
)


log = logging.getLogger(__name__)


PERMISSIONS_ALL = 'all'
PERMISSIONS_NO_PHONE = 'no_phone'
PERMISSIONS_NONE = 'none'


class ValidatePartnerView(BaseBundleView):
    required_grants = ['autofill.validate_partner']
    basic_form = ValidatePartnerForm

    def _build_normalized_host(self, url_parts):
        return url_parts.netloc.lower().encode('idna')

    def _build_normalized_url(self, url_parts):
        # отрезаем query и fragment, добавляем слеш в path, приводим к нижнему регистру, кодируем в ascii
        scheme, netloc, path = url_parts.scheme, url_parts.netloc, url_parts.path
        if path and path[-1] != '/':
            path += '/'
        return url_to_ascii(
            urlunsplit((scheme, netloc, path, '', '')).lower(),
        )

    def get_permissions(self, url, partner_id):
        permissions = PERMISSIONS_NONE
        url_parts = urlsplit(smart_text(url))

        # Сначала пытаемся найти партнёра по полному урлу...
        try:
            normalized_url = self._build_normalized_url(url_parts)
        except UnicodeError:
            # не кодируемого в idna урла заведомо не может быть в таблице
            return permissions

        expected_partner_id = get_partner_id_by_url(normalized_url)
        if expected_partner_id is not None:
            permissions = PERMISSIONS_ALL
        else:
            # ... а если не удалось - то по хосту
            normalized_host = self._build_normalized_host(url_parts)
            expected_partner_id = get_partner_id_by_host(normalized_host)
            if expected_partner_id is not None:
                permissions = PERMISSIONS_NO_PHONE

        if expected_partner_id is not None and expected_partner_id != partner_id:
            log.debug('partner_id=%r was found in YDB, but partner_id=%r was passed', expected_partner_id, partner_id)
            raise PartnerIdNotMatchedError()
        return permissions

    def process_request(self):
        self.process_basic_form()
        partner_id = self.form_values['partner_id']

        self.response_values.update(
            partner_id=partner_id,
            permissions=self.get_permissions(
                url=self.form_values['page_url'],
                partner_id=partner_id,
            ),
        )
