# -*- coding: utf-8 -*-

import logging

from flask import request
from flask.views import View
from passport.backend.api.common.common import track_to_response
from passport.backend.api.common.decorators import (
    get_request_files,
    get_request_values,
)
from passport.backend.api.common.errors import log_internal_error
from passport.backend.api.common.format_response import (
    ok_response,
    simple_error_response,
)
from passport.backend.api.common.logs import setup_log_prefix
from passport.backend.api.common.processes import is_process_allowed
from passport.backend.api.forms.base import (
    RequiredTrackedConsumerForm,
    TrackedConsumerForm,
)
import passport.backend.api.views.bundle.exceptions as exceptions
from passport.backend.api.yasms import api as yasms_api
from passport.backend.core import validators
from passport.backend.core.builders.afisha import AfishaApiTemporaryError
from passport.backend.core.builders.antifraud import AntifraudApiTemporaryError
from passport.backend.core.builders.avatars_mds_api import AvatarsMdsApiTemporaryError
from passport.backend.core.builders.bilet_api import BiletApiTemporaryError
from passport.backend.core.builders.billing.exceptions import BillingTemporaryError
from passport.backend.core.builders.blackbox import (
    BlackboxInvalidResponseError,
    BlackboxTemporaryError,
    BlackboxUnknownError,
    get_blackbox,
)
from passport.backend.core.builders.bot_api.exceptions import (
    BotApiInvalidRequestError,
    BotApiTemporaryError,
)
from passport.backend.core.builders.captcha import (
    CaptchaError,
    CaptchaLocateError,
)
from passport.backend.core.builders.collections import CollectionsApiTemporaryError
from passport.backend.core.builders.datasync_api import (
    DatasyncApiPermanentError,
    DatasyncApiTemporaryError,
)
from passport.backend.core.builders.drive_api.drive_api import DriveApiTemporaryError
from passport.backend.core.builders.federal_configs_api import FederalConfigsApiTemporaryError
from passport.backend.core.builders.geosearch import GeoSearchApiTemporaryError
from passport.backend.core.builders.historydb_api import (
    get_historydb_api,
    HistoryDBApiPermanentError,
    HistoryDBApiTemporaryError,
)
from passport.backend.core.builders.mail_apis.exceptions import (
    HuskyInvalidResponseError,
    HuskyTemporaryError,
)
from passport.backend.core.builders.market import MarketContentApiTemporaryError
from passport.backend.core.builders.messenger_api import MessengerApiTemporaryError
from passport.backend.core.builders.music_api import (
    MusicApiPermanentError,
    MusicApiTemporaryError,
)
from passport.backend.core.builders.oauth import OAuthTemporaryError
from passport.backend.core.builders.octopus import OctopusTemporaryError
from passport.backend.core.builders.perimeter_api.exceptions import (
    PerimeterApiPermanentError,
    PerimeterApiTemporaryError,
)
from passport.backend.core.builders.phone_squatter import PhoneSquatterTemporaryError
from passport.backend.core.builders.push_api.exceptions import (
    PushApiInvalidRequestError,
    PushApiTemporaryError,
)
from passport.backend.core.builders.social_api import (
    get_social_api,
    SocialApiRequestError,
    SocialApiTemporaryError,
)
from passport.backend.core.builders.social_broker import (
    SocialBrokerRequestError,
    SocialBrokerTemporaryError,
)
from passport.backend.core.builders.trust_api import (
    TrustPermanentError,
    TrustTemporaryError,
)
from passport.backend.core.builders.video import VideoApiTemporaryError
from passport.backend.core.builders.yasms import (
    get_yasms,
    YaSmsTemporaryError,
)
from passport.backend.core.dbmanager.exceptions import DBError
from passport.backend.core.exceptions import WrongHostError
from passport.backend.core.grants import (
    check_grant,
    GrantsError,
    InvalidSourceError,
    MissingGrantsError,
    MissingTicketError,
    TicketParsingError,
)
from passport.backend.core.logbroker.exceptions import TransportError as LogbrokerTransportError
from passport.backend.core.models.phones.phones import (
    OperationExpired as PhoneOperationExpired,
    RemoveBankPhoneNumberError,
)
from passport.backend.core.redis_manager.redis_manager import (
    RedisError,
    RedisWatchError,
)
from passport.backend.core.serializers.eav.exceptions import (
    EavDeletedObjectNotFound,
    EavUpdatedObjectNotFound,
)
from passport.backend.core.tracks.exceptions import (
    BaseTrackNotFoundError,
    ConcurrentTrackOperationError,
)
from passport.backend.core.tracks.track_manager import TrackManager
from passport.backend.core.types.account.account import (
    ACCOUNT_TYPE_FEDERAL,
    ACCOUNT_TYPE_KIDDISH,
    ACCOUNT_TYPE_KINOPOISK,
    ACCOUNT_TYPE_KOLONKISH,
    ACCOUNT_TYPE_LITE,
    ACCOUNT_TYPE_MAILISH,
    ACCOUNT_TYPE_NEOPHONISH,
    ACCOUNT_TYPE_NORMAL,
    ACCOUNT_TYPE_PDD,
    ACCOUNT_TYPE_PHONISH,
    ACCOUNT_TYPE_SCHOLAR,
    ACCOUNT_TYPE_SOCIAL,
    ACCOUNT_TYPE_UBER,
    ACCOUNT_TYPE_YAMBOT,
)
from passport.backend.core.utils.decorators import cached_property
from passport.backend.core.ydb.exceptions import (
    YdbPermanentError,
    YdbTemporaryError,
)
from passport.backend.utils.common import ClassMapping
from passport.backend.utils.string import smart_text


log = logging.getLogger('passport.api.view.bundle')

OAUTH_HEADER_PREFIX = 'oauth '


class BaseBundleView(View):
    """
    Базовый класс для всех bundle-вьюшек. Содержит основные свойства и методы,
    полезные всем потомкам. Для непосредственной обработки запроса, класс-потомок
    обязан переопределить метод process_request().
    """

    # Класс основной валидационной формы, с данными которой будет
    # работать потомок. Используется в process_basic_form().
    basic_form = None

    # Требуется ли указание идентификатора трека. Можно переопределить в потомке.
    require_track = False

    # Список необходимых хедеров. Хедера берутся из passport.backend.api.views.bundle.headers
    required_headers = None

    # Список необходимых грантов
    required_grants = None

    # Список дополнительных грантов на работу с определенным типом аккаунта.
    grants_for_account_type = None

    # Имя процесса, к которому принадлежит View
    process_name = None

    # Требуется ли наличие процесса в треке (требование проверяется только при заданном треке)
    require_process = False

    # Имена процессов, которым разрешено использовать (вызывать) данное API;
    # имя процесса текущего View автоматически добавляется в этот список
    allowed_processes = None

    # Statbox логгер
    statbox = None

    # Список необходимых скоупов у User ticket'а
    required_user_ticket_scopes = None

    exceptions_mapping = ClassMapping([
        (AfishaApiTemporaryError, exceptions.AfishaApiUnavailableError),
        (AntifraudApiTemporaryError, exceptions.AntifraudApiUnavailableError),
        (AvatarsMdsApiTemporaryError, exceptions.AvatarsMdsApiUnavailableError),
        (BaseTrackNotFoundError, exceptions.TrackNotFoundError),
        (BiletApiTemporaryError, exceptions.BiletApiUnavailableError),
        (BillingTemporaryError, exceptions.BillingUnavailableError),
        (BlackboxInvalidResponseError, exceptions.BlackboxPermanentError),
        (BlackboxTemporaryError, exceptions.BlackboxUnavailableError),
        (BlackboxUnknownError, exceptions.BlackboxUnavailableError),
        (BotApiInvalidRequestError, exceptions.BotApiRequestFailed),
        (BotApiTemporaryError, exceptions.BotApiUnavailableError),
        (CaptchaError, exceptions.CaptchaUnavailableError),
        (CaptchaLocateError, exceptions.CaptchaNotShownError),
        (CollectionsApiTemporaryError, exceptions.CollectionsApiUnavailableError),
        (ConcurrentTrackOperationError, exceptions.InternalTemporaryError),
        (DBError, exceptions.DatabaseUnavailableError),
        (DatasyncApiTemporaryError, exceptions.DatasyncApiUnavailableError),
        (DatasyncApiPermanentError, exceptions.DatasyncApiUnavailableError),
        (DriveApiTemporaryError, exceptions.DriveApiUnavailableError),
        (EavDeletedObjectNotFound, exceptions.InternalTemporaryError),
        (EavUpdatedObjectNotFound, exceptions.InternalTemporaryError),
        (FederalConfigsApiTemporaryError, exceptions.FederalConfigsApiUnavailableError),
        (GeoSearchApiTemporaryError, exceptions.GeoSearchApiUnavailableError),
        (HistoryDBApiPermanentError, exceptions.HistoryDBPermanentError),
        (HistoryDBApiTemporaryError, exceptions.HistoryDBApiUnavailableError),
        (HuskyInvalidResponseError, exceptions.HuskyApiPermanentError),
        (HuskyTemporaryError, exceptions.HuskyApiUnavailableError),
        (LogbrokerTransportError, exceptions.LogbrokerTemporaryError),
        (MarketContentApiTemporaryError, exceptions.MarketApiUnavailableError),
        (MessengerApiTemporaryError, exceptions.MessengerApiUnavailableError),
        (MusicApiPermanentError, exceptions.MusicApiPermanentError),
        (MusicApiTemporaryError, exceptions.MusicApiUnavailableError),
        (OAuthTemporaryError, exceptions.OAuthUnavailableError),
        (OctopusTemporaryError, exceptions.OctopusUnavailableError),
        (PerimeterApiPermanentError, exceptions.PerimeterApiPermanentError),
        (PerimeterApiTemporaryError, exceptions.PerimeterApiUnavailableError),
        (PhoneOperationExpired, exceptions.InternalTemporaryError),
        (PhoneSquatterTemporaryError, exceptions.PhoneSquatterUnavailableError),
        (PushApiInvalidRequestError, exceptions.PushApiUnavailableError),
        (PushApiTemporaryError, exceptions.PushApiUnavailableError),
        (RedisError, exceptions.RedisUnavailableError),
        (RedisWatchError, exceptions.InternalTemporaryError),
        (RemoveBankPhoneNumberError, exceptions.PhoneIsBankPhoneNumberAliasError),
        (SocialApiRequestError, exceptions.SocialApiPermanentError),
        (SocialApiTemporaryError, exceptions.SocialApiUnavailableError),
        (SocialBrokerRequestError, exceptions.SocialBrokerPermanentError),
        (SocialBrokerTemporaryError, exceptions.SocialBrokerUnavailableError),
        (TrustTemporaryError, exceptions.TrustBindingsUnavailableError),
        (TrustPermanentError, exceptions.TrustBindingsUnavailablePermanentError),
        (VideoApiTemporaryError, exceptions.VideoApiUnavailableError),
        (WrongHostError, exceptions.InvalidHostError),
        (YaSmsTemporaryError, exceptions.YaSmsUnavailableError),
        (YdbTemporaryError, exceptions.YdbUnavailableError),
        (YdbPermanentError, exceptions.YdbPermanentError),
    ])

    @property
    def track_uid(self):
        return int(self.track.uid)

    @cached_property
    def blackbox(self):
        return get_blackbox()

    @cached_property
    def yasms(self):
        return get_yasms()

    @cached_property
    def social_api(self):
        return get_social_api()

    @cached_property
    def historydb_api(self):
        return get_historydb_api()

    @cached_property
    def yasms_api(self):
        return yasms_api.Yasms(self.blackbox, self.yasms, self.request.env)

    # Объединённый словарь с данными из request_values и path_values.
    @cached_property
    def all_values(self):
        values = self.request_values
        values.update(self.path_values)
        return values

    # Возвращает словарь с данными из запроса: для GET-запросов выдаёт данные из
    # query-части, для POST-запросов - из body-части. А также добавляет поле
    # consumer только из query-части.
    @cached_property
    def request_values(self):
        return get_request_values()

    # Возвращает словарь с данными о переданных в форме файлах. Ключом является
    # имя поля формы, значением - тупл переданных файлов, преобразованных
    # в core-тип File.
    @cached_property
    def request_files(self):
        return get_request_files()

    # Отдаёт менеджер треков.
    @cached_property
    def track_manager(self):
        return TrackManager()

    # Создаёт объект транзакции для сохранённого во вьюшке трека.
    @property
    def track_transaction(self):
        return self.track_manager.transaction(track=self.track)

    @property
    def request(self):
        return request

    @property
    def headers(self):
        return request.headers

    @property
    def cookies(self):
        return request.env.cookies

    @property
    def consumer_ip(self):
        return request.env.consumer_ip

    @property
    def client_ip(self):
        return request.env.user_ip

    @property
    def host(self):
        return request.env.host

    @property
    def user_agent(self):
        return request.env.user_agent

    @property
    def referer(self):
        return request.env.referer

    @property
    def authorization(self):
        return request.env.authorization

    @property
    def service_ticket(self):
        return request.env.service_ticket

    @cached_property
    def oauth_token(self):
        auth_header = self.authorization
        if not auth_header.lower().startswith(OAUTH_HEADER_PREFIX):
            raise exceptions.AuthorizationHeaderError()
        return auth_header[len(OAUTH_HEADER_PREFIX):].strip()

    @classmethod
    def as_view(cls, name=None, *args, **kwargs):
        name = name or cls.__name__
        return super(BaseBundleView, cls).as_view(name, *args, **kwargs)

    def __init__(self):
        # Имя потребителя, указанного в запросе.
        self.consumer = None

        # Словарь с данными, полученными при валидации основной формы в process_basic_form().
        self.form_values = {}

        # Словарь с данными, которые вытащил роутер из path-части урла запроса.
        self.path_values = {}

        # Словарь для хранения данных, которые будут выданы в http-ответе как json.
        self.response_values = {}

        # Свойство для хранения трека, если ручка с ним работает.
        self.track = None

        # Идентификатор трека: либо указанный в запросе, либо заново созданный.
        self.track_id = None

        # Вся информация о пользователе из ответа ЧЯ
        self.account = None
        self.tvm_user_ticket = None

        # Объект BaseState.
        self.state = None

        super(BaseBundleView, self).__init__()

    def dispatch_request(self, **kwargs):
        """
        Первоочерёдный метод вьюшки, обрабатывающий пришедший запрос.

        Запоминает входящие параметры как path_values, то есть данные из path-части
        урла запроса, полученные роутером. Прогоняет запрос через основную форму
        для получения consumer и track_id. Вызывает process_request(), переопределённый
        в классе-потомке. В случае выброшенного исключения, включает его обработку
        и выдачу ошибочного ответа. Если исключения не было, выдаёт успешный ответ.

        @param kwargs: Словарь с данными из path-части урла.
        """
        self.path_values = kwargs

        try:
            self.process_root_form()
            if self.required_grants:
                self.check_grant(lambda: self.required_grants)

            if self.required_headers:
                self.check_headers(self.required_headers)

            self.process_request()
        except Exception as e:
            return self.respond_error(e)

        return self.respond_success()

    def process_request(self):
        """
        Основной метод вьюшки, обрабатывающий пришедший запрос. Должен быть
        переопределен в классе-потомке.
        """
        raise NotImplementedError()

    def process_error(self, exception):
        try:
            response_exc = self.exceptions_mapping[exception.__class__]
            return response_exc()
        except KeyError:
            pass

        if isinstance(exception, exceptions.BaseBundleError):
            return exception

        return exceptions.UnhandledBundleError()

    def error_response(self, errors, **extra_response_values):
        return simple_error_response(errors, **extra_response_values)

    def respond_error(self, exception):
        """
        Генерирует и отдаёт ошибочный json-ответ с данными из response_values и
        дополнительными полями status=error и errors=[коды-ошибок].

        Указанное исключение должно быть обработано в методе process_error
        и преобразовано к потомку BaseBundleError в котором переопределены свойства error и errors
        - именно они используются как диагностика ошибки в ответе. Если исключение не отдаёт
        эти коды ошибок, process_error должно вернуть UnhandledBundleError
        со стандартным кодом ошибки - 'exception.unhandled', а само неизвестное исключение
        будет залогировано.

        Есть специфичные случаи:
        - ошибка GrantsError, TicketParsingException, InvalidSourceError, MissingTicketError
        которые кидает метод check_grant() при отсутствии необходимого гранта или правильного тикета.
        Эти ошибки обрабатываем отдельно с помощью respond_grants_forbidden() и respond_service_ticket_forbidden()
        - ошибка BaseTrackNotFoundError, которую кидает track_manager в случае,
        если трэк или нода redis-а не найдены.
        - ошибки при недоступности ЧЯ, YaSMS, DB, Redis преобразуются в
        backend.%s_failed.

        @param exception: Исключение для выдачи правильной диагностики полученной ошибки.
        @return: Объект JsonLoggedResponse.
        """
        setup_log_prefix(self.account)

        log.debug(u'Process exception to response %s: %s' % (type(exception).__name__, smart_text(exception)))

        if isinstance(exception, (TicketParsingError, InvalidSourceError, MissingTicketError)):
            return self.respond_service_ticket_forbidden(exception)
        if isinstance(exception, MissingGrantsError):
            return self.respond_grants_forbidden(exception)
        if isinstance(exception, exceptions.TvmUserTicketInvalidError):
            return self.respond_user_ticket_forbidden(exception)

        # Обработаем ошибки, получив предусмотренное view-исключение
        processed_error = self.process_error(exception)

        errors = processed_error.errors

        # Запишем в лог необработанное исключение
        if isinstance(processed_error, exceptions.UnhandledBundleError):
            log_internal_error(exception)

        return self.error_response(errors, **self.response_values)

    def respond_grants_forbidden(self, exception):
        """
        Генерирует и отдаёт ошибочный json-ответ об отсутствии необходимых доступов.

        Скопировано из common.error_handler, так как хотим отдавать 403 код с
        аналогичным ответом в других api-ручках.

        @param exception: Исключение GrantsError.
        @return: Объект JsonLoggedResponse.
        """

        error = u'Access denied for ip: %s; consumer: %s; tvm_client_id: %s. Required grants: %s' % (
            exception.ip,
            exception.consumer,
            exception.tvm_client_id,
            exception.missing,
        )
        return self.error_response(['access.denied'], code=403, error_message=error)

    def respond_service_ticket_forbidden(self, exception):
        """
        Генерирует и отдаёт ошибочный json-ответ об отсутствии правильного тикета.

        @param exception: Исключение TicketParsingException или InvalidSourceError.
        @return: Объект JsonLoggedResponse.
        """

        error = u'Access denied for ip: %s; consumer: %s; tvm_client_id: %s' % (
            exception.ip,
            exception.consumer,
            exception.tvm_client_id,
        )
        return self.error_response(['access.denied'], code=403, error_message=error)

    def respond_user_ticket_forbidden(self, exception):
        """
        Генерирует и отдаёт ошибочный json-ответ об отсутствии правильного тикета.

        @param exception: Исключение TvmUserTicketInvalidError.
        @return: Объект JsonLoggedResponse.
        """

        if isinstance(exception, exceptions.TvmUserTicketMissingScopes):
            missing_scopes = ', '.join(exception.missing_scopes)
            ticket_scopes = ', '.join(exception.ticket_scopes)
            error = u'Ticket missing one of scopes %s (ticket scopes are %s)' % (missing_scopes, ticket_scopes)
        elif isinstance(exception, exceptions.TvmUserTicketNoUid):
            uids = ', '.join(map(str, exception.known_uids))
            error = u'Specified or default uid not in ticket (ticket uids are %s)' % uids
        else:
            error = smart_text(exception)
        return self.error_response(exception.errors, error_message=error)

    def ok_response(self, **response_values):
        return ok_response(**response_values)

    def respond_success(self):
        """
        Генерирует и отдаёт успешный json-ответ с данными из response_values и
        дополнительным полем status=ok.

        @return: Объект JsonLoggedResponse.
        """
        if self.state:
            self.state.update_response(self.response_values)

        setup_log_prefix(self.account)

        return self.ok_response(**self.response_values)

    def read_or_create_track(self, track_type, process_name=None):
        """
        Вызывает метод чтения трека, если в запросе был указан идентификатор
        трека, иначе вызывает метод создания нового трека.
        """
        if self.track_id:
            self.read_track()
        else:
            self.create_track(track_type, process_name=process_name)

    def create_track(self, track_type, process_name=None, ttl=None):
        """
        Создаёт новый трек с указанным типом и сохраняет его вместе с
        идентификатором в соответствующие свойства.
        """
        self.track = self.track_manager.create(
            track_type,
            self.consumer,
            process_name=process_name,
            ttl=ttl,
        )
        self.track_id = self.track.track_id

    def read_track(self):
        """
        Читает существующий трек по полученному ранее идентификатору и
        сохраняет его в соответствующее свойство.
        """
        self.track = self.track_manager.read(self.track_id)
        self.check_process_allowed()

    def check_header(self, header):
        """
        Проверяет наличие обязательного хедера в запросе. Если хедера нет,
        кидает исключение HeadersEmptyError.

        @param header: хедер из .headers
        @raise: HeadersEmptyError
        """
        self.check_headers([header])

    def check_headers(self, headers, check_any=False):
        """
        Проверяет наличие обязательных хедеров в запросе. Кидает исключение
        HeadersEmptyError с перечислением отсутствующих хедеров.

        @param headers: Список хедеров из .headers
        @param check_any: Если True, то при наличии хотя бы одного заголовка headers ошибки не будет
        @raise: HeadersEmptyError
        """
        codes = [
            header.code_for_error for header in headers
            if (
                header.name not in self.headers or
                (not self.headers.get(header.name, '').strip() and not header.allow_empty_value)
            )
        ]

        if codes:
            if check_any and len(codes) < len(headers):
                pass  # ок, ошибок меньше, чем заголовков, значит хотя бы что-то есть!
            else:
                raise exceptions.HeadersEmptyError(codes)

    def check_grant(self, grant):
        """
        Проверяет наличие указанного гранта у потребителя и выбрасывает
        ошибку при его отсутствии.

        @param grant: Имя необходимого гранта.
        @raise: GrantsError
        """
        check_grant(grant, self.consumer_ip, self.consumer, self.service_ticket)

    def has_grant(self, grant):
        """
        Проверяет наличие указанного гранта у потребителя.

        @param grant: Имя необходимого гранта.
        @returns: bool
        """
        try:
            check_grant(grant, self.consumer_ip, self.consumer, self.service_ticket)
        except GrantsError:
            return False
        else:
            return True

    def check_grants_for_account_type(self, grants=None):
        """
        Проверяем, что у консумера есть грант, соответствующий доступу к типу нашего
        аккаунта. Применяется для случаев, когда определенному консумеру надо выдать
        доступ на операции только с определенным типом аккаунта (e.g. ПДД, Кинопоиск).

        Каждый из элементов словаря grants_for_account_type на ручке обозначает
        строчку с соответствующим коду аккаунта грантом.
        Пример: {'pdd': 'account.delete_pdd'}.

        Возможные коды: normal, pdd, lite, mailish, social, phonish.

        Специальный аргумент 'any' обозначает "глобальный" грант, обладатель
        которого может совершать операции с любыми аккаунтами.
        """
        grants = grants or self.grants_for_account_type

        account_type_mapping = {
            ACCOUNT_TYPE_NORMAL: 'normal',
            ACCOUNT_TYPE_PDD: 'pdd',
            ACCOUNT_TYPE_LITE: 'lite',
            ACCOUNT_TYPE_MAILISH: 'mailish',
            ACCOUNT_TYPE_SOCIAL: 'social',
            ACCOUNT_TYPE_PHONISH: 'phonish',
            ACCOUNT_TYPE_KINOPOISK: 'kinopoisk',
            ACCOUNT_TYPE_UBER: 'uber',
            ACCOUNT_TYPE_KOLONKISH: 'kolonkish',
            ACCOUNT_TYPE_YAMBOT: 'yambot',
            ACCOUNT_TYPE_NEOPHONISH: 'neophonish',
            ACCOUNT_TYPE_KIDDISH: 'kiddish',
            ACCOUNT_TYPE_SCHOLAR: 'scholar',
            ACCOUNT_TYPE_FEDERAL: 'federal',
        }

        if grants is None:
            raise ValueError("Account specific grants has not been specified!")

        anything_grant = grants.get('any')
        if not anything_grant:
            raise ValueError('Grant account type "any" has not been specified!')

        # Если у пользователя есть "зонтичный" грант на любые аккаунты, то
        # в дальнейшей проверке более специфичного гранта нет смысла.
        if self.has_grant(anything_grant):
            return

        type_code = account_type_mapping.get(self.account.type)
        applicable_grant = grants.get(type_code)
        if applicable_grant:
            self.check_grant(applicable_grant)
        else:
            # Если не указан специфичный грант, то выводим ошибку с "зонтичным" грантом.
            self.check_grant(anything_grant)

    def check_process_allowed(self):
        """
        Проверяет, что доступ к API разрешен для записанного в треке процесса.
        """
        allowed_processes = list(self.allowed_processes or [])
        if self.process_name:
            allowed_processes.append(self.process_name)
        if not is_process_allowed(
            allowed_processes,
            is_process_required=self.require_process,
            track=self.track,
        ):
            raise exceptions.InvalidTrackStateError()

    def process_form(self, form, values):
        """
        Валидирует данные с помощью указанной формы.

        Если валидация пройдена успешно, возвращает результат обработки. Обычно,
        это те же данные, но слегка обработанные: у полей убраны лишние пробелы,
        некоторые могут быть преобразованы в сложные объекты и так далее.

        Если валидация пройдена с ошибками, кидает исключение ValidationFailedError.
        С помощью него можно получить список всех найденных ошибок.

        @param form: Объект формы, с помощью которой будет проводиться
        валидация. Форма должна быть отнаследована от passport.validators.Schema
        (formencode.Scheme только с нашими патчами).
        @param values: Словарь с данными, которые нужно прогнать через форму.
        @return: Обработанный словарь с данными.
        @raise: ValidationFailedError
        """
        state = validators.State(self.request.env, self.request_files)

        try:
            result = form.to_python(values, state)
        except validators.Invalid as e:
            exception = exceptions.ValidationFailedError.from_invalid(e)
            log.info(
                'Form validation: status=error form=%s errors=%s original_error="%s"',
                form.__class__.__name__,
                ','.join(exception.errors),
                e,
            )
            raise exception

        return result

    def process_basic_form(self):
        """
        Валидирует весь запрос с помощью основной формы, определённой в потомке,
        и сохраняет результат в form_values.
        """
        self.form_values = self.process_form(self.basic_form(), self.all_values)

    def process_root_form(self):
        """
        Проверяет и сохраняет в соответствующие свойства основные (базовые)
        данные запроса: consumer и track_id.

        Если потомок взвёл require_track, наличие идентификатора трека становится обязательным.
        """
        form = RequiredTrackedConsumerForm if self.require_track else TrackedConsumerForm

        values = self.process_form(form(), self.request_values)

        self.consumer = values['consumer']
        self.track_id = values['track_id']

    def fill_response_with_track_fields(self, *field_names):
        """
        Копирует в ответ поля из трека. Пропускает поля со значением None.
        """
        self.response_values.update(track_to_response(self.track, field_names))

    def set_uid_to_track(self, uid, track=None):
        """Выставляет в трек uid, если его там ещё нет"""
        track = track or self.track
        if track.uid and str(track.uid) != str(uid):
            log.warning('Attempt to change uid in track: %s != %s', track.uid, uid)
            raise exceptions.InvalidTrackStateError()
        elif not track.uid:
            # Сбросим флаги авторизации, чтобы избежать уязвимостей при гонках.
            # TODO: избавиться от этого костыля, поменяв архитектуру уязвимых ручек
            track.allow_authorization = False
            track.allow_oauth_authorization = False
            track.uid = uid
