# -*- coding: utf-8 -*-

import logging

from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.exceptions import BillingUnavailableError
from passport.backend.api.views.bundle.headers import (
    HEADER_CLIENT_COOKIE,
    HEADER_CLIENT_HOST,
    HEADER_CLIENT_USER_AGENT,
    HEADER_CONSUMER_CLIENT_IP,
)
from passport.backend.api.views.bundle.mixins import (
    BundleAccountGetterMixin,
    BundleAccountPropertiesMixin,
)
from passport.backend.api.views.bundle.mixins.account import UserMetaDataMixin
from passport.backend.api.views.bundle.mixins.kolmogor import KolmogorMixin
from passport.backend.api.views.bundle.mixins.mail import MailMixin
from passport.backend.api.views.bundle.mixins.push import BundlePushMixin
from passport.backend.core.builders.billing import (
    BaseBillingError,
    BillingXMLRPC,
)
from passport.backend.core.logging_utils.loggers.statbox import StatboxLogger
from passport.backend.core.utils.decorators import cached_property

from .forms import (
    CheckBindingForm,
    CreateBindingForm,
    DoBindingForm,
    ListPaymentMethodsForm,
    UnbindCardForm,
)


log = logging.getLogger('passport.api.views.bundle.billing.controllers')


class BaseBillingView(BaseBundleView, BundleAccountGetterMixin, BundleAccountPropertiesMixin):
    require_track = True

    required_headers = (
        HEADER_CONSUMER_CLIENT_IP,
        HEADER_CLIENT_USER_AGENT,
        HEADER_CLIENT_HOST,
        HEADER_CLIENT_COOKIE,
    )

    method = None

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode='billing',
            method=self.method,
            ip=self.client_ip,
            host=self.host,
            user_agent=self.user_agent,
            yandexuid=self.cookies.get('yandexuid'),
            consumer=self.consumer,
        )

    def make_billing_request(self):
        raise NotImplementedError()  # pragma: no cover

    def post_request(self):
        pass

    def get_account(self):
        self.get_account_from_session()

    def process_request(self):
        self.process_basic_form()
        self.read_track()

        # Всегда получаем дефолтный аккаунт сессии, т.к. API Биллинга в части ручек
        # также работает с сессией и только с дефолтным аккаунтом
        self.get_account()
        self.statbox.bind_context(uid=self.account.uid)

        try:
            response = self.make_billing_request()
            self.response_values.update(
                billing_response=response,
                sensitive_fields=['billing_response'],
            )
            self.statbox.log(action='called')
            self.post_request()
        except BaseBillingError:
            raise BillingUnavailableError()


class ListPaymentsMethodView(BaseBillingView):
    method = 'list_payment_methods'

    required_grants = ['billing.list_payment_methods']

    basic_form = ListPaymentMethodsForm

    def make_billing_request(self):
        return BillingXMLRPC().list_payment_methods(
            self.account.uid,
            self.client_ip,
            self.form_values['ym_schema'],
        )


class CreateBindingView(
    UserMetaDataMixin,
    BaseBillingView,
):
    method = 'create_binding'

    required_grants = ['billing.create_binding']

    basic_form = CreateBindingForm

    def get_account(self):
        self.get_account_from_session(emails=True)

    def make_billing_request(self):
        return BillingXMLRPC().create_binding(
            self.account.uid,
            **self.form_values
        )


class DoBindingView(BaseBillingView):
    method = 'do_binding'

    required_grants = ['billing.do_binding']

    basic_form = DoBindingForm

    def make_billing_request(self):
        return BillingXMLRPC().do_binding(
            self.form_values['purchase_token'],
        )


class CheckBindingView(BaseBillingView):
    method = 'check_binding'

    required_grants = ['billing.check_binding']

    basic_form = CheckBindingForm

    def make_billing_request(self):
        return BillingXMLRPC().check_binding(
            self.form_values['purchase_token'],
        )


class UnbindCardView(
    MailMixin,
    UserMetaDataMixin,
    BaseBillingView,
    BundlePushMixin,
    KolmogorMixin,
):
    method = 'unbind_card'

    required_grants = ['billing.unbind_card']

    basic_form = UnbindCardForm

    def __init__(self):
        super(UnbindCardView, self).__init__()
        self.card_number = None

    def get_account(self):
        self.get_account_from_session(emails=True)

    def make_billing_request(self):
        response = BillingXMLRPC().list_payment_methods(
            self.account.uid,
            self.client_ip,
        )
        self.card_number = response.get('payment_methods', dict()).get(self.form_values['card'], dict()).get('number')

        return BillingXMLRPC().unbind_card(
            self.cookies['Session_id'],
            self.client_ip,
            self.host,
            self.form_values['card'],
        )

    def post_request(self):
        self.send_account_modification_push(
            event_name='card_change',
            context=dict(track_id=self.track_id),
        )
        self.send_account_modification_mail(
            event_name='card_delete',
            context=dict(
                NUMBER=self.card_number or '****',
            ),
        )
