# -*- coding: utf-8 -*-
import logging

from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.billing.trust_3ds.exceptions import (
    Trust3DSFailedError,
    Trust3DSNotEnoughFundsError,
    Trust3DSNotPassedError,
)
from passport.backend.api.views.bundle.billing.trust_3ds.forms import Trust3DSSubmitForm
from passport.backend.api.views.bundle.exceptions import InvalidTrackStateError
from passport.backend.api.views.bundle.headers import (
    HEADER_CLIENT_COOKIE,
    HEADER_CLIENT_HOST,
    HEADER_CLIENT_USER_AGENT,
    HEADER_CONSUMER_CLIENT_IP,
)
from passport.backend.api.views.bundle.mixins import BundleAccountGetterMixin
from passport.backend.core.builders.trust_api import get_trust_payments
from passport.backend.core.builders.trust_api.trust_payments import (
    PAYMENT_RESP_CODE_NOT_ENOUGH_FUNDS,
    PAYMENT_STATUS_AUTHORIZED,
    PAYMENT_STATUS_CANCELLED,
    PAYMENT_STATUS_STARTED,
)
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.loggers import StatboxLogger
from passport.backend.core.utils.decorators import cached_property


log = logging.getLogger(__name__)


class BaseTrust3DSView(BaseBundleView, BundleAccountGetterMixin):
    require_track = True
    mode = None

    required_headers = (
        HEADER_CLIENT_COOKIE,
        HEADER_CLIENT_HOST,
        HEADER_CLIENT_USER_AGENT,
        HEADER_CONSUMER_CLIENT_IP,
    )

    @cached_property
    def statbox(self):
        return StatboxLogger(
            consumer=self.consumer,
            track_id=self.track_id,
            ip=self.client_ip,
            user_agent=self.user_agent,
            mode=self.mode,
            yandexuid=self.cookies.get('yandexuid'),
        )

    @cached_property
    def trust_api(self):
        return get_trust_payments()


class BaseTrust3DSManageView(BaseTrust3DSView):
    required_grants = ['billing.trust_3ds']

    def process_request(self):
        if self.basic_form:
            self.process_basic_form()
        self.read_track()
        if not self.track.uid:
            raise InvalidTrackStateError()

        self.get_account_from_session()
        self.statbox.bind_context(uid=self.account.uid)

        self._process()

    def _process(self):
        raise NotImplementedError()


class Trust3DSSubmitView(BaseTrust3DSManageView):
    mode = '3ds_start_payment'
    basic_form = Trust3DSSubmitForm

    def _process(self):
        if not self.track.paymethod_id:
            raise InvalidTrackStateError()

        purchase_token = self.trust_api.create_basket(
            uid=self.account.uid,
            paymethod_id=self.track.paymethod_id,
            product_id=settings.TRUST_3DS_CHALLENGE_PRODUCT_ID,
            amount=settings.TRUST_3DS_CHALLENGE_PRODUCT_PRICE,
            return_path=self.form_values['frontend_url'],
            use_new_trust_form=self.form_values['use_new_trust_form'],
            use_mobile_layout=self.form_values['use_mobile_layout'],
            login_id=self.login_id,
            is_chaas=True,
        )
        self.trust_api.start_payment(
            uid=self.account.uid,
            purchase_token=purchase_token,
        )

        with self.track_transaction.rollback_on_error() as track:
            track.purchase_token = purchase_token
            track.payment_status = None

        self.statbox.log(status='ok')


class Trust3DSCommitView(BaseTrust3DSManageView):
    mode = '3ds_verify_status'

    def _process(self):
        if not self.track.purchase_token:
            raise InvalidTrackStateError()

        basket_status = self.trust_api.get_basket_status(
            uid=self.account.uid,
            purchase_token=self.track.purchase_token,
        )
        payment_status = basket_status['payment_status']
        self.statbox.log(status=payment_status)

        if payment_status == PAYMENT_STATUS_STARTED:
            url_3ds = basket_status.get('payment_url') or basket_status['3ds_transaction_info'].get('redirect_url')
            self.response_values.update(url_3ds=url_3ds)
            raise Trust3DSNotPassedError()
        elif payment_status == PAYMENT_STATUS_CANCELLED:
            # прошли весь процесс, отменили платёж, больше ничего делать не надо
            pass
        elif payment_status == PAYMENT_STATUS_AUTHORIZED:
            with self.track_transaction.rollback_on_error():
                self.track.payment_status = payment_status

                self.trust_api.cancel_payment(
                    uid=self.account.uid,
                    purchase_token=self.track.purchase_token,
                )
        else:
            self.response_values.update(payment_status=payment_status)
            log.debug('3DS failed: status=%s, description=%s', payment_status, basket_status.get('payment_resp_desc'))

            if basket_status['payment_resp_code'] == PAYMENT_RESP_CODE_NOT_ENOUGH_FUNDS:
                raise Trust3DSNotEnoughFundsError()

            raise Trust3DSFailedError()
