# -*- coding: utf-8 -*-
from datetime import (
    datetime,
    timedelta,
)

from passport.backend.api.common.webauthn import get_suggested_webauthn_credentials
from passport.backend.api.views.bundle.mixins.phone import OctopusAvailabilityMixin
from passport.backend.core.am_pushes.subscription_manager import get_pushes_subscription_manager
from passport.backend.core.builders.push_api import BasePushApiError
from passport.backend.core.builders.trust_api import PAYMENT_STATUS_AUTHORIZED
from passport.backend.core.conf import settings
from passport.backend.core.models.email import Email
from passport.backend.core.types.answer import normalize_answer
from passport.backend.core.types.email.email import (
    mask_email_for_challenge,
    unicode_email,
)
from passport.backend.core.types.phone_number.phone_number import (
    InvalidPhoneNumber,
    PhoneNumber,
)
from passport.backend.core.utils.decorators import cached_property
from passport.backend.core.utils.experiments import is_experiment_enabled_by_uid

from .forms import (
    CHALLENGE_3DS,
    CHALLENGE_DICTATION_PHONE_CONFIRMATION,
    CHALLENGE_EMAIL,
    CHALLENGE_EMAIL_CODE,
    CHALLENGE_PHONE,
    CHALLENGE_PHONE_CONFIRMATION,
    CHALLENGE_PUSH_2FA,
    CHALLENGE_QUESTION,
    CHALLENGE_WEBAUTHN,
)


def get_emails_for_challenge(account):
    return [
        email
        for email in account.emails.confirmed_external
        if not (account.lite_alias and email.address == account.lite_alias.alias)
    ]


class BaseChallenge(object):
    name = None
    conflicts_with_challenge_classes = []

    def __init__(self, parent, form_values, ignore_antifraud_tags=False):
        self.parent = parent
        self.form_values = form_values
        self.ignore_antifraud_tags = ignore_antifraud_tags

    @property
    def account(self):
        return self.parent.account

    @property
    def track(self):
        return self.parent.track

    @property
    def is_enabled(self):
        """Можно ли применять челленж для текущего аккаунта"""
        for conflicting_challenge_class in self.conflicts_with_challenge_classes:
            conflicting_challenge = conflicting_challenge_class(parent=self.parent, form_values=self.form_values)
            if (
                 hasattr(self.parent, 'challenges_available') and
                 (conflicting_challenge.name in [challenge_dict for challenge_dict in self.parent.challenges_available if bool(challenge_dict)][0])
                 and conflicting_challenge.is_enabled and not conflicting_challenge.is_passed
            ):
                return False

        return True

    @property
    def is_passed(self):
        """Признак, что челленж уже пройден"""
        return False

    def make_hint(self):
        """Подсказка, которую дадим пользователю"""
        raise NotImplementedError()  # pragma: no cover

    @property
    def extra_data(self):
        """Дополнительные данные, которые могут потребоваться фронту для показа челленжа"""
        return {}

    @property
    def extra_log_data(self):
        return {}

    def check(self, answer):
        """Проверить ответ пользователя"""
        raise NotImplementedError()  # pragma: no cover

    def change_track(self):
        """Если необходимо что-то поменять в треке при назначении челленджа"""
        pass


class BaseChallengePhoneConfirmation(BaseChallenge):
    name = CHALLENGE_PHONE_CONFIRMATION
    use_bank_number = True
    use_secure_number = True

    @property
    def is_enabled(self):
        return super(BaseChallengePhoneConfirmation, self).is_enabled and self.challenged_phone is not None

    @property
    def is_passed(self):
        # Если секьюрный номер уже был подтверждён в ходе авторизации (или
        # восстановления логина), то челлендж подтверждения номера нет смысла
        # спрашивать.
        # Для банковского номера - аналогично
        # в данном случае флаг успешности проверки лежит в треке и методы check и is_passed совпадают
        return self.check()

    @property
    def challenged_phone(self):
        phone = None
        if self.account.phones.secure and self.use_secure_number:
            phone = self.account.phones.secure
        elif self.account.bank_phonenumber_alias and self.use_bank_number:
            phone = self.account.phones.by_number(self.account.bank_phonenumber_alias.alias)
        return phone

    def make_hint(self):
        return self.challenged_phone.number.masked_format_for_challenge

    @property
    def extra_data(self):
        return {
            'phone_number': self.challenged_phone.number.e164,
            'phone_id': self.challenged_phone.id,
        }

    def check(self, answer=None):
        if self.account.phones.secure and self.use_secure_number:
            return self.parent.is_secure_phone_confirmed_in_track(allow_by_flash_call=True)
        elif self.account.bank_phonenumber_alias and self.use_bank_number:
            return self.parent.is_bank_phone_confirmed_in_track(allow_by_flash_call=True)


class ChallengePhoneConfirmationOnAuth(BaseChallengePhoneConfirmation):
    def __init__(self, parent, form_values, ignore_antifraud_tags=False):
        super(ChallengePhoneConfirmationOnAuth, self).__init__(parent, form_values, ignore_antifraud_tags)
        self._extra_log_data = {}
        self._can_be_handled_by_client = form_values['can_send_sms']
        self._can_only_be_ordered_by_antifraud = False
        self.antifraud_tags_resolution = self.ignore_antifraud_tags or (self.track.antifraud_tags and bool(set(self.track.antifraud_tags) & {'sms', 'call', 'flash_call', 'bank_sms'}))

    @property
    def is_enabled(self):
        if not super(ChallengePhoneConfirmationOnAuth, self).is_enabled:
            return False
        # описание эвристик, если прошла базовая проверка:
        #       bool(self.track.x_token_client_id or self.track.device_id)?
        #        /                          \
        #       / True                       -- False ---------------------------------\
        #      /                                                                       |
        #  uid подпадает под PHONE_CONFIRMATION_CHALLENGE_ENABLED_FOR_AM_DENOMINATOR?  |
        #    /                          \                                              |
        #   / Да                         -- Нет -------------------------------------->|
        #  /                                                                           |
        # PHONE_CONFIRMATION_CHALLENGE_FOR_ALL_APPS?                                   |
        #      /                        \                                              |
        #     / Да                       | Нет                                         |
        #    /                           |                                             |
        #   |                            |                                             |
        #   |                            |                                             |
        #   |             track.device_application входит в                            |
        #   |       множество PHONE_CONFIRMATION_CHALLENGE_APP_IDS ?                   |
        #   |            /                   \                                         |
        #   |          / Да                   -- Нет --------------------------------->|
        #   |        /                                                                 |
        # номер актуализирован менее 3 месяцев назад?                                  |
        #        /                   \                                                 |
        #      / Да                   -- Нет ----------------------------------------->|
        #    /                                                                         |
        # Челлендж по смс включён!                                                     |
        #                                                                              V
        #                                               Челлендж только для is_easily_hacked или sms_2fa_on или зашедших по ЛЦА
        #                                                        или есть банковским номер или если АнтиФрод попросил
        can_show_for_am = False

        is_mobile = bool(self.track.x_token_client_id or self.track.device_id)  # эвристика для определения account manager
        uid_in_experiment = is_experiment_enabled_by_uid(
            self.account.uid,
            settings.PHONE_CONFIRMATION_CHALLENGE_ENABLED_FOR_AM_DENOMINATOR,
        )

        enabled_for_all_apps = settings.PHONE_CONFIRMATION_CHALLENGE_FOR_ALL_APPS
        allowed_app_ids = settings.PHONE_CONFIRMATION_CHALLENGE_APP_IDS
        enabled_for_app_id = enabled_for_all_apps
        if not enabled_for_app_id:
            current_app_id = str(self.track.device_application or '').lower()
            enabled_for_app_id = current_app_id in allowed_app_ids

        days = settings.PHONE_CONFIRMATION_CHALLENGE_ACTUALITY_THRESHOLD_DAYS
        is_actual = (
            self.challenged_phone and
            self.challenged_phone.confirmed >= datetime.now() - timedelta(days=days)
        )

        # Финальное решение по челленджу для обычных аккаунтов
        if uid_in_experiment and is_mobile and enabled_for_app_id and is_actual:
            can_show_for_am = True

        # TODO я думаю можно логирование выпилить в будущем, но может пригодиться сейчас на время плавной раскатки
        self._extra_log_data.update(
            uid_in_experiment=uid_in_experiment,
            phone_confirmation_enabled_for_app_id=enabled_for_app_id,
            client_can_send_sms=self._can_be_handled_by_client,
        )

        used_phonenumber_alias_for_login = self._login_is_secure_phone()
        return (
            self._can_be_handled_by_client and
            (self.antifraud_tags_resolution or (
                not self._can_only_be_ordered_by_antifraud and (
                    self.account.is_easily_hacked or self.account.sms_2fa_on or can_show_for_am
                    or used_phonenumber_alias_for_login or (self.use_bank_number and bool(self.account.bank_phonenumber_alias))
                )
            ))
        )

    def _login_is_secure_phone(self):
        if not self.account.phones.secure:
            return False
        if not self.track.user_entered_login:
            return False
        login = self.track.user_entered_login.split('@', 1)[0]
        try:
            number = PhoneNumber.parse(
                login,
                country=self.account.person.country,
                allow_impossible=True,
            )
        except InvalidPhoneNumber:
            return False
        return number == self.account.phones.secure.number

    @property
    def extra_log_data(self):
        return self._extra_log_data


class ChallengeDictationPhoneConfirmation(ChallengePhoneConfirmationOnAuth, OctopusAvailabilityMixin):
    """
    Челлендж диктовкой по номеру телефона
    """
    name = CHALLENGE_DICTATION_PHONE_CONFIRMATION

    def __init__(self, parent, form_values, ignore_antifraud_tags=False):
        super(ChallengeDictationPhoneConfirmation, self).__init__(parent, form_values, ignore_antifraud_tags)
        self.antifraud_tags_resolution = self.ignore_antifraud_tags or (self.track.antifraud_tags and ('dictation' in self.track.antifraud_tags))
        self._can_only_be_ordered_by_antifraud = True
        self.client_ip = parent.client_ip  # для работы метода check_valid_for_call
        self.statbox = parent.statbox  # для работы метода check_valid_for_call

    @property
    def is_enabled(self):
        return (
            super(ChallengeDictationPhoneConfirmation, self).is_enabled and
            self.check_valid_for_call(self.parent.consumer, self.challenged_phone.number).valid_for_call
        )

    def check(self, answer=None):
        if self.account.phones.secure and self.use_secure_number:
            return self.parent.is_secure_phone_confirmed_in_track(allow_by_sms=False)
        elif self.account.bank_phonenumber_alias and self.use_bank_number:
            return self.parent.is_bank_phone_confirmed_in_track(allow_by_sms=False)

    def change_track(self):
        if self.track:
            with self.parent.track_transaction.rollback_on_error():
                self.track.phone_valid_for_call = True
                self.track.phone_valid_for_flash_call = False
                self.track.phone_validated_for_call = self.challenged_phone.number.e164


class ChallengePhoneConfirmationStandalone(BaseChallengePhoneConfirmation):
    pass


class ChallengePhoneOnAuth(BaseChallenge):
    name = CHALLENGE_PHONE
    conflicts_with_challenge_classes = [ChallengePhoneConfirmationOnAuth]

    @property
    def is_enabled(self):
        return (
            super(ChallengePhoneOnAuth, self).is_enabled and
            bool(self.account.phones.secure)
        )

    @property
    def is_passed(self):
        # Если пользователь уже подтвердил телефон любым способом, то
        # понятно, что он знает номер и нет смысла в данном челлендже.
        return bool(self.parent.is_secure_phone_confirmed_in_track(allow_by_flash_call=True))

    def make_hint(self):
        return self.account.phones.secure.number.masked_format_for_challenge

    def check(self, answer):
        try:
            number = PhoneNumber.parse(answer, country=self.account.person.country, allow_impossible=True)
        except InvalidPhoneNumber:
            return False
        return bool(self.account.phones.secure and number == self.account.phones.secure.number)


class ChallengeEmailOnAuth(BaseChallenge):
    name = CHALLENGE_EMAIL
    conflicts_with_challenge_classes = [ChallengePhoneConfirmationOnAuth]

    @property
    def _emails_for_challenge(self):
        return get_emails_for_challenge(self.account)

    @property
    def is_enabled(self):
        return super(ChallengeEmailOnAuth, self).is_enabled and bool(self._emails_for_challenge)

    def make_hint(self):
        emails = sorted(
            self._emails_for_challenge,
            key=lambda email: email.confirmed_at,
        )
        return mask_email_for_challenge(unicode_email(emails[-1].address.lower()))

    def check(self, answer):
        answer = answer.lower()
        try:
            normalized_answer = Email.normalize_address(answer)
        except ValueError:
            normalized_answer = answer
        return (
            any(
                normalized_answer == email.normalized_address
                for email in self._emails_for_challenge
            ) and
            mask_email_for_challenge(unicode_email(answer)) == self.make_hint()
        )


class ChallengeWebauthn(BaseChallenge):
    name = CHALLENGE_WEBAUTHN

    @property
    def _suggested_webauthn_credentials(self):
        return get_suggested_webauthn_credentials(account=self.account, env=self.parent.request.env)

    @property
    def is_enabled(self):
        return super(ChallengeWebauthn, self).is_enabled and bool(self._suggested_webauthn_credentials)

    @property
    def extra_data(self):
        return {
            'credentials': [
                {
                    'device_name': cred.device_name,
                    'os_family_name': settings.OS_FAMILY_DECODE.get(cred.os_family_id),
                    'browser_name': settings.BROWSER_DECODE.get(cred.browser_id),
                    'credential_external_id': cred.external_id,
                }
                for cred in self._suggested_webauthn_credentials
            ],
        }

    def make_hint(self):
        # кратко подсказать нечего, а полный список есть в extra_data
        return

    def check(self, answer):
        return (
            self.track.webauthn_confirmed_secret_external_id and
            self.track.webauthn_confirmed_secret_external_id in self.account.webauthn_credentials
        )


class ChallengeQuestionOnAuth(BaseChallenge):
    """
    Контрольная пара вопрос-ответ.
    """
    name = CHALLENGE_QUESTION
    antifraud_tag = 'question'

    @property
    def is_enabled(self):
        return (
            super(ChallengeQuestionOnAuth, self).is_enabled and
            self.account.hint.is_set and
            (self.ignore_antifraud_tags or
             (self.track.antifraud_tags and self.antifraud_tag in self.track.antifraud_tags))
        )

    def make_hint(self):
        return self.account.hint.question.text

    def check(self, answer):
        answer = answer or ''  # если ничего не ввести, то ответ None
        reference_answer = normalize_answer(self.account.hint.answer[:settings.HINT_ANSWER_MAX_LENGTH])
        user_entered_answer = normalize_answer(answer[:settings.HINT_ANSWER_MAX_LENGTH])
        return user_entered_answer == reference_answer


class ChallengePush2faOnAuth(BaseChallenge):
    """ Челлендж с кодом в пуш-сообщении """
    name = CHALLENGE_PUSH_2FA
    antifraud_tag = 'push_2fa'

    @cached_property
    def manager(self):
        return get_pushes_subscription_manager(self.account.uid)

    def _check_has_compatible_devices(self):
        try:
            return self.manager.has_trusted_subscriptions()
        except BasePushApiError:
            # Ошибка пуш-апи - не повод ломаться. Покажем другие челленжи.
            return False

    def _check_experiment(self):
        return is_experiment_enabled_by_uid(
            self.account.uid,
            settings.PUSH_2FA_CHALLENGE_ENABLED_DENOMINATOR,
        )

    def _check_antifraud(self):
        return (
            self.ignore_antifraud_tags or
            (self.track.antifraud_tags and self.antifraud_tag in self.track.antifraud_tags)
        )

    @cached_property
    def is_enabled(self):
        return (
            super(ChallengePush2faOnAuth, self).is_enabled and
            settings.PUSH_2FA_CHALLENGE_ENABLED and
            self._check_experiment() and
            (
                self._check_antifraud() or self.account.sms_2fa_on
            ) and
            self._check_has_compatible_devices()
        )

    def check(self, answer):
        return self.track.push_otp and self.track.push_otp == answer

    def make_hint(self):
        return


class Challenge3DS(BaseChallenge):
    name = CHALLENGE_3DS

    def make_hint(self):
        return

    @property
    def extra_data(self):
        return {}

    @cached_property
    def is_enabled(self):
        return (
            super(Challenge3DS, self).is_enabled and
            self.track.paymethod_id
        )

    def check(self, answer):
        return (
            self.track.paymethod_id and
            self.track.payment_status == PAYMENT_STATUS_AUTHORIZED
        )


class ChallengeEmailCodeOnAuth(BaseChallenge):
    """ Челлендж с кодом в email """
    name = CHALLENGE_EMAIL_CODE
    antifraud_tag = 'email_code'

    def _check_experiment(self):
        return is_experiment_enabled_by_uid(
            self.account.uid,
            settings.EMAIL_CODE_CHALLENGE_ENABLED_DENOMINATOR,
        )

    def _check_antifraud(self):
        return (
            self.ignore_antifraud_tags or
            (self.track.antifraud_tags and self.antifraud_tag in self.track.antifraud_tags)
        )

    def _check_account(self):
        return self.account.emails.confirmed_external

    @cached_property
    def is_enabled(self):
        return bool(
            super(ChallengeEmailCodeOnAuth, self).is_enabled and
            settings.EMAIL_CODE_CHALLENGE_ENABLED and
            self._check_experiment() and
            self._check_antifraud() and
            self._check_account()
        )

    def check(self, answer):
        return bool(self.track.email_check_ownership_passed)

    def make_hint(self):
        return [
            mask_email_for_challenge(email.address) for email in
            self.account.emails.confirmed_external
        ]
