# -*- coding: utf-8 -*-
from collections import OrderedDict
import logging

from frozendict import frozendict
from passport.backend.api.common.authorization import log_auth_challenge_shown
from passport.backend.api.common.profile.profile import process_env_profile
from passport.backend.api.views.bundle.auth.base import BundleBaseAuthorizationMixin
from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.exceptions import (
    ActionImpossibleError,
    InvalidTrackStateError,
)
from passport.backend.api.views.bundle.headers import (
    HEADER_CLIENT_COOKIE,
    HEADER_CLIENT_HOST,
    HEADER_CLIENT_USER_AGENT,
    HEADER_CONSUMER_CLIENT_IP,
)
from passport.backend.api.views.bundle.mixins import (
    BindRelatedPhonishAccountMixin,
    BundleAccountGetterMixin,
    BundleAccountResponseRendererMixin,
    BundleAssertCaptchaMixin,
    BundleAuthNotificationsMixin,
    BundleDeviceInfoMixin,
    BundlePhoneMixin,
)
from passport.backend.core.authtypes import AUTH_TYPE_OAUTH_CREATE
from passport.backend.core.builders.antifraud import get_antifraud_api
from passport.backend.core.builders.trust_api import get_trust_payments
from passport.backend.core.conf import settings
from passport.backend.core.counters import auth_challenge_per_ip
from passport.backend.core.geobase import Region
from passport.backend.core.logging_utils.loggers import (
    AntifraudLogger,
    StatboxLogger,
)
from passport.backend.core.runner.context_managers import UPDATE
from passport.backend.core.utils.decorators import cached_property
from passport.backend.utils.time import get_unixtime

from .challenges import (
    Challenge3DS,
    ChallengeDictationPhoneConfirmation,
    ChallengeEmailCodeOnAuth,
    ChallengeEmailOnAuth,
    ChallengePhoneConfirmationOnAuth,
    ChallengePhoneConfirmationStandalone,
    ChallengePhoneOnAuth,
    ChallengePush2faOnAuth,
    ChallengeQuestionOnAuth,
    ChallengeWebauthn,
)
from .exceptions import (
    CardIdNotFoundError,
    ChallengeLimitExceededError,
    ChallengeNotEnabledError,
    ChallengeNotPassedError,
)
from .forms import (
    BaseChallengeForm,
    ChallengeCommitForm,
    ChallengeStandaloneCreateTrackForm,
    SendAuthEmailForm,
)


log = logging.getLogger('passport.backend.api.bundle.challenge')


AF_TAG_TO_CHALLENGE_MAP = {
    'sms': ChallengePhoneConfirmationOnAuth,
    'flash_call': ChallengePhoneConfirmationOnAuth,
    'call': ChallengePhoneConfirmationOnAuth,
    'phone_hint': ChallengePhoneOnAuth,
    'email_hint': ChallengeEmailOnAuth,
    'question': ChallengeQuestionOnAuth,
    'bank_sms': ChallengePhoneConfirmationOnAuth,
    'webauthn': ChallengeWebauthn,
    'push_2fa': ChallengePush2faOnAuth,
    '3ds': Challenge3DS,
    'email_code': ChallengeEmailCodeOnAuth,
    'dictation': ChallengeDictationPhoneConfirmation,
    # "captcha" :  # такого челленджа нет
}


def challenges_available(self, challenge_classes, form_values=frozendict(can_send_sms=True), ignore_antifraud_tags=False):
    # Показывает какие челленджи доступны для аккаунта из переданных в challenge_classes в порядке приоритета. Учитывает ответ антифрода (если есть в треке)
    challenges_available_our_opinion = OrderedDict([
        (challenge_cls.name, challenge_cls(self, form_values, ignore_antifraud_tags=ignore_antifraud_tags))
        for challenge_cls in challenge_classes
    ])

    # челленджи что нам разрешил показывать антифрод, в порядке заданном антифродом
    challenges_available_af = OrderedDict()
    if self.track.antifraud_tags:
        antifraud_allowed_challenge_names = [
            AF_TAG_TO_CHALLENGE_MAP[tag].name for tag in self.track.antifraud_tags
            if tag in AF_TAG_TO_CHALLENGE_MAP
        ]
        for challenge_name in antifraud_allowed_challenge_names:
            if (
                challenge_name in challenges_available_our_opinion
                and challenge_name not in challenges_available_af
            ):
                challenges_available_af[challenge_name] = challenges_available_our_opinion[challenge_name]

    return challenges_available_af, challenges_available_our_opinion


def reset_challenge_counters(account):
    account.auth_email_datetime = None
    account.failed_auth_challenge_checks_counter.reset()


class BaseChallengeView(BaseBundleView,
                        BundleAccountGetterMixin,
                        BundleAccountResponseRendererMixin,
                        BundleAssertCaptchaMixin,
                        BundleDeviceInfoMixin,
                        BundlePhoneMixin,
                        BundleBaseAuthorizationMixin):
    require_track = True
    basic_form = BaseChallengeForm
    challenge_classes = ()  # упорядочен по убыванию приоритета
    mode = None

    @cached_property
    def statbox(self):
        return StatboxLogger(
            consumer=self.consumer,
            track_id=self.track_id,
            ip=self.client_ip,
            user_agent=self.user_agent,
            mode=self.mode,
            yandexuid=self.cookies.get('yandexuid'),
        )

    def incr_counters(self):
        counter = auth_challenge_per_ip.get_counter()
        counter.incr(self.client_ip)

        with UPDATE(
            self.account,
            self.request.env,
            {'action': '%s_attempt' % self.mode},
        ):
            self.account.failed_auth_challenge_checks_counter.incr(
                expire_in=settings.AUTH_CHALLENGE_PERIOD,
            )

    def check_counters(self):
        if self.account.failed_auth_challenge_checks_counter.value >= settings.AUTH_CHALLENGE_MAX_ATTEMPTS:
            raise ChallengeLimitExceededError()

        counter = auth_challenge_per_ip.get_counter()
        if counter.hit_limit_by_ip(self.client_ip):
            raise ChallengeLimitExceededError()

    @cached_property
    def challenges_available(self):
        # вернет tuple (челенджи что доступны по мнению паспорта И антифрода, челенджи что доступны по мнению паспорта)
        return challenges_available(self, self.challenge_classes, self.form_values)

    @cached_property
    def challenges_enabled(self):
        # если список челленджей основаный на тэгах антифрода пуст, используем свой
        available_challenges_af, available_challenges_our_opinion = self.challenges_available
        _challenges_enabled_af = [(challenge_af, challenge_af.is_enabled) for challenge_af in available_challenges_af.values()]
        _available_challenges_our_opinion = [(challenge, challenge.is_enabled) for challenge in available_challenges_our_opinion.values()]
        # TODO: убрать фолбек на _available_challenges_our_opinion через 3 часа после выкатки PASSP-37822 (он станет недостижим)
        _challenges_enabled = _challenges_enabled_af if _challenges_enabled_af and any(zip(*_challenges_enabled_af)[1]) else _available_challenges_our_opinion

        challenges = OrderedDict()
        for challenge, is_enabled in _challenges_enabled:
            if is_enabled:
                challenges[challenge.name] = dict(
                    hint=challenge.make_hint(),
                    **challenge.extra_data
                )
            self.statbox.bind(**challenge.extra_log_data)
        return challenges

    @cached_property
    def challenges_enabled_and_not_passed(self):
        # если список челленджей основаный на тэгах антифрода пуст, используем свой
        available_challenges_af, available_challenges_our_opinion = self.challenges_available
        _challenges_enabled_and_not_passed_af = [(challenge_af, challenge_af.is_enabled and not challenge_af.is_passed) for challenge_af in available_challenges_af.values()]
        _challenges_enabled_and_not_passed_our_opinion = [(challenge, challenge.is_enabled and not challenge.is_passed) for challenge in available_challenges_our_opinion.values()]
        _challenges_enabled = _challenges_enabled_and_not_passed_af if _challenges_enabled_and_not_passed_af and any(zip(*_challenges_enabled_and_not_passed_af)[1]) else _challenges_enabled_and_not_passed_our_opinion

        challenges = OrderedDict()
        for challenge, is_enabled_and_not_passed in _challenges_enabled:
            if is_enabled_and_not_passed:
                challenges[challenge.name] = dict(
                    hint=challenge.make_hint(),
                    **challenge.extra_data
                )
            self.statbox.bind(**challenge.extra_log_data)
        return challenges

    def change_track(self):
        _, available_challenges_our_opinion = self.challenges_available
        for challenge_name in self.challenges_enabled_and_not_passed.keys():
            available_challenges_our_opinion[challenge_name].change_track()

    def show_challenges(self):
        if not self.challenges_enabled_and_not_passed:
            raise ActionImpossibleError()

        self.response_values.update(
            challenges_enabled=self.challenges_enabled_and_not_passed,
            default_challenge=list(self.challenges_enabled_and_not_passed.keys())[0],
        )
        self.statbox.log(
            action='shown',
            default_challenge=list(self.challenges_enabled_and_not_passed.keys())[0],
            challenges=','.join(self.challenges_enabled_and_not_passed.keys()),
        )

    def check_challenge_answer(self):
        if not self.challenges_enabled:
            raise ActionImpossibleError()

        challenge_name = self.form_values['challenge']
        if challenge_name not in self.challenges_enabled:
            raise ChallengeNotEnabledError()

        _, available_challenges_our_opinion = self.challenges_available
        challenge = available_challenges_our_opinion[challenge_name]
        if not challenge.check(self.form_values['answer']):
            self.response_values.update()
            self.statbox.log(
                action='failed',
                challenge=challenge_name,
                method=challenge_name,
            )
            self.incr_counters()
            raise ChallengeNotPassedError()

        # всё хорошо - челленж пройден
        self.statbox.log(
            action='passed',
            challenge=challenge_name,
            method=challenge_name,
        )
        return challenge_name


class BaseChallengeOnAuthView(BaseChallengeView):
    required_grants = ['auth_password.base']
    challenge_classes = (
        ChallengePush2faOnAuth,
        ChallengePhoneConfirmationOnAuth,
        ChallengePhoneOnAuth,
        ChallengeEmailOnAuth,
        ChallengeEmailCodeOnAuth,
        ChallengeQuestionOnAuth,
    )
    mode = 'auth_challenge'

    def get_and_validate_prerequisites(self):
        self.read_track()
        self.process_basic_form()

        device_info = self.track_to_oauth_params(self.get_device_params_from_track())
        self.statbox.bind(**device_info)
        self.statbox.bind(
            is_mobile=bool(self.track.x_token_client_id or self.track.device_id),
        )

        self.response_values['track_id'] = self.track_id
        self.check_auth_not_passed()
        self.check_track_for_captcha()
        if not self.track.is_auth_challenge_shown:
            raise InvalidTrackStateError()

        self.get_account_from_track(
            check_disabled_on_deletion=True,
            need_phones=True,
            emails=True,
            get_webauthn_credentials=True,
        )
        self.statbox.bind_context(uid=self.account.uid)

        self.check_counters()


class ChallengeOnAuthSubmitView(BaseChallengeOnAuthView):
    def process_request(self):
        self.get_and_validate_prerequisites()
        self.show_challenges()


class ChallengeOnAuthCommitView(BindRelatedPhonishAccountMixin, BaseChallengeOnAuthView):
    basic_form = ChallengeCommitForm
    challenge_names_for_trusted_token = {
        ChallengePhoneConfirmationOnAuth.name,
        ChallengePush2faOnAuth.name,
    }

    def process_request(self):
        self.get_and_validate_prerequisites()
        challenge_name = self.check_challenge_answer()

        process_env_profile(self.account, track=self.track)

        with self.track_transaction.commit_on_error():
            self.track.allow_authorization = True
            self.track.auth_challenge_type = challenge_name
            if challenge_name in self.challenge_names_for_trusted_token:
                log.debug('Allow trusted xtoken by challenge type {}'.format(challenge_name))
                self.track.allow_set_xtoken_trusted = True
            else:
                log.debug('Disallow trusted xtoken by challenge type {}'.format(challenge_name))

        self.try_bind_related_phonish_account()


class BaseChallengeStandaloneView(BaseChallengeView):
    required_grants = ['challenge.base']
    challenge_classes = (
        ChallengeWebauthn,
        ChallengeDictationPhoneConfirmation,
        ChallengePhoneConfirmationStandalone,
        Challenge3DS,
    )
    required_headers = (
        HEADER_CLIENT_COOKIE,
        HEADER_CLIENT_HOST,
        HEADER_CLIENT_USER_AGENT,
        HEADER_CONSUMER_CLIENT_IP,
    )
    mode = 'challenge'

    def get_af_features_for_passed_challenge(self):
        region_info = Region(ip=self.client_ip)
        return {
            'uid': self.account.uid,
            'external_id': self.track.antifraud_external_id or 'track-{}'.format(self.track_id),
            'channel': 'challenge',
            'sub_channel': 'chaas',
            'retpath': self.track.retpath,
            'user_agent': self.user_agent,
            'ip': str(self.client_ip),
            'AS': region_info.AS_list[0] if region_info.AS_list else None,
            'authid': self.session_info.authid,
            'login_id': self.login_id,
            'yandexuid': self.cookies.get('yandexuid'),
            'challenge': self.track.auth_challenge_type,
            'status': 'OK',
        }

    def read_and_validate_track(self):
        self.read_track()
        if not self.track.uid:
            raise InvalidTrackStateError()
        self.response_values['track_id'] = self.track_id

    def get_account(self):
        self.get_account_from_session(
            check_disabled_on_deletion=True,
            need_phones=True,
            get_webauthn_credentials=True,
            multisession_uid=int(self.track.uid),
        )
        self.statbox.bind_context(uid=self.account.uid)


class ChallengeStandaloneCreateTrackView(BaseChallengeStandaloneView):
    required_grants = ['challenge.standalone_create_track']
    require_track = False
    basic_form = ChallengeStandaloneCreateTrackForm
    required_headers = ()

    def assert_card_id_valid(self, uid, card_id):
        available_payment_methods = get_trust_payments().get_payment_methods(uid=uid)
        available_payment_method_ids = set()
        for payment_method in available_payment_methods:
            if payment_method.get('card_id'):
                available_payment_method_ids.add(payment_method['card_id'])
            if payment_method.get('aliases'):
                available_payment_method_ids.update(payment_method['aliases'])

        if card_id not in available_payment_method_ids:
            if available_payment_method_ids:
                description = ', '.join(available_payment_method_ids)
            else:
                description = 'no bound payment_methods'
            log.info(
                'Requested card_id_for_3ds=%s, but uid %s has %s', card_id, uid, description)
            raise CardIdNotFoundError()

    def process_request(self):
        self.process_basic_form()
        if self.form_values['card_id_for_3ds'] and settings.VALIDATE_CARD_ID_BEFORE_CREATING_CHAAS_TRACK:
            self.assert_card_id_valid(
                uid=self.form_values['uid'],
                card_id=self.form_values['card_id_for_3ds'],
            )

        self.create_track('universal')
        self.response_values['track_id'] = self.track_id

        with self.track_transaction.commit_on_error() as track:
            track.uid = self.form_values['uid']
            track.retpath = self.form_values['retpath']
            track.antifraud_tags = self.form_values['antifraud_tags']
            track.antifraud_external_id = self.form_values['antifraud_external_id']
            track.paymethod_id = self.form_values['card_id_for_3ds']


class ChallengeStandaloneSubmitView(BaseChallengeStandaloneView):
    def process_request(self):
        self.process_basic_form()
        self.read_and_validate_track()
        self.response_values['retpath'] = self.track.retpath
        self.get_account()

        self.check_counters()
        self.change_track()
        self.show_challenges()


class ChallengeStandaloneCommitView(BaseChallengeStandaloneView):
    basic_form = ChallengeCommitForm

    def log_passed_challenge_to_antifraud(self):
        logger = AntifraudLogger()
        features = self.get_af_features_for_passed_challenge()
        features.update(request=None)
        logger.log(**features)

    def process_request(self):
        self.process_basic_form()

        self.read_and_validate_track()
        self.response_values['retpath'] = self.track.retpath
        self.get_account()

        self.check_counters()
        challenge_name = self.check_challenge_answer()
        with self.track_transaction.commit_on_error():
            self.track.auth_challenge_type = challenge_name

        self.log_passed_challenge_to_antifraud()


class ChallengeStandaloneSaveView(BaseChallengeStandaloneView):
    def report_to_antifraud(self):
        af_api = get_antifraud_api()
        features = self.get_af_features_for_passed_challenge()
        features.update({'t': get_unixtime() * 1000})
        af_api.save(features)

    def process_request(self):
        self.read_track()
        if not (self.track.uid and self.track.auth_challenge_type):
            raise InvalidTrackStateError()

        self.response_values.update(
            retpath=self.track.retpath,
        )
        self.get_account_from_session(
            check_disabled_on_deletion=True,
            need_phones=True,
            multisession_uid=int(self.track.uid),
        )
        if settings.ANTIFRAUD_ON_CHALLENGE_ENABLED:
            self.report_to_antifraud()


class SendAuthEmailView(BaseBundleView,
                        BundleAccountGetterMixin,
                        BundleAuthNotificationsMixin):
    required_grants = ['challenge.send_email']
    required_headers = (
        HEADER_CONSUMER_CLIENT_IP,
        HEADER_CLIENT_USER_AGENT,
    )
    basic_form = SendAuthEmailForm

    def process_request(self):
        self.process_basic_form()
        uid = self.form_values['uid']
        self.get_account_by_uid(
            uid=uid,
            emails=True,
        )
        # ToDo: разделить отправку пушей и почты, логировать по отдельности
        notifications_sent = self.try_send_auth_notifications(
            device_name=self.form_values['device_name'],
            is_challenged=self.form_values['is_challenged'],
        )
        log_auth_challenge_shown(
            self.request.env,
            uid,
            auth_type=AUTH_TYPE_OAUTH_CREATE,
        )
        self.response_values.update(email_sent=notifications_sent)
