# -*- coding: utf-8 -*-
from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.exceptions import (
    AccountUidMismatchError,
    InvalidTrackStateError,
    UidNotInSessionError,
)
from passport.backend.api.views.bundle.headers import (
    HEADER_CLIENT_HOST,
    HEADER_CONSUMER_CLIENT_IP,
)
from passport.backend.api.views.bundle.mixins import (
    BundleAccountGetterMixin,
    BundleAvatarMixin,
)
from passport.backend.core.logging_utils.loggers import StatboxLogger
from passport.backend.core.utils.decorators import cached_property


BASIC_GRANT = 'change_avatar.base'


class BaseAvatarBundleView(BaseBundleView, BundleAccountGetterMixin, BundleAvatarMixin):
    required_headers = (
        HEADER_CLIENT_HOST,
        HEADER_CONSUMER_CLIENT_IP,
    )
    require_track = False
    required_grants = [BASIC_GRANT]
    track_type = 'authorize'

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode='avatar_change',
            ip=self.client_ip,
            useragent=self.user_agent,
            uid=self.account.uid,
            track_id=self.track_id,
        )

    def get_and_check_account(self):
        raise NotImplementedError()     # pragma: no cover

    def process_request(self):
        if self.basic_form is not None:
            self.process_basic_form()

        self.get_and_check_account()
        self.process()

    def process(self):
        raise NotImplementedError()  # pragma: no cover

    def upload(self, is_temp=False):
        return super(BaseAvatarBundleView, self).upload(
            url=self.form_values['url'],
            file_obj=self.form_values['file'],
            is_temp=is_temp,
        )


class BaseAvatarTrackRequiredBundleView(BaseAvatarBundleView):
    require_track = True

    def get_and_check_account(self):
        """Получим аккаунт из session_id или по oauth_token и сравним с uid из трека."""
        uid = self.track_uid
        try:
            self.get_account_from_session_or_oauth_token(
                multisession_uid=uid,
                check_disabled_on_deletion=True,
            )
        except (UidNotInSessionError, AccountUidMismatchError) as e:
            raise InvalidTrackStateError(
                'track.uid (%s) is not in the current user\'s uids by credentials (%s)' % (
                    uid,
                    e.known_uids,
                ),
            )

    def check_track(self):
        if not self.track.is_avatar_change:
            raise InvalidTrackStateError()

    def process_request(self):
        self.check_header(HEADER_CLIENT_HOST)
        self.read_track()
        self.response_values['track_id'] = self.track_id
        self.check_track()
        super(BaseAvatarTrackRequiredBundleView, self).process_request()
