# -*- coding: utf-8 -*-

from passport.backend.api.views.bundle.mixins import BundleAccountResponseRendererMixin
from passport.backend.core.conf import settings

from ..constants import (
    AUTHENTICATION_MEDIA_SESSION,
    AUTHENTICATION_MEDIA_TOKEN,
)
from ..exceptions import (
    RequestCredentialsAllMissingError,
    RequestCredentialsSeveralPresentError,
)
from .base import (
    BaseAvatarBundleView,
    BaseAvatarTrackRequiredBundleView,
)
from .forms import (
    InitForm,
    UploadCommonForm,
    UploadForm,
)


AVATAR_BY_UID_GRANT = 'change_avatar.by_uid'
AVATAR_PDD_BY_UID_GRANT = 'change_avatar.pdd_by_uid'


class AvatarGetListStartTrack(BaseAvatarBundleView, BundleAccountResponseRendererMixin):
    basic_form = InitForm

    def initialize_track(self):
        self.create_track(self.track_type)

        with self.track_transaction.rollback_on_error():
            self.track.uid = self.account.uid
            self.track.is_avatar_change = True

        self.response_values['track_id'] = self.track_id

    def get_and_check_account(self):
        """
        У нас есть либо сессионная кука, либо авторизационный заголовок с токеном.
        """
        self.get_account_from_session_or_oauth_token(
            multisession_uid=self.form_values.get('uid'),
            check_disabled_on_deletion=True,
        )

    def process(self):
        self.initialize_track()
        self.update_response_with_avatars()
        self.fill_response_with_account()
        self.statbox.log(action='get_init_page')


class AvatarDeleteDefault(BaseAvatarTrackRequiredBundleView):
    def process(self):
        if not self.account.person or not self.account.person.default_avatar or self.account.person.is_avatar_empty:
            return

        self.check_delete_counter()
        self.delete()

        self.statbox.log(action='delete_default')


class AvatarGetList(BaseAvatarTrackRequiredBundleView):
    def process(self):
        self.update_response_with_avatars()


class AvatarUpload(BaseAvatarTrackRequiredBundleView):
    basic_form = UploadForm

    def process(self):
        avatar_id = self.upload(is_temp=self.form_values['is_temp'])
        avatar_url = settings.GET_AVATAR_URL_SCHEMALESS % avatar_id
        self.response_values.update(avatar_url=avatar_url)


class AvatarUploadV2(BaseAvatarBundleView):
    """
    Общая ручка загрузки аватарки через Паспорт.
    Принимает UID, cookie или oauth_token, получает аккаунт,
    проверяет гранты на редактирование аватара по UID
    """
    basic_form = UploadCommonForm
    grants_for_account_type = {
        'any': AVATAR_BY_UID_GRANT,
        'pdd': AVATAR_PDD_BY_UID_GRANT,
    }

    def process_request(self):
        self.process_basic_form()

        # Почему ниже так хитро сделано:
        # get_account_from_uid_or_session_or_oauth_token может работать только с
        # uid, или только с сессией или только с токеном.
        # Если передана сессия и uid например, то будет ошибка.

        # Надо уметь работать с сессией + uid для мультикук, но при этом оставить возможность работать
        # только по uid для ПДД.

        try:
            auth_media = self.select_authentication_media_name(
                enabled_media=[
                    AUTHENTICATION_MEDIA_SESSION,
                    AUTHENTICATION_MEDIA_TOKEN,
                ],
                # с этими параметрами не возвращается AUTHENTICATION_MEDIA_UID, и это хорошо,
                # потому что в результате ожидаются только AUTHENTICATION_MEDIA_{SESSION,TOKEN}
                ignore_grants=False,
                by_uid_grant=None,
            )
        except RequestCredentialsAllMissingError:
            # нету сессии или токена и нету uid в запросе
            if not self.form_values['uid']:
                raise

            # нету сессии или токена, но есть uid в запросе
            self.get_account_by_uid(
                self.form_values['uid'],
                check_disabled_on_deletion=True,
            )
            self.check_grants_for_account_type()
        else:
            # с uid имеет смысл только сессия
            if self.form_values['uid'] is not None and auth_media != AUTHENTICATION_MEDIA_SESSION:
                raise RequestCredentialsSeveralPresentError()

            self.get_account_from_session_or_oauth_token(
                check_disabled_on_deletion=True,
                multisession_uid=self.form_values['uid'],
            )

        avatar_id = self.upload()
        self.response_values.update(
            avatar_url=settings.GET_AVATAR_URL % (
                avatar_id,
                self.form_values['avatar_size'],
            ),
        )
