# -*- coding: utf-8 -*-

import logging

from passport.backend.api.common.account import default_revokers
from passport.backend.api.common.authorization import (
    build_auth_cookies_and_session,
    get_session_policy_by_ttl,
    set_authorization_track_fields,
    user_session_scope,
    users_from_multisession,
)
from passport.backend.api.common.processes import PROCESS_VOLUNTARY_PASSWORD_CHANGE
from passport.backend.api.common.profile.profile import process_env_profile
from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.challenge.controllers import reset_challenge_counters
from passport.backend.api.views.bundle.exceptions import (
    Account2FAEnabledError,
    ActionNotRequiredError,
    InvalidTrackStateError,
)
from passport.backend.api.views.bundle.headers import (
    HEADER_CLIENT_COOKIE,
    HEADER_CLIENT_HOST,
    HEADER_CLIENT_USER_AGENT,
    HEADER_CONSUMER_CLIENT_IP,
)
from passport.backend.api.views.bundle.mixins import (
    ActiveDirectoryMixin,
    BundleAccountGetterMixin,
    BundleAccountPropertiesMixin,
    BundleAccountResponseRendererMixin,
    BundleAssertCaptchaMixin,
    BundleFixPDDRetpathMixin,
    BundleFrodoMixin,
    BundlePasswordChangeMixin,
    BundlePasswordValidationMixin,
    BundlePasswordVerificationMethodMixin,
    BundlePhoneMixin,
    BundleVerifyPasswordMixin,
)
from passport.backend.api.views.bundle.mixins.kolmogor import KolmogorMixin
from passport.backend.api.views.bundle.mixins.password import CAPTCHA_VALIDATION_METHOD
from passport.backend.api.views.bundle.mixins.push import BundlePushMixin
from passport.backend.api.views.bundle.states import PasswordChangeForbidden
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.loggers.statbox import StatboxLogger
from passport.backend.core.runner.context_managers import UPDATE
from passport.backend.core.utils.decorators import cached_property

from .forms import (
    ChangePasswordCommitForm,
    ChangePasswordCommitIntranetForm,
    ChangePasswordOptionalLogoutForm,
    ChangePasswordSubmitForm,
)


ACCOUNT_CHANGE_PASSWORD_BASE_GRANT = 'account.change_password'

log = logging.getLogger('passport.api.view.bundle.account.change_password')


class ChangePasswordBaseView(BaseBundleView,
                             BundlePhoneMixin,
                             BundleAccountGetterMixin,
                             BundleAccountResponseRendererMixin,
                             BundleAccountPropertiesMixin):
    track_type = 'authorize'

    required_grants = [ACCOUNT_CHANGE_PASSWORD_BASE_GRANT]

    process_name = PROCESS_VOLUNTARY_PASSWORD_CHANGE

    @cached_property
    def statbox(self):
        return StatboxLogger(
            consumer=self.consumer,
            mode='change_password_voluntarily',
            track_id=self.track_id,
            yandexuid=self.cookies.get('yandexuid'),
            user_agent=self.user_agent,
            ip=self.client_ip,
            origin=self.track.origin if self.track else None,
            uid=self.track.uid if self.track else None,
        )

    def check_account_policies(self):
        """
        Проверим, что пользователь может сменить пароль
        """
        if self.account.is_pdd and not self.account.domain.can_users_change_password:
            return PasswordChangeForbidden()
        return self.check_have_password()

    def get_updated_cookies(self):
        multi_session_users = users_from_multisession(self.session_info)

        cookies, _, service_guard_container = build_auth_cookies_and_session(
            self.request.env,
            self.track,
            self.account.type,
            authorization_session_policy=get_session_policy_by_ttl(int(self.track.old_session_ttl)),
            is_yandexoid=self.account.is_yandexoid,
            is_betatester=self.account.is_betatester,
            extend_session=True,
            multi_session_users=multi_session_users,
            display_name=self.account.person.display_name,
            need_extra_sessguard=True,
            is_child=self.account.is_child,
        )

        if service_guard_container:
            self.response_values.update({
                'service_guard_container': service_guard_container.pack(),
            })

        self.response_values.update({
            'cookies': cookies,
            'default_uid': self.account.uid,
            'sensitive_fields': ['cookies'],
        })

    def verify_change_password_preconditions_met(self):
        if self.account.totp_secret.is_set:
            raise Account2FAEnabledError()

        if self.account.is_pdd:
            response = self.blackbox.hosted_domains(domain=self.account.domain.domain)
            self.account.parse(response)

        # Отдаём информацию про аккаунт, чтобы отрисовать залогиновую страничку
        self.fill_response_with_account()

        redirect_state = self.check_account_policies()
        return redirect_state


class ChangePasswordSubmitView(ChangePasswordBaseView,
                               BundleFixPDDRetpathMixin,
                               BundlePasswordVerificationMethodMixin):
    require_track = False
    required_headers = (
        HEADER_CLIENT_HOST,
        HEADER_CLIENT_COOKIE,
        HEADER_CONSUMER_CLIENT_IP,
    )

    basic_form = ChangePasswordSubmitForm

    @property
    def is_sms_validation_required(self):
        return False

    def process_request(self, *args, **kwargs):
        self.process_basic_form()
        # TODO оставить только создание трека с процессом; сейчас фронт приходит с готовым треком
        self.read_or_create_track(self.track_type, process_name=self.process_name)

        self.get_account_from_session(
            emails=True,
            check_disabled_on_deletion=True,
            need_phones=True,
        )
        self.response_values['track_id'] = self.track_id

        with self.track_transaction.commit_on_error() as track:
            self.set_uid_to_track(self.account.uid, track)
            track.login = self.account.login
            track.country = self.account.person.country
            track.is_strong_password_policy_required = self.account.is_strong_password_required
            track.origin = self.form_values['origin'] or track.origin

            if self.account.is_pdd:
                self.fix_pdd_retpath()
            if self.form_values['retpath']:
                track.retpath = self.form_values['retpath']
            self.response_values['retpath'] = track.retpath

            redirect_state = self.verify_change_password_preconditions_met()
            if redirect_state:
                self.state = redirect_state
                return

            self.save_secure_number_in_track()

            track.is_password_change = self.state is None
            # При добровольной смене пароля капча всегда обязательна.
            track.is_captcha_required = True
            validation_method = self.get_validation_method()
            self.response_values['validation_method'] = validation_method

            self.response_values['revokers'] = default_revokers(
                allow_select=not self.account.is_strong_password_required,
            )

            self.statbox.log(
                action='defined_validation_method',
                validation_method=validation_method,
                uid=self.account.uid,
            )


class ChangePasswordCommitView(
    BundlePushMixin,
    KolmogorMixin,
    ChangePasswordBaseView,
    BundlePasswordChangeMixin,
    BundleAssertCaptchaMixin,
    BundlePasswordValidationMixin,
    BundleVerifyPasswordMixin,
    ActiveDirectoryMixin,
    BundleFrodoMixin,
):
    require_track = True
    required_headers = [
        HEADER_CLIENT_HOST,
        HEADER_CLIENT_COOKIE,
        HEADER_CONSUMER_CLIENT_IP,
        HEADER_CLIENT_USER_AGENT,
    ]

    @property
    def basic_form(self):
        if settings.IS_INTRANET:
            return ChangePasswordCommitIntranetForm
        return ChangePasswordCommitForm

    def check_track(self):
        # Проверим что трек подходит
        if not self.track.is_password_change:
            raise InvalidTrackStateError('Track is not for password change')

    def _validate_and_set_new_password_to_active_directory(self):
        # Выполняем только базовые проверки: историю паролей проверит AD. Полностью на откуп AD не отдаём,
        # так как там недостаточно понятные ошибки.
        password, _ = self.validate_password()
        self.change_password_in_active_directory(
            old_password=self.form_values['current_password'],
            new_password=password,
            is_forced=False,
        )
        events = {
            'action': 'change_password',
            'consumer': self.consumer,
        }
        with UPDATE(self.account, self.request.env, events):
            self.update_revokers(
                global_logout=False,
                revoke_web_sessions=self.form_values['revoke_web_sessions'],
                revoke_tokens=self.form_values['revoke_tokens'],
                revoke_app_passwords=self.form_values['revoke_app_passwords'],
            )

    def _validate_and_set_new_password_to_db(self):
        is_strong_policy = self.account.is_strong_password_required
        self.verify_password(
            self.account.uid,
            self.form_values['current_password'],
            ignore_bruteforce_status=True,  # Не смотрим на bruteforce_status, потому что всегда требуем ввести капчу.
            useragent=self.user_agent,
            referer=self.referer,
            yandexuid=self.cookies.get('yandexuid'),
            retpath=self.track.retpath,
            dbfields=[],  # Не выбирать из ЧЯ доп информацию о пользователе
            attributes=[],  # Аналогично выше
            check_disabled_on_deletion=True,
        )

        # Валидация нового пароля происходит уже после проверки старого.
        # Проверка на совпадение со старым паролем происходит в валидаторе.
        password, quality = self.validate_password(
            is_strong_policy=is_strong_policy,
            old_password_hash=self.account.password.serialized_encrypted_password,
            required_check_password_history=True,
            emails=set([email.address for email in self.account.emails.native]),
            phone_number=self.account_phones.secure.number if self.account_phones.secure else None,
        )

        self.frodo_status = self.frodo_check_spammer(
            action='change_password_voluntarily',
            old_password_quality=self.account.password.quality,
            is_ssl_session_cookie_valid=True,
            account_karma=str(self.account.karma.value) if self.account.karma else None,
        )
        self.assert_account_not_compromised(self.frodo_status)

        events = {
            'action': 'change_password',
            'consumer': self.consumer,
        }
        with UPDATE(self.account, self.request.env, events):
            self.change_password(
                password,
                quality,
                global_logout=is_strong_policy,
                # TODO убрать возможность задать флаги после обновления фронтенда
                revoke_web_sessions=self.form_values['revoke_web_sessions'],
                revoke_tokens=self.form_values['revoke_tokens'],
                revoke_app_passwords=self.form_values['revoke_app_passwords'],
            )
            process_env_profile(self.account, track=self.track)
            reset_challenge_counters(self.account)

    def validate_and_set_new_password(self):
        if settings.IS_INTRANET:
            self._validate_and_set_new_password_to_active_directory()
        else:
            self._validate_and_set_new_password_to_db()

    def process_request(self):
        self.process_basic_form()
        self.read_track()
        self.check_track()  # TODO удалить после перехода на процесс в треке
        self.check_track_for_captcha()
        self.statbox.log(action='submitted')

        self.get_pinned_account_from_session(
            emails=True,
            check_disabled_on_deletion=True,
            need_phones=True,
        )
        self.statbox.bind_context(uid=self.account.uid)

        redirect_state = self.verify_change_password_preconditions_met()
        if redirect_state:
            self.state = redirect_state
            return
        self.response_values['validation_method'] = CAPTCHA_VALIDATION_METHOD

        self.response_values['revokers'] = default_revokers(
            allow_select=not self.account.is_strong_password_required,
        )
        if self.track.retpath:
            self.response_values['retpath'] = self.track.retpath

        with self.track_transaction.commit_on_error():
            self.frodo_status = None

            # Заполним поля трека из текущей сессии
            set_authorization_track_fields(
                self.account,
                self.track,
                allow_create_session=True,
                allow_create_token=False,
                old_session_info=self.session_info,
                password_passed=True,
                session_scope=user_session_scope(self.session_info, self.account.uid),
            )

            self.validate_and_set_new_password()

            # Сбросим флажок, чтобы не было повторных посещений этой ручки
            self.track.is_password_change = False

            self.get_updated_cookies()

            self.send_notifications(self.frodo_status, self.secure_number_or_none)
            self.send_account_modification_push(event_name='changed_password')

            self.statbox.log(
                password_quality=self.form_values['quality'],
                action='changed_password',
                optional_logout_possible=not self.account.is_strong_password_required,
                old_authid=self.session_info.authid,
            )


class ChangePasswordOptionalLogoutView(ChangePasswordBaseView,
                                       BundlePasswordChangeMixin,
                                       BundleAssertCaptchaMixin):
    require_process = True  # TODO это же требование в ручке commit

    require_track = True
    required_headers = [
        HEADER_CLIENT_HOST,
        HEADER_CLIENT_COOKIE,
        HEADER_CONSUMER_CLIENT_IP,
        HEADER_CLIENT_USER_AGENT,
    ]

    basic_form = ChangePasswordOptionalLogoutForm

    def process_request(self):
        self.process_basic_form()
        self.read_track()
        self.check_track_for_captcha()
        self.statbox.log(action='optional_logout_submitted')

        self.get_pinned_account_from_session(
            emails=True,
            check_disabled_on_deletion=True,
            need_phones=True,
        )
        self.statbox.bind_context(uid=self.account.uid)

        if self.account.is_strong_password_required:
            # Для такого аккаунта выход сделали в ручке commit
            raise ActionNotRequiredError()

        redirect_state = self.verify_change_password_preconditions_met()
        if redirect_state:
            self.state = redirect_state
            return

        if self.track.retpath:
            self.response_values['retpath'] = self.track.retpath

        with self.track_transaction.commit_on_error():
            # Заполним поля трека из текущей сессии
            set_authorization_track_fields(
                self.account,
                self.track,
                allow_create_session=True,
                allow_create_token=False,
                old_session_info=self.session_info,
                password_passed=True,
                session_scope=user_session_scope(self.session_info, self.account.uid),
            )

            events = {
                'action': 'change_password_optional_logout',
                'consumer': self.consumer,
            }
            with UPDATE(self.account, self.request.env, events):
                self.update_revokers(
                    global_logout=False,
                    revoke_web_sessions=self.form_values['revoke_web_sessions'],
                    revoke_tokens=self.form_values['revoke_tokens'],
                    revoke_app_passwords=self.form_values['revoke_app_passwords'],
                )

            if self.form_values['revoke_web_sessions']:
                self.get_updated_cookies()

            self.statbox.log(
                action='optional_logout_committed',
                web_sessions_revoked=self.form_values['revoke_web_sessions'],
                tokens_revoked=self.form_values['revoke_tokens'],
                app_passwords_revoked=self.form_values['revoke_app_passwords'],
            )
