# -*- coding: utf-8 -*-

import logging

from passport.backend.api.common.account import default_revokers
from passport.backend.api.common.authorization import (
    set_authorization_track_fields,
    user_session_scope,
)
from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.change_password.forms import (
    ChangeKinopoiskPasswordCommitForm,
    ChangeKinopoiskPasswordSubmitForm,
)
from passport.backend.api.views.bundle.exceptions import (
    AccountInvalidTypeError,
    InvalidTrackStateError,
)
from passport.backend.api.views.bundle.headers import (
    HEADER_CLIENT_COOKIE,
    HEADER_CLIENT_HOST,
    HEADER_CLIENT_USER_AGENT,
    HEADER_CONSUMER_CLIENT_IP,
)
from passport.backend.api.views.bundle.mixins import (
    BundleAccountGetterMixin,
    BundleAccountPropertiesMixin,
    BundleAccountResponseRendererMixin,
    BundleAssertCaptchaMixin,
    BundleFixPDDRetpathMixin,
    BundleFrodoMixin,
    BundlePasswordChangeMixin,
    BundlePasswordValidationMixin,
    BundlePhoneMixin,
    BundleVerifyPasswordMixin,
)
from passport.backend.api.views.bundle.mixins.password import CAPTCHA_VALIDATION_METHOD
from passport.backend.core.logging_utils.loggers.statbox import StatboxLogger
from passport.backend.core.runner.context_managers import UPDATE
from passport.backend.core.utils.decorators import cached_property


log = logging.getLogger('passport.api.view.bundle.account.change_password')


class ChangeKinopoiskPasswordBaseView(BaseBundleView,
                                      BundleAccountGetterMixin,
                                      BundlePhoneMixin,
                                      BundleAccountResponseRendererMixin,
                                      BundleAccountPropertiesMixin):
    required_grants = ['account.change_kinopoisk_password']
    track_type = 'authorize'

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode='change_kinopoisk_password',
            track_id=self.track_id,
        )


class ChangeKinopoiskPasswordSubmitView(ChangeKinopoiskPasswordBaseView,
                                        BundleFixPDDRetpathMixin):
    require_track = False

    required_headers = (
        HEADER_CLIENT_HOST,
        HEADER_CLIENT_COOKIE,
        HEADER_CONSUMER_CLIENT_IP,
    )

    basic_form = ChangeKinopoiskPasswordSubmitForm

    def process_request(self, *args, **kwargs):
        self.process_basic_form()

        self.get_account_from_session(
            emails=False,
            check_disabled_on_deletion=True,
            need_phones=False,
        )

        if not self.account.is_kinopoisk:
            raise AccountInvalidTypeError()

        self.read_or_create_track(self.track_type)

        self.response_values['track_id'] = self.track_id

        with self.track_transaction.commit_on_error() as track:
            if track.uid and track.uid != str(self.account.uid):
                log.warning('Attempt to change uid in track: %s != %s', self.track.uid, self.account.uid)
                raise InvalidTrackStateError()
            track.uid = self.account.uid
            track.country = self.account.person.country
            track.is_strong_password_policy_required = self.account.is_strong_password_required

            if self.form_values['retpath']:
                track.retpath = self.form_values['retpath']
            self.response_values['retpath'] = self.track.retpath

            self.fill_response_with_account()

            # Проверяем, есть ли вообще пароль у пользователя
            self.check_have_password()

            track.is_password_change = True
            # При добровольной смене пароля капча всегда обязательна.
            track.is_captcha_required = True
            self.response_values['validation_method'] = CAPTCHA_VALIDATION_METHOD

            self.response_values['revokers'] = default_revokers(
                allow_select=not self.account.is_strong_password_required,
            )


class ChangeKinopoiskPasswordCommitView(ChangeKinopoiskPasswordBaseView,
                                        BundlePasswordValidationMixin,
                                        BundlePasswordChangeMixin,
                                        BundleVerifyPasswordMixin,
                                        BundleFrodoMixin,
                                        BundleAssertCaptchaMixin):
    require_track = True
    track_type = 'authorize'

    required_grants = ['account.change_kinopoisk_password']

    required_headers = [
        HEADER_CLIENT_HOST,
        HEADER_CLIENT_COOKIE,
        HEADER_CONSUMER_CLIENT_IP,
        HEADER_CLIENT_USER_AGENT,
    ]

    basic_form = ChangeKinopoiskPasswordCommitForm

    def check_track(self):
        # Проверим что трек подходит
        if not self.track.is_password_change:
            raise InvalidTrackStateError('Track is not for password change')

    def process_request(self):
        self.statbox.bind_context(
            user_agent=self.user_agent,
            ip=self.client_ip,
            consumer=self.consumer,
        )
        self.statbox.log(
            action='submitted',
        )
        self.process_basic_form()

        self.get_account_from_session(
            emails=False,
            check_disabled_on_deletion=True,
            need_phones=False,
        )

        if not self.account.is_kinopoisk:
            raise AccountInvalidTypeError()

        self.read_track()
        self.check_track()
        self.statbox.bind_context(uid=self.account.uid)

        # Всегда отдаём информацию про аккаунт при всех ошибках
        # Необходимо фронтенду, чтобы отрисовать залогиновую страничку
        self.fill_response_with_account()

        # Перестрахуемся
        self.assert_uid_in_track_is_valid()

        self.check_have_password()
        self.response_values['validation_method'] = CAPTCHA_VALIDATION_METHOD

        self.check_track_for_captcha()

        with self.track_transaction.commit_on_error():
            is_strong_policy = self.account.is_strong_password_required

            self.verify_password(
                self.account.uid,
                self.form_values['current_password'],
                ignore_bruteforce_status=True,  # Не смотрим на bruteforce_status, потому что всегда требуем ввести капчу.
                useragent=self.user_agent,
                referer=self.referer,
                yandexuid=self.cookies.get('yandexuid'),
                retpath=self.track.retpath,
                dbfields=[],  # Не выбирать из ЧЯ доп информацию о пользователе
                attributes=[],  # Аналогично выше
            )

            # Валидация нового пароля происходит уже после проверки старого.
            # Проверка на совпадение со старым паролем происходит в валидаторе.
            password, quality = self.validate_password(
                is_strong_policy=is_strong_policy,
                old_password_hash=self.account.password.serialized_encrypted_password,
                required_check_password_history=True,
                emails=set([email.address for email in self.account.emails.native]),
            )

            events = {
                'action': 'change_password',
                'consumer': self.consumer,
            }

            # Заполним поля трека из текущей сессии
            set_authorization_track_fields(
                self.account,
                self.track,
                allow_create_session=True,
                allow_create_token=False,
                old_session_info=self.session_info,
                password_passed=True,
                session_scope=user_session_scope(self.session_info, self.account.uid),
            )

            with UPDATE(self.account, self.request.env, events):
                self.change_password(
                    password,
                    quality,
                    global_logout=is_strong_policy,
                    revoke_web_sessions=self.form_values['revoke_web_sessions'],
                    revoke_tokens=self.form_values['revoke_tokens'],
                    revoke_app_passwords=self.form_values['revoke_app_passwords'],
                )

            # Сбросим флажок, чтобы не было повторных посещений этой ручки
            self.track.is_password_change = False

            self.statbox.log(
                password_quality=self.form_values['quality'],
                action='changed_password',
            )
