# -*- coding: utf-8 -*-

import logging

from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.change_password.forms import ChangeScholarPasswordForm
from passport.backend.api.views.bundle.headers import (
    HEADER_CLIENT_USER_AGENT,
    HEADER_CONSUMER_CLIENT_IP,
)
from passport.backend.api.views.bundle.mixins import (
    BundleAccountGetterMixin,
    BundlePasswordChangeMixin,
    BundleScholarMixin,
)
from passport.backend.core.logging_utils.loggers.statbox import StatboxLogger
from passport.backend.core.runner.context_managers import UPDATE
from passport.backend.core.utils.decorators import cached_property


ACCOUNT_CHANGE_SCHOLAR_PASSWORD_GRANT = 'account.change_scholar_password'

log = logging.getLogger('passport.api.view.bundle.account.change_password')


class ChangeScholarPasswordView(
    BundleScholarMixin,
    BundlePasswordChangeMixin,
    BundleAccountGetterMixin,
    BaseBundleView,
):
    basic_form = ChangeScholarPasswordForm
    required_grants = [ACCOUNT_CHANGE_SCHOLAR_PASSWORD_GRANT]
    required_headers = [
        HEADER_CLIENT_USER_AGENT,
        HEADER_CONSUMER_CLIENT_IP,
    ]

    def process_request(self):
        self.process_basic_form()
        self.statbox.log(action='submitted')
        self.get_account_by_uid(self.form_values.get('uid'))
        self.validate_scholar_password()

        with UPDATE(
            self.account,
            self.request.env,
            dict(action='change_scholar_password', consumer=self.consumer),
        ):
            self.set_scholar_password(self.form_values.get('password'))

            if self.form_values.get('revoke_credentials'):
                self.update_revokers()

        self.statbox.log(
            action='scholar_password_changed',
            uid=str(self.account.uid),
        )

    @cached_property
    def statbox(self):
        return StatboxLogger(
            consumer=self.consumer,
            ip=self.client_ip,
            mode='change_scholar_password',
            user_agent=self.user_agent,
        )

    def is_account_type_allowed(self):
        return bool(self.account and self.account.scholar_alias)
