# -*- coding: utf-8 -*-

from passport.backend.api.common.authorization import (
    set_authorization_track_fields,
    user_session_scope,
)
from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.constants import (
    AUTHENTICATION_MEDIA_SESSION,
    AUTHENTICATION_MEDIA_TOKEN,
)
from passport.backend.api.views.bundle.exceptions import (
    AccountUidMismatchError,
    InvalidTrackStateError,
    UidNotInSessionError,
)
from passport.backend.api.views.bundle.headers import (
    HEADER_CLIENT_ACCEPT_LANGUAGE,
    HEADER_CLIENT_COOKIE,
    HEADER_CLIENT_HOST,
    HEADER_CLIENT_USER_AGENT,
    HEADER_CONSUMER_AUTHORIZATION,
    HEADER_CONSUMER_CLIENT_IP,
)
from passport.backend.api.views.bundle.mixins import BundlePhoneMixin
from passport.backend.api.views.bundle.mixins.account import (
    BundleAccountGetterMixin,
    BundleAccountPropertiesMixin,
    BundleAccountResponseRendererMixin,
    RequestCredentialsAllMissingError,
)
from passport.backend.api.views.bundle.mixins.common import BundleFrodoMixin
from passport.backend.api.views.bundle.states import UpgradeCookies
from passport.backend.api.views.bundle.utils import assert_valid_host
from passport.backend.core.logging_utils.loggers.statbox import StatboxLogger
from passport.backend.core.models.account import Service
from passport.backend.core.utils.decorators import cached_property


BASIC_GRANT = 'account.complete'


class BaseAccountCompleteView(BaseBundleView,
                              BundleAccountGetterMixin,
                              BundleAccountPropertiesMixin,
                              BundleFrodoMixin,
                              BundleAccountResponseRendererMixin,
                              BundlePhoneMixin):
    required_headers = [
        HEADER_CLIENT_ACCEPT_LANGUAGE,
        HEADER_CONSUMER_CLIENT_IP,
        HEADER_CLIENT_USER_AGENT,
    ]
    required_grants = [BASIC_GRANT]

    track_type = 'complete'

    MODE = 'complete'

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode=self.MODE,
            track_id=self.track_id,
            consumer=self.consumer,
            user_agent=self.user_agent,
            ip=self.client_ip,
        )

    def is_completion_available(self):
        return (
            self.account.is_lite or
            self.account.is_neophonish or
            self.account.is_social or

            # Соц пользователь, выставивший себе нормальный логин, но еще
            # не имеющий пароля.
            (self.account.is_subscribed(Service.by_slug('social')) and not self.account.have_password)
        )

    def get_account(self, is_submit=False):
        """
        Получим аккаунт из трека или средства аутентикации.

        Из трека получаем аккаунт, если пройдена аутентикация паролем или
        по одноразовому ключу.

        Убедимся, что uid из трека совпадает с uid-ом в средстве аутентикации.
        """

        if self.track.is_key_auth_passed or self.track.is_password_passed:
            self.get_account_from_track(
                check_disabled_on_deletion=True,
                need_phones=True,
                get_public_id=True,
            )
            if not is_submit and self.authentication_media.is_extendable():
                # Необходимо также проверить через ЧЯ сессию пользователя --
                # её мы будем расширять.
                self.authentication_media.get_metadata()
        else:
            expected_uid = int(self.track.uid) if self.track.uid else None
            try:
                self.authentication_media.get_account(
                    expected_uid=expected_uid,
                    need_phones=True,
                    check_disabled_on_deletion=True,
                    get_public_id=True,
                )
            except AccountUidMismatchError as e:
                raise InvalidTrackStateError(
                    "track.uid (%s) is not in the current user's authentication "
                    'media (%s)' % (self.track.uid, e.known_uids),
                )
        self.statbox.bind_context(uid=self.account.uid)

    def select_authentication_media(self):
        try:
            media = self.select_authentication_media_name(
                enabled_media={
                    AUTHENTICATION_MEDIA_SESSION,
                    AUTHENTICATION_MEDIA_TOKEN,
                },
            )
        except RequestCredentialsAllMissingError:
            media = AUTHENTICATION_MEDIA_SESSION
        if media == AUTHENTICATION_MEDIA_SESSION:
            return SessionAuthenticationMedia(self)
        elif media == AUTHENTICATION_MEDIA_TOKEN:
            return TokenAuthenticationMedia(self)
        else:
            raise NotImplementedError()  # pragma: no cover


class SessionAuthenticationMedia(object):
    def __init__(self, view):
        self._view = view

    def check_http_requirements(self):
        self._view.check_headers([
            HEADER_CLIENT_HOST,
            HEADER_CLIENT_COOKIE,
        ])
        assert_valid_host(self._view.request.env)

    def save_to_track_as_old(self):
        self._view.set_old_session_track_fields(self._view.session_info)

    def set_track_fields(self, password_passed):
        set_authorization_track_fields(
            self._view.account,
            self._view.track,
            allow_create_session=True,
            allow_create_token=False,
            password_passed=password_passed,
            session_scope=user_session_scope(self._view.session_info, self._view.account.uid),
        )

    def fill_response(self):
        self._view.fill_response_with_account_and_session(
            cookie_session_info=self._view.session_info,
            personal_data_required=True,
        )

    def is_extendable(self):
        return True

    def get_metadata(self):
        self._view.session_info = self._view.check_session_cookie()
        return self._view.session_info

    def get_account(self, expected_uid=None, **kwargs):
        if expected_uid is not None:
            kwargs['multisession_uid'] = expected_uid
        try:
            return self._view.get_account_from_session(**kwargs)
        except UidNotInSessionError as e:
            raise AccountUidMismatchError(known_uids=e.known_uids)

    def set_state_if_completion_not_required(self):
        # В случае дорегистрации по ссылке или принудительной дорегистрации лайта
        # может возникнуть гонка в разных браузерах, если в одном дорегистрировались,
        # а во втором кука стала невалидна.
        session_response = self._view.session_info.response
        if session_response and session_response['is_lite_session']:
            self._view.state = UpgradeCookies()


class TokenAuthenticationMedia(object):
    def __init__(self, view):
        self._view = view

    def check_http_requirements(self):
        self._view.check_headers([HEADER_CONSUMER_AUTHORIZATION])

    def save_to_track_as_old(self):
        pass

    def set_track_fields(self, password_passed):
        set_authorization_track_fields(
            self._view.account,
            self._view.track,
            allow_create_session=False,
            allow_create_token=True,
            password_passed=password_passed,
        )

    def fill_response(self):
        self._view.fill_response_with_account(personal_data_required=True)

    def is_extendable(self):
        return False

    def get_metadata(self):
        raise NotImplementedError()  # pragma: no cover

    def get_account(self, expected_uid=None, **kwargs):
        if expected_uid is not None:
            kwargs['uid'] = expected_uid
        self._view.get_account_by_oauth_token(**kwargs)

    def set_state_if_completion_not_required(self):
        pass
