# -*- coding: utf-8 -*-
from datetime import datetime
import logging

from passport.backend.api.common.account import (
    build_default_person_info,
    fill_person_from_args,
)
from passport.backend.api.common.profile.profile import process_env_profile
from passport.backend.api.forms.register import (
    CAPTCHA_VALIDATION_METHOD,
    PHONE_VALIDATION_METHOD,
)
from passport.backend.api.views.bundle.auth.base import BundleBaseAuthorizationMixin
from passport.backend.api.views.bundle.exceptions import (
    EulaIsNotAcceptedError,
    InvalidTrackStateError,
    LoginLikePasswordError,
    UserNotVerifiedError,
)
from passport.backend.api.views.bundle.headers import HEADER_CONSUMER_CLIENT_IP
from passport.backend.api.views.bundle.mixins import (
    BindRelatedPhonishAccountMixin,
    BundleAccountPropertiesMixin,
    BundleAssertCaptchaMixin,
    BundlePasswordChangeMixin,
    BundlePasswordValidationMixin,
    BundleVerifyPasswordMixin,
)
from passport.backend.api.views.bundle.mixins.common import BundleCleanWebMixin
from passport.backend.api.views.bundle.register.exceptions import RegistrationSmsSendPerIPLimitExceededError
from passport.backend.api.views.bundle.states import (
    RedirectToForcedLiteCompletion,
    RedirectToLiteCompletion,
    RedirectToNeophonishCompletion,
    RedirectToSocialCompletion,
    RedirectToSocialCompletionWithLogin,
)
from passport.backend.api.views.bundle.utils import (
    make_hint_question,
    write_phone_to_log,
)
from passport.backend.api.views.register import (
    incr_registration_sms_per_ip_counter,
    is_registration_sms_per_ip_counter_exceed,
)
from passport.backend.core.conf import settings
from passport.backend.core.models.password import check_password_equals_sha256_hash
from passport.backend.core.runner.context_managers import UPDATE
from passport.backend.core.services import Service
from passport.backend.core.subscription import add_subscription
from passport.backend.core.types.login.login import normalize_login

from . import forms
from .base import BaseAccountCompleteView


log = logging.getLogger('passport.api.view.bundle.account.complete')


class CompleteStatusView(BaseAccountCompleteView):
    require_track = False
    required_grants = ['account.completion_status']
    required_headers = [
        HEADER_CONSUMER_CLIENT_IP,
    ]
    basic_form = forms.CompleteStatusForm

    def process_request(self):
        self.authentication_media = self.select_authentication_media()
        self.authentication_media.check_http_requirements()

        self.process_basic_form()

        self.authentication_media.get_account(
            need_phones=True,
            check_disabled_on_deletion=True,
            get_public_id=True,
        )

        is_complete = (self.account.is_normal or self.account.is_pdd) and self.account.have_password
        is_completion_available = not is_complete and self.is_completion_available()

        if not is_completion_available:
            # Дорегистрация невозможна, предлагать нечего.
            is_completion_recommended = False
        elif self.account.is_neophonish and not self.account.person.firstname and not self.account.person.lastname:
            # Неофониши 2.0. Им предлагаем дорегистрацию, если ранее не предлагали или же если с последнего отказа
            # прошло достаточно времени.
            completion_postponed_at = self.form_values['completion_postponed_at']
            is_completion_recommended = (
                completion_postponed_at is None or
                datetime.now() - completion_postponed_at >= settings.NEOPHONISH_COMPLETION_SUGGEST_INTERVAL
            )
        else:
            # Остальным пока не предлагаем дорегистрацию.
            is_completion_recommended = False

        is_completion_required = False  # пока никого не принуждаем

        self.response_values.update(
            is_complete=is_complete,
            is_completion_available=is_completion_available,
            is_completion_recommended=is_completion_recommended,
            is_completion_required=is_completion_required,
        )
        if is_completion_available:
            if self.consumer == settings.MOBILEPROXY_CONSUMER:
                completion_url_template = settings.COMPLETION_URL_AM_TEMPLATE
                default_tld = 'com'
            else:
                completion_url_template = settings.COMPLETION_URL_TEMPLATE
                default_tld = settings.PASSPORT_DEFAULT_TLD

            tld = settings.LANGUAGE_TO_TLD_MAPPING.get(self.form_values['language'], default_tld)
            self.response_values.update(
                completion_url=completion_url_template % {'tld': tld},
            )


class CompleteSubmitView(BaseAccountCompleteView):
    require_track = False
    basic_form = forms.CompleteSubmitForm

    def process_request(self):
        self.authentication_media = self.select_authentication_media()
        self.authentication_media.check_http_requirements()

        self.read_or_create_track(self.track_type)
        self.process_basic_form()

        self.get_account(is_submit=True)

        self.response_values['track_id'] = self.track_id
        if self.form_values['retpath']:
            self.response_values['retpath'] = self.form_values['retpath']

        self.fill_response_with_account()

        if not self.is_completion_available():
            # Вернем ответ без state и с retpath (если передан), значит ничего
            # делать не нужно.
            self.authentication_media.set_state_if_completion_not_required()
            return

        with self.track_transaction.rollback_on_error() as track:
            self.set_uid_to_track(self.account.uid, track)
            track.login = self.account.login
            track.is_strong_password_policy_required = self.account.is_strong_password_required
            if self.form_values['retpath']:
                track.retpath = self.form_values['retpath']

        self.response_values['has_recovery_method'] = self.check_has_recovery_method()

        # Выбираем правильный state.
        if self.account.is_lite:
            if self.track.is_force_complete_lite:
                self.state = RedirectToForcedLiteCompletion()
            else:
                self.state = RedirectToLiteCompletion()
        elif self.account.is_neophonish:
            if self.form_values['can_handle_neophonish']:
                self.state = RedirectToNeophonishCompletion()
            else:
                # Костыль для старых АМ: неофониш может пройти по более длинному лайтовому флоу и дорегаться
                self.state = RedirectToLiteCompletion()
        else:
            # Соц аккаунт. account.is_social означает, что логин не был установлен пользователем.
            if self.account.is_social:
                self.state = RedirectToSocialCompletionWithLogin()
            else:
                self.state = RedirectToSocialCompletion()


class CompleteCommitBaseView(BaseAccountCompleteView,
                             BindRelatedPhonishAccountMixin,
                             BundleAccountPropertiesMixin,
                             BundleAssertCaptchaMixin,
                             BundleBaseAuthorizationMixin,
                             BundleCleanWebMixin,
                             BundlePasswordValidationMixin,
                             BundlePasswordChangeMixin,
                             BundleVerifyPasswordMixin):
    basic_form = forms.CompleteCommitBaseForm
    require_track = True
    statbox_type = None

    def process_request(self):
        self.authentication_media = self.select_authentication_media()
        self.authentication_media.check_http_requirements()

        self.read_track()
        self.response_values['track_id'] = self.track_id
        self.response_values['retpath'] = self.track.retpath

        self.process_basic_form()
        self.clean_web_check_form_values()

        self.check_auth_not_passed()
        self.get_account()

        # Всегда отдаём информацию про аккаунт при всех ошибках
        # Необходимо фронтенду, чтобы отрисовать залогиновую страничку
        self.fill_response_with_account()

        if not self.is_completion_available():
            # Вернем ответ без state и с retpath (если передан), значит ничего
            # делать не нужно.
            self.authentication_media.set_state_if_completion_not_required()
            return

        if not self.is_valid_completion_method():
            raise InvalidTrackStateError()

        self.process_account_specific_form()

        self.check_user_validated()

        if self.is_validated_via_phone:
            self.assert_registration_sms_per_ip_counter_not_exceeded()

        with self.track_transaction.rollback_on_error():
            self.authentication_media.save_to_track_as_old()
            self.track.is_strong_password_policy_required = self.account.is_strong_password_required
            self.process_account()
            self.authentication_media.set_track_fields(password_passed=True)
            self.authentication_media.fill_response()

        self.frodo_check_spammer(
            action='complete',
            service=self.consumer,
            increment_counters=True,
            account_karma=str(self.account.karma.value) if self.account.karma else None,
        )

        write_phone_to_log(self.account, self.cookies)
        self.try_bind_related_phonish_account()

        if self.is_validated_via_phone:
            incr_registration_sms_per_ip_counter(self.request.env)

    def check_user_validated(self):
        if self.has_recovery_method:
            return

        if not (self.is_validated_via_phone or self.is_validated_via_captcha):
            raise UserNotVerifiedError()

    def postprocess_account(self):
        # Обновляем display_name
        if self.account.person.display_name.is_social:
            # PASSP-23929 Если стоит атрибут person.dont_use_displayname_as_public_name,
            # следует обнулять display_name.name
            if self.account.person.dont_use_displayname_as_public_name:
                self.account.person.display_name.name = None
            self.account.person.display_name.convert_to_passport()

    def process_account(self):
        if self.account.have_password:
            self.verify_password()
        else:
            password, password_quality = self.validate_password(
                emails=set(e.address for e in self.account.emails.native),
                is_strong_policy=self.account.is_strong_password_required,
            )

        if self.is_validated_via_captcha and not self.has_recovery_method:
            # Контрольный вопрос создаётся заранее, потому что здесь он ещё и
            # проверяется. А все проверки лучше сделать до создания телефонных
            # операций, которые фиксируются в БД.
            question = make_hint_question(
                question=self.form_values['question'],
                question_id=self.form_values['question_id'],
                display_language=self.form_values['display_language'],
            )

        should_save_phone = self.should_save_phone()
        if should_save_phone:
            self.save_phone_submit()

        events = {'action': 'complete', 'consumer': self.consumer}
        with UPDATE(self.account, self.request.env, events):
            if not self.account.portal_alias:
                self.account.set_portal_alias(self.form_values['login'])
                self.account.subscriptions[8].login = self.form_values['login']

            add_subscription(self.account, Service.by_slug('mail'))

            if not self.account.have_password:
                # Обновляем пароль
                self.change_password(
                    password,
                    password_quality,
                    global_logout=False,
                )

            if self.is_validated_via_captcha and not self.has_recovery_method:
                # Обновляем КО/КВ
                self.account.hint.question = question
                self.account.hint.answer = self.form_values['answer']

            if should_save_phone:
                self.save_phone_commit()

            # Обновляем персональные данные
            fill_person_from_args(
                self.account.person,
                self.form_values,
                build_default_person_info(self.client_ip),
            )

            self.postprocess_account()

            validated_via = None
            if self.is_validated_via_phone:
                validated_via = 'phone'
            elif self.is_validated_via_captcha:
                validated_via = 'captcha'
            self.statbox.log(
                validated=validated_via,
                login=self.account.normalized_login,
                type=self.statbox_type,
            )

            process_env_profile(self.account, track=self.track)

        self.track.user_entered_login = self.account.login

        if should_save_phone:
            self.save_phone_after_commit()

    def verify_password(self):
        if 'password' in self.form_values:
            super(CompleteCommitBaseView, self).verify_password(
                self.account.uid,
                self.form_values['password'],
                retpath=self.track.retpath,
                ignore_statbox_log=True,
            )
            # Проверка пароля прошла успешно, сравним выбранный логин с текущим
            # паролем
            is_login_like_password = (
                normalize_login(self.form_values['login']) == normalize_login(self.form_values['password'])
            )
        else:
            logins_to_check = {self.form_values['login'], normalize_login(self.form_values['login'])}
            is_login_like_password = False
            for login in logins_to_check:
                if check_password_equals_sha256_hash(login, self.track.password_hash):
                    is_login_like_password = True
                    break
        if is_login_like_password:
            raise LoginLikePasswordError()

    @property
    def has_recovery_method(self):
        return self.check_has_recovery_method()

    @property
    def is_validated_via_phone(self):
        return (
            self.form_values['validation_method'] == PHONE_VALIDATION_METHOD and
            self.is_phone_confirmed_in_track(allow_by_flash_call=True)
        )

    @property
    def is_validated_via_captcha(self):
        return (self.form_values['validation_method'] == CAPTCHA_VALIDATION_METHOD and
                self.is_captcha_passed)

    def assert_registration_sms_per_ip_counter_not_exceeded(self):
        if is_registration_sms_per_ip_counter_exceed(
            self.request.env,
            self.track,
            mode=self.MODE,
        ):
            raise RegistrationSmsSendPerIPLimitExceededError()

    def should_save_phone(self):
        return self.is_validated_via_phone and not self.account.phones.secure

    def save_phone_submit(self):
        self.save_secure_phone = self.build_save_secure_phone(
            phone_number=self.saved_number,
            check_account_type_on_submit=False,
        )
        self.save_secure_phone.submit()

    def save_phone_commit(self):
        self.save_secure_phone.commit()

    def save_phone_after_commit(self):
        self.save_secure_phone.after_commit()

    def is_valid_completion_method(self):
        raise NotImplementedError()  # pragma: no cover

    def process_account_specific_form(self):
        raise NotImplementedError()  # pragma: no cover


class CompleteCommitSocialView(CompleteCommitBaseView):
    statbox_type = 'social'

    def is_valid_completion_method(self):
        """
        В эту ручку должны приходить только аккаунты с установленным логином и
        без пароля.
        """
        return (
            self.account.social_alias and
            self.account.portal_alias and
            not self.account.have_password
        )

    def process_account_specific_form(self):
        self.form_values.update(
            self.process_form(forms.CompleteCommitSocialForm(), self.all_values),
        )


class CompleteCommitSocialWithLoginView(CompleteCommitBaseView):
    statbox_type = 'social-password'

    def is_valid_completion_method(self):
        """
        В эту ручку должны приходить только аккаунты без логина и без пароля
        (что работает автоматически).
        """
        return self.account.social_alias and not self.account.portal_alias

    def process_account_specific_form(self):
        self.form_values.update(
            self.process_form(forms.CompleteCommitSocialWithLoginForm(), self.all_values),
        )


class CompleteCommitLiteView(CompleteCommitBaseView):
    statbox_type = 'lite'

    def is_valid_completion_method(self):
        """
        В эту ручку должны приходить только lite аккаунты с полноценной сессией. Или же неофониши (из старых АМ).
        """
        return self.account.is_lite or self.account.is_neophonish

    def process_account_specific_form(self):
        self.form_values.update(
            self.process_form(forms.CompleteCommitLiteForm(), self.all_values),
        )

        if not self.form_values['eula_accepted']:
            raise EulaIsNotAcceptedError()


class ForceCompleteCommitLiteView(CompleteCommitBaseView):
    track_type = 'authorize'
    statbox_type = 'lite'

    def is_valid_completion_method(self):
        """
        В эту ручку должны приходить только lite аккаунты с соответствующим
        состоянием трека и проверенным паролем.
        """
        return (
            self.account.is_lite and
            self.track.is_force_complete_lite and
            self.track.is_password_passed
        )

    def process_account_specific_form(self):
        self.form_values.update(
            self.process_form(forms.CompleteCommitLiteForcedForm(), self.all_values),
        )

        if not self.form_values['eula_accepted']:
            raise EulaIsNotAcceptedError()


class CompleteCommitNeophonishView(CompleteCommitBaseView):
    statbox_type = 'neophonish'

    def is_valid_completion_method(self):
        """
        В эту ручку должны приходить только неофониши.
        """
        return (
            self.account.is_neophonish and
            # У всех неофонишей есть основной телефон, но реплика шарда может
            # отстать, и так мы проверяем, что всё консистентно
            self.account.phones.secure
        )

    def process_account_specific_form(self):
        self.form_values.update(
            self.process_form(forms.CompleteCommitNeophonishForm(), self.all_values),
        )

    @property
    def is_validated_via_phone(self):
        """
        Неофониш может авторизоваться только по смс или после регистрации.
        Повторная валидация телефона не требуется.
        """
        return True

    def postprocess_account(self):
        super(CompleteCommitNeophonishView, self).postprocess_account()
        # После дорегистрации неофониша телефон перестаёт быть единственным средством для входа,
        # так что его становится можно отвязывать при превышении лимита.
        self.account.phones.secure.binding.should_ignore_binding_limit = False
