# -*- coding: utf-8 -*-

import logging

from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.device_public_key.exceptions import (
    CheckFailedDevicePublicKeyError,
    ConflictDevicePublicKeyError,
    InvalidOwnerDevicePublicKeyError,
    NeedNewNonceError,
    NotFoundDevicePublicKeyError,
)
from passport.backend.api.views.bundle.device_public_key.forms import (
    CreateForm,
    DeleteForm,
    UpdateForm,
)
from passport.backend.api.views.bundle.exceptions import (
    InternalPermanentError,
    ResourceUnavailableError,
)
from passport.backend.api.views.bundle.headers import HEADER_CONSUMER_CLIENT_IP
from passport.backend.api.views.bundle.mixins.kolmogor import KolmogorMixin
from passport.backend.core.builders.blackbox.constants import BLACKBOX_CHECK_DEVICE_SIGNATURE_RETRIABLE_STATUS
from passport.backend.core.builders.blackbox.exceptions import (
    BlackboxInvalidDeviceSignature,
    BlackboxTemporaryError,
)
from passport.backend.core.conf import settings
from passport.backend.core.dbmanager.exceptions import DBError
from passport.backend.core.device_public_key import (
    delete_device_public_key,
    find_device_public_key,
    insert_device_public_key,
    update_device_public_key,
)
from passport.backend.core.grants import check_grant
from passport.backend.core.logging_utils.loggers.statbox import StatboxLogger
from passport.backend.core.models.device_public_key import DevicePublicKey
from passport.backend.core.utils.decorators import cached_property


log = logging.getLogger(__name__)


DRIVE_DEVICE_CONSUMER = 'drive_device'


class BaseView(BaseBundleView):
    action = None
    grant_base = 'device_public_key_%(action)s.%(owner)s'

    def process_request(self):
        self.process_basic_form()

        grant = self.build_action_grant(self.form_values['owner'])
        self.check_grant(grant)

        self.owner_id = self.owner_to_owner_id(self.form_values['owner'])

        self.process_request2()

    @cached_property
    def statbox(self):
        return StatboxLogger(consumer=self.consumer)

    def respond_success(self):
        self.save_request_status_to_statbox(status='ok')
        return super(BaseView, self).respond_success()

    def respond_error(self, error):
        self.save_request_status_to_statbox(status='error')
        return super(BaseView, self).respond_error(error)

    def escape_grant(self, owner):
        # Точка в названии гранта Паспорта имеет сакральный смысл, поэтому
        # лучше её заменить другой литерой.
        return owner.replace('.', '_')

    def owner_to_owner_id(self, owner):
        owner_id = settings.DEVICE_PUBLIC_KEY_OWNER_TO_OWNER_ID.get(owner)
        if owner_id is None:
            raise InvalidOwnerDevicePublicKeyError()
        return owner_id

    def save_request_status_to_statbox(self, status):
        action = '%s_device_public_key' % self.action
        self.statbox.log(
            status=status,
            action=action,
            device_id=self.form_values.get('device_id'),
            owner=self.form_values.get('owner'),
        )

    def build_action_grant(self, owner):
        owner = self.escape_grant(owner)
        return self.grant_base % dict(action=self.action, owner=owner)


class DriveCreateView(BaseView, KolmogorMixin):
    action = 'create'
    basic_form = CreateForm

    required_headers = [HEADER_CONSUMER_CLIENT_IP]

    def process_request(self):
        self.check_api_enabled()
        super(DriveCreateView, self).process_request()

    def process_request2(self):
        if self.owner_id == settings.DRIVE_PRODUCTION_PUBLIC_KEY_OWNER_ID:
            self.check_device_ip(self.client_ip)
        self.check_device_signature(
            self.form_values['check_nonce'],
            settings.DRIVE_NONCE_SIGN_SPACE,
            self.form_values['device_id'],
            self.form_values['check_signature'],
            self.form_values['public_key'],
            version=1,
        )
        kolmogor_available = self.failsafe_check_kolmogor_counters(
            [
                self.create_rps_counter,
                self.create_rpd_counter,
            ],
        )
        new_key = DevicePublicKey(
            device_id=self.form_values['device_id'],
            public_key=self.form_values['public_key'],
            version=1,
            owner_id=self.owner_id,
        )
        old_key = self.find_device_public_key(self.form_values['device_id'])
        if old_key:
            log.debug('Found public key for device {}, actual owner id is {}, expected owner id is {}'.format(
                self.form_values['device_id'],
                old_key.owner_id,
                self.owner_id,
            ))
            if old_key != new_key:
                raise ConflictDevicePublicKeyError()
        elif not old_key:
            self.insert_device_public_key(new_key)
        if kolmogor_available:
            kolmogor_available = self.failsafe_inc_kolmogor_counters(
                [
                    self.create_rps_counter,
                    self.create_rpd_counter,
                ],
            )

    def check_device_signature(
        self,
        nonce,
        nonce_sign_space,
        device_id,
        signature,
        public_key,
        version,
    ):
        try:
            self.blackbox.check_device_signature(
                nonce,
                nonce_sign_space,
                device_id,
                signature,
                public_key=public_key,
                version=version,
            )
        except BlackboxInvalidDeviceSignature as e:
            log.debug('Invalid signature: %s, %s' % (e, e.security_info))
            if e.status in BLACKBOX_CHECK_DEVICE_SIGNATURE_RETRIABLE_STATUS:
                raise NeedNewNonceError()
            else:
                raise InternalPermanentError()
        except BlackboxTemporaryError:
            log.debug('Blackbox temporary unavailable')
            raise ResourceUnavailableError()

    def insert_device_public_key(self, *args, **kwargs):
        try:
            return insert_device_public_key(*args, **kwargs)
        except DBError as e:
            log.debug('Database temporary unavailable: %s' % type(e).__name__)
            raise ResourceUnavailableError()

    def find_device_public_key(self, device_id):
        try:
            device_public_key = find_device_public_key(device_id)
        except BlackboxTemporaryError:
            log.debug('Blackbox temporary unavailable')
            raise ResourceUnavailableError()
        else:
            return device_public_key

    def check_api_enabled(self):
        if not settings.DRIVE_DEVICE_PUBLIC_KEY_API_ENABLED:
            log.debug('Drive device public key API disabled')
            raise InternalPermanentError()

    def check_device_ip(self, device_ip):
        check_grant(
            'auth_forward_drive.public_access',
            device_ip,
            DRIVE_DEVICE_CONSUMER,
        )

    @cached_property
    def create_rpd_counter(self):
        return self.build_counter(
            settings.DEVICE_PUBLIC_KEY_KOLMOGOR_KEY_SPACE,
            settings.DRIVE_CREATE_DEVICE_PUBLIC_KEY_RPD_COUNTER,
            settings.COUNTERS[settings.DRIVE_CREATE_DEVICE_PUBLIC_KEY_RPD_COUNTER],
        )

    @cached_property
    def create_rps_counter(self):
        return self.build_counter(
            settings.DEVICE_PUBLIC_KEY_SHORT_KOLMOGOR_KEY_SPACE,
            settings.DRIVE_CREATE_DEVICE_PUBLIC_KEY_RPS_COUNTER,
            settings.COUNTERS[settings.DRIVE_CREATE_DEVICE_PUBLIC_KEY_RPS_COUNTER],
        )


class UpdateView(BaseView):
    action = 'update'
    basic_form = UpdateForm

    def process_request2(self):
        self.check_device_signature(
            self.form_values['check_nonce'],
            self.form_values['check_nonce_sign_space'],
            self.form_values['device_id'],
            self.form_values['check_signature'],
            self.form_values['public_key'],
            version=1,
        )
        public_key_found = update_device_public_key(
            self.form_values['device_id'],
            self.owner_id,
            self.form_values['public_key'],
            version=1,
        )

        if not public_key_found:
            raise NotFoundDevicePublicKeyError()

    def check_device_signature(
        self,
        nonce,
        nonce_sign_space,
        device_id,
        signature,
        public_key,
        version,
    ):
        try:
            self.blackbox.check_device_signature(
                nonce,
                nonce_sign_space,
                device_id,
                signature,
                public_key=public_key,
                version=version,
            )
        except BlackboxInvalidDeviceSignature as e:
            log.debug('Invalid signature: %s, %s' % (e, e.security_info))
            raise CheckFailedDevicePublicKeyError()


class DeleteView(BaseView):
    action = 'delete'
    basic_form = DeleteForm

    def process_request2(self):
        public_key_found = delete_device_public_key(
            self.form_values['device_id'],
            self.owner_id,
        )

        if not public_key_found:
            raise NotFoundDevicePublicKeyError()
