# -*- coding: utf-8 -*-

from datetime import datetime
import logging

from passport.backend.api.common.common import extract_tld
from passport.backend.api.email_validator.exceptions import (
    EmailAlreadyConfirmedError,
    EmailGenericError,
    EmailIncorrectKeyError,
    EmailIsNativeError,
)
from passport.backend.api.templatetags import escape_percents
from passport.backend.api.views.bundle.constants import (
    BIND_EMAIL_OAUTH_SCOPE,
    X_TOKEN_OAUTH_SCOPE,
)
from passport.backend.api.views.bundle.exceptions import (
    InvalidTrackStateError,
    PasswordRequiredError,
)
from passport.backend.api.views.bundle.mixins import UserMetaDataMixin
from passport.backend.api.views.bundle.mixins.kolmogor import KolmogorMixin
from passport.backend.api.views.bundle.mixins.mail import MailMixin
from passport.backend.api.views.bundle.mixins.push import BundlePushMixin
from passport.backend.core.conf import settings
from passport.backend.core.mailer.utils import (
    get_tld_by_country,
    login_shadower,
    MailInfo,
    make_email_context,
    send_mail_for_account,
)
from passport.backend.core.models.account import get_preferred_language
from passport.backend.core.models.persistent_track import (
    PersistentTrack,
    TRACK_TYPE_EMAIL_CONFIRMATION_CODE,
)
from passport.backend.core.runner.context_managers import UPDATE
from passport.backend.utils.common import generate_random_code
from passport.backend.core.types.email.email import mask_email_for_statbox
from passport.backend.utils.string import smart_str
from six.moves.urllib.parse import (
    urlencode,
    urlsplit,
    urlunsplit,
)

from .base import (
    BaseEmailBundleView,
    BASIC_GRANT,
)
from .exceptions import (
    EmailNotFoundError,
    EmailValidatorKeyCheckLimitExceededError,
)
from .forms import (
    CheckEmailOwnershipConfirmForm,
    ConfirmEmailForm,
    EmailForm,
    SendConfirmationEmailForm,
    SetupConfirmedEmailForm,
    UidForm,
)


CATEGORY_NATIVE = 'native'
CATEGORY_FOR_RESTORE = 'for_restore'
CATEGORY_FOR_NOTIFICATIONS = 'for_notifications'
CATEGORY_RPOP = 'rpop'
CATEGORY_OTHER = 'other'

EXTENDED_GRANT = 'email_bundle.manage'
CHECK_OWNERSHIP_GRANT = 'email_bundle.check_ownership'


log = logging.getLogger('passport.backend.api.views.bundle.email.controllers')


class BaseNewEmailBundleView(BaseEmailBundleView):
    required_grants = [BASIC_GRANT, EXTENDED_GRANT]
    required_scope = X_TOKEN_OAUTH_SCOPE

    def require_password_verification(self):
        if self.is_password_verification_required(uid=self.form_values['uid']):
            raise PasswordRequiredError()

    def process_request(self):
        if self.basic_form:
            self.process_basic_form()

        self.get_account_from_session_or_oauth_token(
            emails=True,
            email_attributes='all',
            multisession_uid=self.form_values.get('uid'),
            required_scope=self.required_scope,
        )
        self.statbox.bind_context(uid=self.account.uid)

        try:
            self.process()
        except PasswordRequiredError:
            redirect_state = self.check_have_password()
            if redirect_state:
                self.state = redirect_state
                return
            raise


class ListEmailsView(BaseNewEmailBundleView):
    basic_form = UidForm

    def get_category(self, email):
        if email.is_native:
            return CATEGORY_NATIVE
        elif email.is_silent:
            return CATEGORY_OTHER
        elif email.is_rpop:
            return CATEGORY_RPOP
        elif email.is_external and not email.is_unsafe:
            return CATEGORY_FOR_RESTORE
        else:
            return CATEGORY_FOR_NOTIFICATIONS

    def process(self):
        self.response_values.update(
            emails={
                email.address: {
                    # базовые параметры писем
                    'native': bool(email.is_native),
                    'default': bool(email.is_default),
                    'confirmed': bool(email.is_confirmed),
                    'rpop': bool(email.is_rpop),
                    'unsafe': bool(email.is_unsafe),
                    'silent': bool(email.is_silent),
                    # вычисляемые особенности
                    'category': self.get_category(email),
                }
                for email in self.account.emails.all
            },
        )


class DeleteEmailView(
    BaseNewEmailBundleView,
    BundlePushMixin,
    KolmogorMixin,
    MailMixin,
    UserMetaDataMixin,
):
    basic_form = EmailForm

    def process(self):
        email = self.account.emails.find(self.form_values['email'])
        if not email:
            raise EmailNotFoundError()
        elif email.is_native:
            raise EmailIsNativeError(email.address)

        with UPDATE(self.account, self.request.env, {'action': 'validator_delete'}):
            self.account.emails.pop(email.address)

        if email.is_confirmed:
            self.send_account_modification_mail(
                event_name='email_delete',
                specific_email=email.address,
            )
            self.send_account_modification_push(
                event_name='email_change',
            )
        self.statbox.log(
            action='delete',
            address=mask_email_for_statbox(email.address),
            status='ok',
        )


class SendConfirmationEmailView(BaseNewEmailBundleView):
    basic_form = SendConfirmationEmailForm
    require_track = True
    required_scope = [X_TOKEN_OAUTH_SCOPE, BIND_EMAIL_OAUTH_SCOPE]

    def construct_validation_url(self, validation_code, uid, retpath=None):
        scheme, netloc, path, query, fragment = urlsplit(self.form_values['validator_ui_url'])

        query = {
            'key': validation_code,
            'uid': uid,
        }
        if retpath:
            query['retpath'] = smart_str(retpath)

        return urlunsplit((scheme, netloc, path, urlencode(query), None))

    def send_validation_message(self, account, address, code,
                                short_code=None, tld=None,
                                language=None, retpath=None, code_only=False):
        language = get_preferred_language(account, language)
        translations = settings.translations.NOTIFICATIONS[language]

        user_tld = tld or get_tld_by_country(account.person.country)
        context_data = {
            'SHORT_CODE': short_code,
            'code_only': code_only,
        }
        if not code_only:
            url = self.construct_validation_url(
                code,
                uid=account.uid,
                retpath=retpath,
            )
            context_data.update({
                'VALIDATION_URL': url,
            })

        context = make_email_context(
            language=language,
            account=account,
            context=context_data,
        )

        send_mail_for_account(
            'mail/email_validation_link_message_new.html',
            MailInfo(
                subject=translations['emails.confirmation_email_sent.subject'],
                from_=translations['email_sender_display_name'],
                tld=user_tld,
            ),
            context,
            account,
            context_shadower=login_shadower,
            is_plain_text=False,
            send_to_native=False,
            send_to_external=False,
            specific_email=address,
        )

    def process(self):
        self.read_track()
        is_safe = self.form_values['is_safe']
        address = self.form_values['email']
        if is_safe:
            redirect_state = self.check_have_password()
            if redirect_state:
                self.state = redirect_state
                raise PasswordRequiredError()
        with UPDATE(self.account, self.request.env, {'action': 'validator_send'}):
            try:
                persistent_track = self.validate_by_email(
                    address=address,
                    language=self.form_values['language'],
                    retpath=self.form_values['retpath'],
                    tld=extract_tld(self.request.env.host, settings.PASSPORT_TLDS) or settings.PASSPORT_DEFAULT_TLD,
                    is_unsafe=not is_safe,
                    force=False,
                    code_only=self.form_values['code_only'],
                )
            except EmailGenericError:
                raise EmailAlreadyConfirmedError(
                    address=address,
                    uid=self.account.uid,
                    code=None,
                )

        with self.track_transaction.rollback_on_error() as track:
            track.persistent_track_id = persistent_track.track_id
            track.invalid_email_key_count.reset()

        self.statbox.log(
            action='send_confirmation_email',
            address=mask_email_for_statbox(address),
            status='ok',
            is_safe=is_safe,
        )
        self.response_values.update(track_id=self.track_id)


class BaseConfirmEmailView(
    BaseNewEmailBundleView,
    BundlePushMixin,
    KolmogorMixin,
    MailMixin,
    UserMetaDataMixin,
):
    basic_form = ConfirmEmailForm
    action = None
    required_scope = [X_TOKEN_OAUTH_SCOPE, BIND_EMAIL_OAUTH_SCOPE]

    def get_persistent_track(self, persistent_track_id):
        bb_response = self.blackbox.get_track(
            self.account.uid,
            persistent_track_id,
        )

        track = PersistentTrack().parse(bb_response)
        if not track.type == TRACK_TYPE_EMAIL_CONFIRMATION_CODE or not track.content:
            raise EmailIncorrectKeyError(self.account.uid, persistent_track_id)

        return track

    def confirm_email_by_persistent_track(self, persistent_track):
        email = self.account.emails.find(persistent_track.content.get('address'))
        if not email:
            raise EmailIncorrectKeyError(self.account.uid, persistent_track.track_id)

        if email.confirmed_at:
            raise EmailAlreadyConfirmedError(
                address=email.address,
                uid=self.account.uid,
                code=persistent_track.track_id,
            )

        with UPDATE(self.account, self.request.env, {'action': 'validator_confirm'}):
            email.confirmed_at = email.bound_at = datetime.now()
            self.account.emails.add(email)

        return email

    def check_key_and_confirm(self):
        raise NotImplementedError()  # pragma: no cover

    def process(self):
        email = self.check_key_and_confirm()
        self.send_account_modification_mail(
            event_name='email_add',
        )
        self.send_account_modification_push(
            event_name='email_change',
            context={'track_id': self.track_id} if self.track_id else None,
        )
        self.statbox.log(
            action=self.action,
            address=mask_email_for_statbox(email.address),
            authid=self.session_info.authid if self.session_info else None,
            track_id=self.track_id,
            user_agent=self.user_agent,
            ip=self.client_ip,
            status='ok',
        )


class ConfirmEmailByLinkView(BaseConfirmEmailView):
    action = 'confirm_by_link'

    def check_key_and_confirm(self):
        persistent_track = self.get_persistent_track(self.form_values['key'])
        return self.confirm_email_by_persistent_track(persistent_track)


class ConfirmEmailByCodeView(BaseConfirmEmailView):
    action = 'confirm_by_code'
    require_track = True

    def check_key_and_confirm(self):
        self.read_track()
        if not self.track.persistent_track_id:
            raise InvalidTrackStateError()

        limit = settings.ALLOWED_EMAIL_SHORT_CODE_FAILED_CHECK_COUNT
        attempts_left = limit - self.track.invalid_email_key_count.get(default=0)
        if attempts_left <= 0:
            raise EmailValidatorKeyCheckLimitExceededError()

        persistent_track = self.get_persistent_track(self.track.persistent_track_id)
        if persistent_track.content['short_code'] != self.form_values['key']:
            with self.track_transaction.rollback_on_error() as track:
                track.invalid_email_key_count.incr()
            self.response_values.update(
                attempts_left=max(0, attempts_left - 1),
            )
            raise EmailIncorrectKeyError(self.account.uid, self.form_values['key'])
        return self.confirm_email_by_persistent_track(persistent_track)


class SetupConfirmedEmailView(BaseNewEmailBundleView):
    basic_form = SetupConfirmedEmailForm

    def process(self):
        if self.form_values['is_safe']:
            self.require_password_verification()

        email = self.account.emails.find(self.form_values['email'])
        if not email or not email.is_confirmed:
            raise EmailNotFoundError()
        elif email.is_native:
            raise EmailIsNativeError(email.address)

        with UPDATE(self.account, self.request.env, {'action': 'validator_setup'}):
            if self.form_values['is_safe'] is not None:
                email.is_unsafe = not self.form_values['is_safe']

        self.statbox.log(
            action='setup_confirmed_email',
            address=mask_email_for_statbox(email.address),
            status='ok',
        )


class CheckEmailOwnershipSendCodeView(BaseEmailBundleView, KolmogorMixin, UserMetaDataMixin):
    required_grants = [BASIC_GRANT, CHECK_OWNERSHIP_GRANT]
    require_track = True

    def process(self):
        self.read_track()
        self.get_account_from_track(emails=True)

        counter_1h = self.build_counter(
            keyspace=settings.KOLMOGOR_KEY_SPACE_EMAIL_CHECK_OWNERSHIP_SENT_COUNTER_1H,
            name=settings.EMAIL_CHECK_OWNERSHIP_SENT_COUNTER_1H % self.account.uid,
            limit=settings.COUNTERS[settings.EMAIL_CHECK_OWNERSHIP_SENT_COUNTER_1H],
        )
        counter_24h = self.build_counter(
            keyspace=settings.KOLMOGOR_KEY_SPACE_EMAIL_CHECK_OWNERSHIP_SENT_COUNTER_24H,
            name=settings.EMAIL_CHECK_OWNERSHIP_SENT_COUNTER_24H % self.account.uid,
            limit=settings.COUNTERS[settings.EMAIL_CHECK_OWNERSHIP_SENT_COUNTER_24H],
        )
        self.failsafe_check_kolmogor_counters([counter_1h, counter_24h])

        code = generate_random_code(settings.EMAIL_CODE_CHALLENGE_CODE_LENGTH)
        with self.track_transaction.rollback_on_error() as track:
            track.email_check_ownership_code = code

        language = get_preferred_language(self.account)
        phrases = settings.translations.NOTIFICATIONS[language]

        context = make_email_context(
            language=language,
            account=self.account,
            context={
                'BROWSER': escape_percents(self.get_browser() or '?'),
                'login': self.account.login,
                'CODE': code,
                'greeting_key': 'greeting.noname',
            },
        )

        info = MailInfo(
            subject=phrases['emails.check_ownership_code.title'],
            from_=phrases['email_sender_display_name'],
            tld=get_tld_by_country(self.account.person.country),
        )

        send_mail_for_account(
            'mail/email_check_ownership_code.html',
            info,
            context,
            self.account,
            context_shadower=login_shadower,
            is_plain_text=False,
        )
        self.statbox.log(
            action='send_code',
            mode='email_ownership_confirmation',
            track_id=self.track_id,
            user_agent=self.user_agent,
            ip=self.client_ip,
            status='ok',
            uid=self.account.uid,
        )


class CheckEmailOwnershipConfirmView(BaseEmailBundleView):
    required_grants = [BASIC_GRANT, CHECK_OWNERSHIP_GRANT]
    require_track = True
    basic_form = CheckEmailOwnershipConfirmForm

    def process(self):
        self.process_basic_form()
        self.read_track()

        if not self.track.uid:
            log.debug('No uid in track')
            raise InvalidTrackStateError()

        if not self.track.email_check_ownership_code:
            log.debug('No email confirmation code in track')
            raise InvalidTrackStateError()

        if self.track.email_ownership_checks_count.get(0) >= settings.EMAIL_CHECK_OWNERSHIP_CODE_ATTEMPTS:
            log.debug('Email confirmation code attempts limit exceeded {} > {}'.format(
                self.track.email_ownership_checks_count.get(),
                settings.EMAIL_CHECK_OWNERSHIP_CODE_ATTEMPTS,
            ))
            raise EmailValidatorKeyCheckLimitExceededError()

        with self.track_transaction.rollback_on_error() as track:
            track.email_ownership_checks_count.incr()

        if self.track.email_check_ownership_code != self.form_values['code']:
            log.debug('Incorrect email confirmation code')
            raise EmailIncorrectKeyError(self.track.uid, self.track_id)

        with self.track_transaction.commit_on_error():
            self.track.email_check_ownership_passed = True
        self.statbox.log(
            action='check_code',
            mode='email_ownership_confirmation',
            track_id=self.track_id,
            user_agent=self.user_agent,
            ip=self.client_ip,
            status='ok',
            uid=self.track.uid,
        )
