# -*- coding: utf-8 -*-

from datetime import datetime

from passport.backend.api.email_validator.exceptions import (
    EmailAlreadySentError,
    EmailIsNativeError,
)
from passport.backend.api.views.bundle.exceptions import (
    EmailLimitPerProfileReachedError,
    InvalidTrackStateError,
)
from passport.backend.api.views.bundle.headers import HEADER_CLIENT_HOST
from passport.backend.api.views.bundle.mixins.common import BundleAdminActionMixin
from passport.backend.core.conf import settings
from passport.backend.core.runner.context_managers import UPDATE
from passport.backend.core.types.email.email import mask_email_for_statbox

from .base import (
    BaseEmailBundleView,
    BASIC_GRANT,
)
from .exceptions import EmailNotFoundError
from .forms import (
    DeleteEmailByAdminForm,
    SendConfirmationEmailFormV1,
    SetupConfirmedEmailFormV1,
)


CONFIRMATION_EMAIL_GRANT = 'email_bundle.confirmation_email'

MODIFY_CONFIRMED_GRANT = 'email_bundle.modify_confirmed'

CREATE_CONFIRMED_GRANT = 'email_bundle.create_confirmed'

DELETE_EMAIL_BY_ADMIN_GRANT = 'email_bundle.delete_by_admin'


class SendConfirmationEmailViewV1(BaseEmailBundleView):
    required_grants = [BASIC_GRANT, CONFIRMATION_EMAIL_GRANT]
    require_track = True
    basic_form = SendConfirmationEmailFormV1

    def get_uid(self):
        if self.track.uid is None:
            raise InvalidTrackStateError('No uid in track')
        return int(self.track.uid)

    def process(self):
        self.read_track()

        uid = self.get_uid()
        self.get_account_by_uid(
            uid,
            emails=True,
            email_attributes='all',
        )

        redirect_state = self.check_have_password()
        if redirect_state:
            self.state = redirect_state
            return

        email = self.form_values['email']
        email_retpath = self.form_values['email_retpath']
        language = self.form_values['language']

        self.statbox.bind(
            action='send_confirmation_email',
            uid=uid,
            email=mask_email_for_statbox(email),
            email_retpath=email_retpath,
        )
        events = {
            'action': 'send_confirmation_email',
        }

        try:
            with UPDATE(self.account, self.request.env, events):
                self.validate_by_email(
                    email,
                    language=language,
                    retpath=email_retpath,
                )
                self.statbox.bind(
                    status='ok',
                )
        except EmailAlreadySentError:
            self.statbox.bind(
                status='error',
                error='already_sent',
            )
            raise
        except EmailIsNativeError:
            self.statbox.bind(
                status='error',
                error='is_native',
            )
            raise
        finally:
            self.statbox.log()


class SetupConfirmedEmailViewV1(BaseEmailBundleView):
    basic_form = SetupConfirmedEmailFormV1

    required_grants = [
        BASIC_GRANT,
        CREATE_CONFIRMED_GRANT,
        MODIFY_CONFIRMED_GRANT,
    ]

    required_headers = [
        HEADER_CLIENT_HOST,
    ]

    def process(self):
        uid = self.form_values['uid']
        email = self.form_values['email']
        is_silent = self.form_values['is_silent']
        self.get_account_by_uid(
            uid,
            emails=True,
            email_attributes='all',
        )

        self.statbox.bind(
            action='setup_confirmed_email',
            uid=uid,
            email=mask_email_for_statbox(email),
            is_silent=is_silent,
        )

        events = {
            'action': 'setup_confirmed_email',
        }

        address = self.account.emails.find(email)
        self.check_email_not_native(email)

        if not address:
            if len(self.account.emails) >= settings.MAX_EMAILS_PER_PROFILE:
                raise EmailLimitPerProfileReachedError()
            address = self.create_confirmed_email(email)

        with UPDATE(self.account, self.request.env, events):
            address.is_unsafe = True
            if not address.is_confirmed:
                address.confirmed_at = address.bound_at = datetime.now()

            # Возможно в т.ч. снятие флага для уже существующего адреса
            if is_silent is not None:
                address.is_silent = is_silent
            self.account.emails.add(address)

        self.statbox.log(status='ok')


class DeleteEmailByAdminView(BaseEmailBundleView, BundleAdminActionMixin):
    basic_form = DeleteEmailByAdminForm

    required_grants = [
        BASIC_GRANT,
        DELETE_EMAIL_BY_ADMIN_GRANT,
    ]

    def process(self):
        uid = self.form_values['uid']
        email_address = self.form_values['email']
        self.get_account_by_uid(
            uid,
            emails=True,
            email_attributes='all',
        )

        email = self.account.emails.find(email_address)
        if not email:
            raise EmailNotFoundError()
        if email.is_native:
            raise EmailIsNativeError(email_address)

        events = {
            'action': 'delete_email_by_admin',
            'consumer': self.consumer,
        }
        self.mark_admin_action(events)

        with UPDATE(self.account, self.request.env, events):
            self.account.emails.pop(email.address)
