# -*- coding: utf-8 -*-

from passport.backend.api.exceptions import BaseApiError
from passport.backend.core.validators.utils import convert_formencode_invalid_to_error_list
from passport.backend.utils.string import smart_text
import six


@six.python_2_unicode_compatible
class BaseBundleError(BaseApiError):
    """Базовый класс исключения для всех ошибок в бандлах, которые будут
    сконвертированы в простой текстовый код ошибки"""
    error = None

    @property
    def errors(self):
        if not self.error:
            return []
        return [self.error]

    def __str__(self):
        result = u'errors: [%s]' % u', '.join(self.errors)
        args = self.args
        if args:
            result += u'; args: %s' % u', '.join(map(smart_text, args))
        return result

    def __repr__(self):
        return '<%s: %s>' % (self.__class__.__name__, str(self))


# Обобщенные ошибки
class BaseBundleInternalError(BaseBundleError):
    """Базовая ошибка для ошибок, отдаваемых приложениям"""
    error = 'internal.error'


class ResourceUnavailableError(BaseBundleError):
    """Временная ошибка от стороннего компонента"""
    error = 'backend.try_again'


class UnhandledBundleError(BaseBundleError):
    error = 'exception.unhandled'


# Конкретные ошибки
class AccountAlreadyCompletedError(BaseBundleError):
    """Пользователь уже полноценен"""
    error = 'account.already_completed'


class AccountAlreadyRegisteredError(BaseBundleError):
    """Трэк повторно используется для регистрации"""
    error = 'account.already_registered'


class AccountCompletionRequiredError(BaseBundleError):
    """
    Чтобы выполнить действие требуется дорегистрировать аккаунт.
    """
    error = 'account.required_completion'


class AccountCompromisedError(BaseBundleError):
    """Пользователь скомпрометирован - перенести в смену пароля"""
    error = 'account.compromised'


class AccountDeletionOperationNotFound(BaseBundleError):
    """
    Не найдена операция удаления аккаунта.
    """
    error = 'account_deletion_operation.not_found'


class AccountDisabledError(BaseBundleError):
    """Аккаунт заблокирован"""
    error = 'account.disabled'


class AccountDisabledOnDeletionError(BaseBundleError):
    """Аккаунт при удалении из-за наличия подписок на блокирующие сиды"""
    error = 'account.disabled_on_deletion'


class Account2FAEnabledError(BaseBundleError):
    """Аккаунт имеет включенный 2FA"""
    # TODO: Этот код ошибки диссонирует с именем атрибута из ЧЯ 'password.2fa_on'
    error = 'account.2fa_enabled'


class AccountSms2FAEnabledError(BaseBundleError):
    """Аккаунт имеет включенный sms-2FA"""
    error = 'account.sms_2fa_enabled'


class AccountGlobalLogoutError(BaseBundleError):
    """Был произведен глобальный выход на аккаунте"""
    error = 'account.global_logout'


class AccountHasBlockingSIDs(BaseBundleError):
    """У аккаунта есть подписки на сервисы, которые блокируют его удаление."""
    error = 'account.has_blocking_sids'


class AccountInvalidTypeError(BaseBundleError):
    """Аккаунт не поддерживает выполнение операции"""
    error = 'account.invalid_type'


class AccountNoAnswerError(BaseBundleError):
    """Ответ не задан в аккаунте"""
    error = 'account.no_answer'


class AccountNoQuestionError(BaseBundleError):
    """Вопрос не задан в аккаунте"""
    error = 'account.no_question'


class AccountNotFoundError(BaseBundleError):
    """Аккаунт не найден/не существует"""
    error = 'account.not_found'


class AccountNotSubscribedError(BaseBundleError):
    """Нет нужного сида на аккаунте"""
    error = 'account.not_subscribed'


class AccountRequiredChangePasswordError(BaseBundleError):
    """Аккаунту необходимо сменить пароль (причина смены не важна)"""
    error = 'account.required_change_password'


class AccountIsChildError(BaseBundleError):
    """Аккаунт является ребёнком, текущее действие ему не разрешено"""
    error = 'account.is_child'


class SecondStepRequired(UnhandledBundleError):
    """
    Данное действие невозможно без прохождения второго шага.
    Ошибка отдаётся с кодом second_step.not_implemented, так как до потребителя она доходить не должна,
    а должна перехватываться и обрабатываться.
    """
    error = 'second_step.not_implemented'

    def __init__(self, allowed_second_steps=None, *args):
        super(SecondStepRequired, self).__init__(*args)
        self.allowed_second_steps = allowed_second_steps or []


class AccountStrongPasswordPolicyError(BaseBundleError):
    """Аккаунту запрещено авторизовываться соц. профилем из-за политики сложного пароля"""
    error = 'account.strong_password_policy_enabled'


class AccountUidMismatchError(BaseBundleError):
    """UID аккаунта, полученного по credentials, не совпадает с переданным"""
    error = 'account.uid_mismatch'

    def __init__(self, *args, **kwargs):
        super(AccountUidMismatchError, self).__init__(*args)
        self.known_uids = kwargs.get('known_uids')


class AccountWithoutPasswordError(BaseBundleError):
    """Не установлен обязательный пароль на выбранном аккаунте"""
    error = 'account.without_password'


class AccountWithoutLoginError(BaseBundleError):
    """Не установлен обязательный логин на выбранном аккаунте"""
    error = 'account.without_login'


class ActionImpossibleError(BaseBundleError):
    """Требуемая операция не может быть выполнена по тем или иным причинам"""
    error = 'action.impossible'


class ActionNotRequiredError(BaseBundleError):
    """Запрашиваемое действие уже было выполнено"""
    error = 'action.not_required'


class AfishaApiUnavailableError(ResourceUnavailableError):
    """Временная ошибка от апи Афиши"""
    error = 'backend.afisha_api_failed'


class AntifraudApiUnavailableError(ResourceUnavailableError):
    """Временная ошибка от апи Антифрода"""
    error = 'backend.antifraud_api_failed'


class AvatarsMdsApiUnavailableError(ResourceUnavailableError):
    """Временная ошибка от MDS"""
    error = 'backend.avatars_mds_api_failed'


class AnswerEmptyError(BaseBundleError):
    """Не передан ответ на контрольный вопрос"""
    error = 'answer.empty'


class AuthorizationHeaderError(BaseBundleError):
    """Невалидный заголовок авторизации"""
    error = 'authorization.invalid'


class BiletApiUnavailableError(ResourceUnavailableError):
    """Временная ошибка от Билет апи"""
    error = 'backend.bilet_api_failed'


class BillingUnavailableError(ResourceUnavailableError):
    """Временная ошибка от Биллинга"""
    error = 'backend.billing_failed'


class TrustBindingsUnavailableError(ResourceUnavailableError):
    """Временная ошибка от апи Траста"""
    error = 'backend.trust_bindings_failed'


class TrustBindingsUnavailablePermanentError(ResourceUnavailableError):
    """Постоянная ошибка от апи Траста"""
    error = 'backend.trust_api_permanent_error'


class BlackboxPermanentError(ResourceUnavailableError):
    """Постоянная ошибка от ЧЯ"""
    error = 'backend.blackbox_permanent_error'


class BotApiRequestFailed(BaseBundleError):
    """У пользователя нет ПП"""
    error = 'bot_api.request_failed'


class BotApiUnavailableError(ResourceUnavailableError):
    """Временная ошибка от BotApi"""
    error = 'backend.bot_api_failed'


class BlackboxUnavailableError(ResourceUnavailableError):
    """Временная ошибка от ЧЯ"""
    error = 'backend.blackbox_failed'


class CaptchaUnavailableError(ResourceUnavailableError):
    """Временная ошибка от капчи"""
    error = 'backend.captcha_failed'


class CaptchaRequiredAndCompareNotMatchedError(BaseBundleError):
    """Объединённая ошибка для случая, когда необходимо
    показать пользователю капчу и когда он ввёл что-то не так
    """

    @property
    def errors(self):
        return [CaptchaRequiredError.error, CompareNotMatchedError.error]


class CaptchaRequiredAndPasswordNotMatchedError(BaseBundleError):
    """Объединённая ошибка для случая, когда необходимо
    показать пользователю капчу и когда он ввёл неправильно пароль
    """

    @property
    def errors(self):
        return [CaptchaRequiredError.error, PasswordNotMatchedError.error]


class CaptchaRequiredError(BaseBundleError):
    """Требуется верификация с помощью капчи"""
    error = 'captcha.required'


class CaptchaNotShownError(BaseBundleError):
    """Пришли проверять капчу, не показав юзеру картинку"""
    error = 'captcha.not_shown'


class CodeInvalidError(BaseBundleError):
    """Указанный код подтверждения телефона не совпадает с отправленным"""
    error = 'code.invalid'


class CollectionsApiUnavailableError(ResourceUnavailableError):
    """Временная ошибка от апи Коллекций"""
    error = 'backend.collections_api_failed'


class CompareNotMatchedError(BaseBundleError):
    """При выполнении сравнения данных, результат оказался отрицательным"""
    error = 'compare.not_matched'


class DatabaseUnavailableError(ResourceUnavailableError):
    """Ошибка от базы данных"""
    error = 'backend.database_failed'


class DatasyncApiUnavailableError(ResourceUnavailableError):
    """Временная ошибка от апи Датасинк"""
    error = 'backend.datasync_api_failed'


class DomainAlreadyExistsError(BaseBundleError):
    """Создаваемый ПДД-домен уже существует"""
    error = 'domain.already_exists'


class DomainInvalidTypeError(BaseBundleError):
    """Домен не подходит для выбранной операции"""
    error = 'domain.invalid_type'


class DomainMissingError(BaseBundleError):
    """Домен пуст"""
    error = 'domain.missing'


class DomainNotFoundError(BaseBundleError):
    """Домен не зарегистрирован"""
    error = 'domain.not_found'


class EmailConfirmationsLimitExceededError(BaseBundleError):
    """Подтверждение email-адреса невозможно, превышен лимит."""
    error = 'email_confirmations_limit.exceeded'


class EmailSendLimitExceededError(BaseBundleError):
    """Отправка сообщения на email-адрес невозможна, превышен лимит."""
    error = 'email.send_limit_exceeded'


class EulaIsNotAcceptedError(BaseBundleError):
    """Пользовательское соглашение не принято"""
    error = 'eula_accepted.not_accepted'


class FederalConfigsApiUnavailableError(BaseBundleError):
    """Временная ошибка от FederalConfigs Api"""
    error = 'backend.federal_configs_api_failed'


class GeoSearchApiUnavailableError(BaseBundleError):
    """Временная ошибка от GeoSearch Api"""
    error = 'backend.geosearch_api_failed'


class HeadersEmptyError(BaseBundleError):
    """Отсутствуют необходимые заголовки запроса"""

    def __init__(self, aliases, *args, **kwargs):
        # Список коротких алиасов
        self.aliases = aliases
        super(HeadersEmptyError, self).__init__(*args, **kwargs)

    @property
    def errors(self):
        return [('%s.empty' % a) for a in self.aliases]


class HistoryDBApiUnavailableError(ResourceUnavailableError):
    """Временная ошибка от historydb-api"""
    error = 'backend.historydb_api_failed'


class HistoryDBPermanentError(ResourceUnavailableError):
    """Постоянная ошибка от historydb-api"""
    error = 'backend.historydb_api_permanent_error'


class HuskyApiPermanentError(ResourceUnavailableError):
    """Не похожая на временную ошибка от husky api"""
    error = 'backend.husky_permanent_error'


class HuskyApiUnavailableError(ResourceUnavailableError):
    """Временная ошибка от husky api"""
    error = 'backend.husky_failed'


class IncompleteTrackError(UnhandledBundleError):
    """В треке не хватает полей.
    Ошибку не обрабатываем, чтобы не маскировать ошибки в коде.
    Отдаётся с кодом exception.unhandled
    """


class InternalPermanentError(BaseBundleInternalError):
    """
    Внутренняя ошибка, отдаваемая приложению.
    Повторные запросы не имеют смысла
    """
    error = 'internal.permanent'


class InternalTemporaryError(BaseBundleInternalError):
    """
    Внутренняя ошибка отдаваемая приложению,
    при падении какого-то нашего бекенда(ЧЯ, Редиса и т.д.)
    Имеет смысл повторить запрос
    """
    error = 'internal.temporary'


class InvalidCSRFTokenError(BaseBundleError):
    """Передан csrf_token, не совпадающий с csrf_token в треке"""
    error = 'csrf_token.invalid'


class InvalidHostError(BaseBundleError):
    """Запросы с данного хоста запрещены"""
    error = 'host.invalid'


class InvalidTrackStateError(BaseBundleError):
    """
    Пользователь пришел в ручку с неверным треком:
    не прошел необходимый шаг или повторно с "отработанным" треком в ту же
    """
    error = 'track.invalid_state'


class LdapUnavailableError(ResourceUnavailableError):
    """Временная ошибка LDAP"""
    error = 'backend.ldap_failed'


class LoginLikePasswordError(BaseBundleError):
    """Если lite-аккаунт выбрал себе логин такой же, как его текуший пароль"""
    error = 'login.like_password'


class LoginNotavailableError(BaseBundleError):
    """
    Выбранный логин уже используется другим пользователем или имеет
    недопустимую форму.
    """
    error = 'login.notavailable'


class MagicLinkExpiredError(BaseBundleError):
    """Волшебная ссылка протухла"""
    error = 'magic_link.expired'


class MagicLinkNotSentError(BaseBundleError):
    """Волшебная ссылка не была отправлена"""
    error = 'magic_link.not_sent'


class MagicLinkInvalidatedError(BaseBundleError):
    """Волшебная ссылка была инвалидирована пользователем"""
    error = 'magic_link.invalidated'


class MagicLinkSecretNotMatch(BaseBundleError):
    """Секрет из ссылки не совпадает с сохраненным в треке"""
    error = 'magic_link.secret_not_matched'


class MarketApiUnavailableError(ResourceUnavailableError):
    """Временная ошибка от контентного апи Маркета"""
    error = 'backend.market_api_failed'


class MessengerApiUnavailableError(ResourceUnavailableError):
    """Временная ошибка от апи Мессенджера"""
    error = 'backend.messenger_api_failed'


class MusicApiUnavailableError(ResourceUnavailableError):
    """Временная ошибка от апи музыки"""
    error = 'backend.music_api_failed'


class MusicApiPermanentError(BaseBundleError):
    """Постоянная ошибка в ответе от апи музыки"""
    error = 'backend.music_api_permanent_error'


class OAuthTokenValidationError(BaseBundleError):
    """Невалидный токен"""
    error = 'oauth_token.invalid'


class OAuthInvalidScopeError(BaseBundleError):
    """Нет нужного scope"""
    error = 'oauth.invalid_scope'


class OAuthInvalidClientIdError(BaseBundleError):
    """Недействительный OAuth client id"""
    error = 'oauth_client_id.invalid'


class OAuthInvalidClientSecretError(BaseBundleError):
    """Недействительный OAuth client secret"""
    error = 'oauth_client_secret.invalid'


class OAuthDeviceCodeNotFound(BaseBundleError):
    """Недействительный device_code"""
    error = 'oauth_code.not_found'


class OAuthUnavailableError(ResourceUnavailableError):
    """Ошибка от сервера OAuth"""
    error = 'backend.oauth_failed'


class OAuthTokenDeviceNotTrustedError(BaseBundleError):
    """Oauth токен выдан не доверенному устройству. Требуется доверенное"""
    error = 'oauth_token.device_not_trusted'


class OtpNotMatchedError(BaseBundleError):
    """Введенный otp неверен"""
    error = 'otp.not_matched'


class PasswordChangeForbiddenError(BaseBundleError):
    """не удаётся сменить пароль (слишком частые попытки или неподходящий новый пароль)"""
    error = 'password.change_forbidden'


class PasswordNotMatchedError(BaseBundleError):
    """Указанный пароль не совпадает с паролем аккаунта"""
    error = 'password.not_matched'


class PasswordRequiredError(BaseBundleError):
    """Требуется ввод пароля пользователем"""
    error = 'password.required'


class PerimeterApiPermanentError(BaseBundleError):
    """Постоянная ошибка в ответе от perimeter-api"""
    error = 'backend.perimeter_api_permanent_error'


class PerimeterApiUnavailableError(ResourceUnavailableError):
    """Временная ошибка perimeter-api"""
    error = 'backend.perimeter_api_failed'


class PhoneCompromisedError(BaseBundleError):
    """Телефон скомпрометирован - перенести в смену пароля"""
    error = 'phone.compromised'


class PhoneIsBankPhoneNumberAliasError(BaseBundleError):
    """Телефон - банковский алиас, нельзя удалить/отвязать"""
    error = 'phone.is_bank_phonenumber_alias'


class PhoneSquatterUnavailableError(BaseBundleError):
    """Ошибка на стороне Phone Squatter"""
    error = 'backend.phone_squatter_failed'


class PhoneVerificationRequiredError(BaseBundleError):
    """Требуется проверка по sms"""
    error = 'phone.required'


class PushApiUnavailableError(BaseBundleError):
    """Ошибка на стороне Push API"""
    error = 'backend.push_api_failed'


class QuestionEmptyError(BaseBundleError):
    """Вопрос не задан (при валидации формы)"""
    error = 'question.empty'


class QuestionInconsistentError(BaseBundleError):
    """Заданы и question, и question_id"""
    error = 'question.inconsistent'


class RateLimitExceedError(BaseBundleError):
    """Событие происходит слишком часто"""
    error = 'rate.limit_exceeded'


class RedisUnavailableError(ResourceUnavailableError):
    """Ошибка от хранилища треков Redis"""
    error = 'backend.redis_failed'


class RequestCredentialsSeveralPresentError(BaseBundleError):
    """В запросе передано несколько credentials для получения аккаунта"""
    error = 'request.credentials_several_present'


class RequestCredentialsAllMissingError(BaseBundleError):
    """В запросе не передано ни одного credentials для получения аккаунта"""
    error = 'request.credentials_all_missing'


class SecretKeyInvalidError(BaseBundleError):
    """Ошибка для невалидного ключа (конкретная причина не раскрывается)"""
    error = 'secret_key.invalid'


class SecurePhoneNotFoundError(BaseBundleError):
    """У аккаунта нет привязанного подтвержденного secure номера телефона"""
    error = 'phone_secure.not_found'


class SessguardInvalidError(BaseBundleError):
    """Значение sessguard не прошло проверку в ЧЯ"""
    error = 'sessguard.invalid'


class SessionidInvalidError(BaseBundleError):
    """Значение Session_id не прошло проверку в ЧЯ"""
    error = 'sessionid.invalid'


class SocialApiUnavailableError(ResourceUnavailableError):
    """Временная ошибка от SocialApi"""
    error = 'backend.social_api_failed'


class SocialApiPermanentError(ResourceUnavailableError):
    """Постоянная ошибка от SocialApi"""
    error = 'backend.social_api_permanent_error'


class SocialBrokerUnavailableError(ResourceUnavailableError):
    """Временная ошибка от SocialBroker"""
    error = 'backend.social_broker_failed'


class SocialBrokerPermanentError(ResourceUnavailableError):
    """Постоянная ошибка от SocialBroker"""
    error = 'backend.social_broker_permanent_error'


class SubscriptionBlockedError(ActionImpossibleError):
    """
    При попытке отписаться от сервиса определили, что это сделать невозможно.
    Либо подписка Блокирующая, либо Защищенная, либо пользователь заблокирован на этом сервисе
    """
    error = 'subscription.blocked'


class SubscriptionForbiddenError(ActionImpossibleError):
    """
    Нельзя подписаться на данный сервис
    """
    error = 'subscription.forbidden'


class TaskNotFoundError(BaseBundleError):
    """
    Передан идентификатор несуществующего социального task.
    """
    error = 'task.not_found'


class TooFrequentPasswordChangeError(BaseBundleError):
    """Слишком частые попытки смены пароля для этого аккаунта"""
    error = 'password.too_frequent_change'


class TrackNotFoundError(BaseBundleError):
    """Трек с указанным идентификатором не найден"""
    error = 'track.not_found'


class TvmUserTicketInvalidError(BaseBundleError):
    error = 'tvm_user_ticket.invalid'


class TvmUserTicketMissingScopes(TvmUserTicketInvalidError):
    error = 'tvm_user_ticket.missing_scopes'

    def __init__(self, ticket_scopes, missing_scopes):
        super(TvmUserTicketMissingScopes, self).__init__(ticket_scopes, missing_scopes)
        self.ticket_scopes = ticket_scopes
        self.missing_scopes = missing_scopes


class TvmUserTicketNoUid(TvmUserTicketInvalidError):
    error = 'tvm_user_ticket.no_uid'

    def __init__(self, known_uids):
        super(TvmUserTicketNoUid, self).__init__(known_uids)
        self.known_uids = known_uids


class UidNotInSessionError(BaseBundleError):
    """Уид не присутствует в сессии"""
    error = 'sessionid.no_uid'

    def __init__(self, *args, **kwargs):
        super(UidNotInSessionError, self).__init__(*args)
        self.known_uids = kwargs.get('known_uids', [])


class UserNotVerifiedError(BaseBundleError):
    """Пользователь не прошел проверку ни через телефон, ни через капчу"""
    error = 'user.not_verified'


class ValidationFailedError(BaseBundleError):
    """Ошибки в форме валидации"""

    def __init__(self, errors, invalid_exception=None, *args, **kwargs):
        self._errors = errors
        self._invalid_exception = invalid_exception
        super(ValidationFailedError, self).__init__(*args, **kwargs)

    @staticmethod
    def from_invalid(invalid_exception, field=None):
        return ValidationFailedError(
            convert_formencode_invalid_to_error_list(invalid_exception, field=field),
            invalid_exception=invalid_exception,
        )

    @property
    def errors(self):
        return self._errors


class VideoApiUnavailableError(ResourceUnavailableError):
    """Временная ошибка от апи видеопоиска"""
    error = 'backend.video_api_failed'


class YaKeyBackupNotFoundError(BaseBundleError):
    """Бэкапа для данного номера нет"""
    error = 'yakey_backup.not_found'


class YaKeyBackupAlreadyExistsError(BaseBundleError):
    """Бэкап уже существует"""
    error = 'yakey_backup.exists'


class YapicUnavailableError(ResourceUnavailableError):
    """Временная ошибка от Япика или MDS"""
    error = 'backend.yapic_failed'


class YaSmsUnavailableError(ResourceUnavailableError):
    """Временная ошибка от Я.смс"""
    error = 'backend.yasms_failed'


class YdbUnavailableError(ResourceUnavailableError):
    """Временная ошибка от Ydb"""
    error = 'backend.ydb_failed'


class YdbPermanentError(ResourceUnavailableError):
    """Постоянная ошибка от Ydb"""
    error = 'backend.ydb_permanent_error'


class OctopusUnavailableError(ResourceUnavailableError):
    """Временная ошибка от Octopus"""
    error = 'backend.octopus_failed'


class DriveApiUnavailableError(ResourceUnavailableError):
    """Временная ошибка от Drive Api"""
    error = 'backend.drive_api_failed'


class CallsShutDownError(BaseBundleError):
    """Звонки отключены"""
    error = 'calls.shut_down'


class CreateCallFailed(BaseBundleError):
    """Не удалось создать сессию звонка"""
    error = 'create_call.failed'


class FamilyAlreadyExists(ActionImpossibleError):
    """ Пользователь уже состоит в семье """
    error = 'family.already_exists'


class FamilyDoesNotExist(ActionImpossibleError):
    """ Пользователь не состоит в семье """
    error = 'family.does_not_exist'


class FamilyNotIsAdmin(ActionImpossibleError):
    """ Пользователь не является администратором семьи """
    error = 'family.not_is_admin'


class FamilyIsAdmin(ActionImpossibleError):
    """ Пользователь является администратором семьи """
    error = 'family.is_admin'


class FamilyAlreadyIsMemberThis(ActionImpossibleError):
    """ Пользователь является членом этой семьи """
    error = 'family.is_member_this'


class FamilyAlreadyIsMemberOther(ActionImpossibleError):
    """ Пользователь является членом другой семьи """
    error = 'family.is_member_other'


class FamilyIsNotAMember(ActionImpossibleError):
    """ Пользователь не является членом семьи """
    error = 'family.is_not_a_member'


class FamilyInvalidInvite(ActionImpossibleError):
    """ Такого инвайта в семью не существует """
    error = 'family.invalid_invite'


class FamilyInvalidPlace(ActionImpossibleError):
    """ Нет членов семьи с таким place """
    error = 'family.invalid_place'


class FamilyInvalidFamily(ActionImpossibleError):
    """ В запросе передан неверный family_id """
    error = 'family.invalid_family'


class FamilyMaxCapacity(ActionImpossibleError):
    """ Такого инвайта в семью не существует """
    error = 'family.max_capacity'


class LogbrokerTemporaryError(BaseBundleError):
    error = 'backend.logbroker_failed'


class FamilyNotAllowedToManageKiddish(ActionImpossibleError):
    """
    У аккаунта нет полномочий для управления ребёнкишами
    """
    error = 'family.not_allowed_to_manage_kiddish'


class PublicIdUpdateNotAllowed(BaseBundleError):
    """Пользователю не разрешено больше менять public_id"""
    error = 'public_id.update_not_allowed'


class DisplayNameUpdateNotAllowed(BaseBundleError):
    """Пользователю не разрешено менять display_name"""
    error = 'display_name.update_not_allowed'


class EmailLimitPerProfileReachedError(BaseBundleError):
    """Количество email-ов в профиле пользователя достигло предела. Заводить новые нельзя"""
    error = 'email.limit_per_profile_reached'


class EmailExistError(BaseBundleError):
    """Указанная почта занята другим пользователем"""
    error = 'email.exist'


class EmailUnsupportableDomainError(BaseBundleError):
    """Почта пользователя на другом домене"""
    error = 'email.unsupportable_domain'


class EmailInvalidError(BaseBundleError):
    """Почта пользователя содержит запрещенные символы или признана некорректной по другой причине"""
    error = 'email.invalid'


__all__ = [
    cls.__name__
    for cls in locals().values()
    if isinstance(cls, type) and issubclass(cls, BaseBundleError)
]
