# -*- coding: utf-8 -*-
from datetime import datetime
import logging

from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.exceptions import (
    FamilyDoesNotExist,
    FamilyInvalidInvite,
)
from passport.backend.api.views.bundle.family.forms import FamilyInviteForm
from passport.backend.api.views.bundle.headers import HEADER_CONSUMER_CLIENT_IP
from passport.backend.api.views.bundle.mixins import BundleAccountGetterMixin
from passport.backend.api.views.bundle.mixins.family import BundleFamilyMixin
from passport.backend.core.logging_utils.loggers import StatboxLogger
from passport.backend.core.models.family import FamilyInvite
from passport.backend.core.runner.context_managers import UPDATE
from passport.backend.core.utils.decorators import cached_property


log = logging.getLogger('passport.api.view.bundle.family.accept_invite')


class AcceptInviteView(BaseBundleView,
                       BundleAccountGetterMixin,
                       BundleFamilyMixin):

    basic_form = FamilyInviteForm

    required_headers = (
        HEADER_CONSUMER_CLIENT_IP,
    )

    required_grants = ['family.accept_invite']
    by_uid_grant = 'family.accept_invite_by_uid'

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode='family',
            consumer=self.consumer,
            ip=self.client_ip,
            user_agent=self.user_agent,
        )

    def process_request(self):
        self.process_basic_form()

        # Найти инвайт
        invite = self.find_invite(self.form_values['invite_id'])
        if invite is None:
            message = 'Invite %s not found' % self.form_values['invite_id']
            log.debug(message)
            raise FamilyInvalidInvite(message)

        # Загрузить аккаунт
        self.get_account_from_uid_or_session(
            by_uid_grant=self.by_uid_grant,
            multisession_uid=self.form_values['multisession_uid'],
            enabled_required=True,
            get_family_info=True,
        )

        # Проверить, что пользователь не состоит в семье
        self.assert_has_no_family(invite.family_id)

        # Загрузить семью инвайта
        try:
            self.load_family_info_by_family_id(invite.family_id)
        except FamilyDoesNotExist as err:
            log.debug('Replacing FamilyDoesNotExist with FamilyInvalidInvite')
            raise FamilyInvalidInvite(str(err))

        # Проверить что в семье есть места
        self.assert_family_has_empty_slots(invite.invite_id)

        # Удалить инвайт
        self.delete_invite(invite.invite_id)

        # Добавить аккаунт в семью
        place = self.get_family_member_free_place()
        with UPDATE(
            self.family_info,
            self.request.env,
            events=dict(
                action='family_accept_invite',
                consumer=self.consumer,
                invite_id=invite.invite_id,
                family_id=invite.family_id,
                send_method=FamilyInvite.send_method_to_text(invite.send_method),
                contact=invite.contact,
            ),
            datetime_=datetime.now(),
        ):
            self.family_info.add_member_uid(self.account.uid, place)

        # Записать о принятии инвайта в statbox
        self.statbox.log(
            action='family_accept_invite',
            uid=str(self.account.uid),
            **self.invite_to_statbox(invite)
        )

        log.debug('Added uid=%s place=%s to family_id=%s with invite=%s' % (
            self.account.uid,
            place,
            self.family_info.family_id,
            invite.invite_id,
        ))
