# -*- coding: utf-8 -*-
import logging

from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.exceptions import FamilyAlreadyIsMemberThis
from passport.backend.api.views.bundle.family.forms import FamilyMemberUidForm
from passport.backend.api.views.bundle.headers import HEADER_CONSUMER_CLIENT_IP
from passport.backend.api.views.bundle.mixins import BundleAccountGetterMixin
from passport.backend.api.views.bundle.mixins.family import BundleFamilyMixin
from passport.backend.core.runner.context_managers import UPDATE


log = logging.getLogger('passport.api.view.bundle.family.add_member')


class AddMemberView(BaseBundleView,
                    BundleAccountGetterMixin,
                    BundleFamilyMixin):
    basic_form = FamilyMemberUidForm

    required_headers = (
        HEADER_CONSUMER_CLIENT_IP,
    )

    required_grants = ['family.add_member']
    by_uid_grant = 'family.add_member_by_uid'

    def process_request(self):
        self.process_basic_form()

        # Загрузить аккаунт
        self.get_account_from_uid_or_session(
            by_uid_grant=self.by_uid_grant,
            multisession_uid=self.form_values['multisession_uid'],
            enabled_required=True,
            get_family_info=True,
        )

        # Проверить, что пользователь - админ семьи
        self.assert_is_family_admin()

        # Проверить, что добавляемый uid не принадлежит текущему пользователю
        if str(self.account.uid) == str(self.form_values['member_uid']):
            message = (
                'User being added is current user and an admin of a family %s'
                % self.account.family_info.family_id
            )
            log.debug(message)
            raise FamilyAlreadyIsMemberThis(message)

        # Получить объект добавляемого аккаунта по user_id
        current_family_id = self.account.family_info.family_id
        self.get_account_by_uid(
            uid=self.form_values['member_uid'],
            get_family_info=True,
        )

        # Проверить, что добавляемый аккаунт не состоит в семье
        self.assert_has_no_family(current_family_id)

        # Загрузить информацию о семье запрошенного аккаунта
        # TODO в принципе, тут можно создать фиктивный объект семьи,
        #  и добавлять в него, не ходя в ЧЯ, но это костыль
        self.load_family_info_by_family_id(current_family_id)
        if self.account.uid in self.family_info:
            message = 'User being added became a member of a family %s ' \
                      'during request' % self.family_info.family_id
            raise FamilyAlreadyIsMemberThis(message)

        # Добавить аккаунт в семью
        place = self.get_family_member_free_place()
        events = {
            'action': 'family_add_member',
            'consumer': self.consumer,
        }
        with UPDATE(self.family_info, self.request.env, events):
            self.family_info.add_member_uid(self.account.uid, place)

        log.debug('Added uid=%s place=%s to family family_id=%s' % (
            self.account.uid,
            place,
            current_family_id,
        ))

        # Вернуть family id
        self.response_values.update(
            family_id=current_family_id,
        )
