# -*- coding: utf-8 -*-

import logging

from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.constants import (
    AUTHENTICATION_MEDIA_SESSION,
    AUTHENTICATION_MEDIA_UID,
    AUTHENTICATION_MEDIA_USER_TICKET,
    FAMILY_MANAGE_SCOPE,
    SESSIONID_SCOPE,
)
from passport.backend.api.views.bundle.exceptions import AccountInvalidTypeError
from passport.backend.api.views.bundle.family.forms import ChangeChildAvatarForm
from passport.backend.api.views.bundle.mixins import (
    BundleAccountGetterMixin,
    BundleAvatarMixin,
    BundleChildMixin,
    BundleFamilyMixin,
    BundleTvmUserTicketMixin,
)
from passport.backend.core.logging_utils.loggers import StatboxLogger
from passport.backend.core.utils.decorators import cached_property


log = logging.getLogger(__name__)


class ChangeChildAvatarView(
    BundleChildMixin,
    BundleFamilyMixin,
    BundleAccountGetterMixin,
    BundleTvmUserTicketMixin,
    BundleAvatarMixin,
    BaseBundleView,
):
    basic_form = ChangeChildAvatarForm
    required_grants = ['family.change_child_avatar']
    by_uid_grant = 'family.change_child_avatar_by_uid'

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode='family',
            consumer=self.consumer,
            ip=self.client_ip,
            user_agent=self.user_agent,
        )

    def check_prerequires(self):
        self.get_account_from_available_media(
            by_uid_grant=self.by_uid_grant,
            enabled_media=(
                AUTHENTICATION_MEDIA_SESSION,
                AUTHENTICATION_MEDIA_UID,
                AUTHENTICATION_MEDIA_USER_TICKET,
            ),
            get_family_info=True,
            multisession_uid=self.form_values.get('multisession_uid'),
            required_scope=(FAMILY_MANAGE_SCOPE, SESSIONID_SCOPE),
        )
        # является ли этот персонаж админом, значит может управлять семьей в ней
        self.assert_is_family_admin(load_family_info=True)
        # и сам не является ребенком
        if self.account.is_child:
            raise AccountInvalidTypeError()

        self.get_account_by_uid(
            self.form_values.get('child_uid'),
            enabled_required=False,
            get_family_info=True,
        )
        self.check_child_consistent(self.account)

    def process_request(self):
        self.process_basic_form()

        self.check_prerequires()

        avatar_url = self.change_child_avatar(
            url=self.form_values['url'],
            file_obj=self.form_values['file'],
            is_temp=self.form_values['is_temp'],
            avatar_size=self.form_values['avatar_size'],
        )
        self.response_values.update(avatar_url=avatar_url)
