# -*- coding: utf-8 -*-

import logging

from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.constants import (
    AUTHENTICATION_MEDIA_SESSION,
    AUTHENTICATION_MEDIA_UID,
    AUTHENTICATION_MEDIA_USER_TICKET,
    FAMILY_MANAGE_SCOPE,
    SESSIONID_SCOPE,
)
from passport.backend.api.views.bundle.exceptions import (
    AccountInvalidTypeError,
    InternalTemporaryError,
)
from passport.backend.api.views.bundle.family.forms import ChangeKiddishForm
from passport.backend.api.views.bundle.mixins.account import (
    BundleAccountGetterMixin,
    BundleAccountResponseRendererMixin,
)
from passport.backend.api.views.bundle.mixins.common import BundleTvmUserTicketMixin
from passport.backend.api.views.bundle.mixins.family import BundleFamilyMixin
from passport.backend.api.views.bundle.mixins.kiddish import BundleKiddishMixin
from passport.backend.core import Undefined
from passport.backend.core.runner.context_managers import UPDATE


log = logging.getLogger(__name__)


class ChangeKiddishView(
    BundleKiddishMixin,
    BundleFamilyMixin,
    BundleAccountResponseRendererMixin,
    BundleAccountGetterMixin,
    BundleTvmUserTicketMixin,
    BaseBundleView,
):
    basic_form = ChangeKiddishForm
    required_grants = ['family.change_kiddish']
    by_uid_grant = 'family.change_kiddish_by_uid'

    def process_request(self):
        self.process_basic_form()

        self.get_account_from_available_media(
            by_uid_grant=self.by_uid_grant,
            enabled_media=(
                AUTHENTICATION_MEDIA_SESSION,
                AUTHENTICATION_MEDIA_UID,
                AUTHENTICATION_MEDIA_USER_TICKET,
            ),
            get_family_info=True,
            multisession_uid=self.form_values.get('multisession_uid'),
            required_scope=(FAMILY_MANAGE_SCOPE, SESSIONID_SCOPE),
        )

        self.assert_allowed_to_manage_kiddish()
        self.load_family_info_by_family_id(self.account.family_info.family_id)

        self.get_account_by_uid(
            self.form_values.get('kiddish_uid'),
            enabled_required=False,
            get_family_info=True,
            ip=self.client_ip,
        )

        try:
            self.check_kiddish_consistent(self.account)
        except AccountInvalidTypeError:
            # Это состояние возможно, если в момент запроса инфы о семье
            # аккаунт был ещё ребёнкишем, а в момент запроса инфы о ребёнкише
            # он стал взрослым участником семьи.
            log.debug('Family kids collection contains not-kiddish account')
            raise InternalTemporaryError()

        with UPDATE(
            self.account,
            self.request.env,
            events={'action': 'kiddish_change', 'consumer': self.consumer}
        ):
            person_attrs = ['birthday', 'firstname', 'gender', 'lastname']
            for attr in person_attrs:
                if self.form_values[attr] is not Undefined:
                    setattr(self.account.person, attr, self.form_values[attr])

            if self.form_values['avatar_id'] is not Undefined:
                self.account.person.default_avatar = self.form_values['avatar_id']

            if self.form_values['content_rating_class'] is not Undefined:
                self.account.content_rating_class = self.form_values['content_rating_class']

            if self.form_values['display_name'] is not None:
                self.account.person.display_name = self.form_values['display_name']

            if self.form_values['music_content_rating_class'] is not Undefined:
                self.account.music_content_rating_class = self.form_values['music_content_rating_class']

            if self.form_values['video_content_rating_class'] is not Undefined:
                self.account.video_content_rating_class = self.form_values['video_content_rating_class']

        self.fill_response_with_kiddish()
