# -*- coding: utf-8 -*-
import logging

from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.constants import (
    AUTHENTICATION_MEDIA_SESSION,
    AUTHENTICATION_MEDIA_UID,
    AUTHENTICATION_MEDIA_USER_TICKET,
)
from passport.backend.api.views.bundle.family.forms import FamilyBasicForm
from passport.backend.api.views.bundle.headers import HEADER_CONSUMER_CLIENT_IP
from passport.backend.api.views.bundle.mixins import BundleAccountGetterMixin
from passport.backend.api.views.bundle.mixins.common import BundleTvmUserTicketMixin
from passport.backend.api.views.bundle.mixins.family import BundleFamilyMixin
from passport.backend.core.models.family import FamilyInfo
from passport.backend.core.runner.context_managers import CREATE


log = logging.getLogger('passport.api.view.bundle.family.create')


class CreateFamilyView(
    BaseBundleView,
    BundleAccountGetterMixin,
    BundleFamilyMixin,
    BundleTvmUserTicketMixin,
):
    basic_form = FamilyBasicForm

    required_headers = (
        HEADER_CONSUMER_CLIENT_IP,
    )

    required_grants = ['family.create']
    by_uid_grant = 'family.create_by_uid'

    def process_request(self):
        self.process_basic_form()

        # Загрузить аккаунт
        self.get_account_from_available_media(
            enabled_media=(
                AUTHENTICATION_MEDIA_UID,
                AUTHENTICATION_MEDIA_USER_TICKET,
                AUTHENTICATION_MEDIA_SESSION,
            ),
            by_uid_grant=self.by_uid_grant,
            enabled_required=True,
            get_family_info=True,
            multisession_uid=self.form_values['multisession_uid'],
        )

        # Проверить, что пользователь пока не семейный
        self.assert_has_no_family()

        # Создать семью
        events = {'action': 'family_create', 'consumer': self.consumer}
        with CREATE(FamilyInfo(), self.request.env, events) as family_info:
            family_info.admin_uid = self.account.uid
            family_info.add_member_uid(self.account.uid, 0)

        log.debug('Created family id=%s admin=%s'.format(
            family_info.family_id,
            self.account.uid,
        ))

        # Вернуть family id
        self.response_values.update(
            family_id=family_info.family_id,
        )
