# -*- coding: utf-8 -*-
from datetime import datetime

from passport.backend.api.views.bundle.mixins.family import BundleFamilyMixin
from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.family.forms import CreateChildForm
from passport.backend.core.utils.decorators import cached_property
from passport.backend.core.logging_utils.loggers.statbox import StatboxLogger

from passport.backend.api.views.bundle.mixins import (
    BundleAccountGetterMixin,
    BundleAccountResponseRendererMixin,
    BundlePhoneMixin,
    BundlePasswordValidationMixin,
)
from passport.backend.api.views.bundle.headers import (
    HEADER_CLIENT_COOKIE,
    HEADER_CONSUMER_CLIENT_IP,
    HEADER_CLIENT_HOST,
    HEADER_CLIENT_USER_AGENT,
)
from passport.backend.api.views.bundle.exceptions import (
    AccountAlreadyRegisteredError,
    EulaIsNotAcceptedError,
)

from passport.backend.core.runner.context_managers import (
    CREATE,
    UPDATE,
)
from passport.backend.api.common.account import (
    build_default_person_registration_info,
    default_account,
)


class CreateChildView(
    BundleFamilyMixin,
    BundleAccountResponseRendererMixin,
    BundleAccountGetterMixin,
    BundlePasswordValidationMixin,
    BundlePhoneMixin,
    BaseBundleView,
):
    # Требуется трек
    require_track = True
    # Поля из формы
    basic_form = CreateChildForm
    # Необходимые заголовки, нам нужна кука пользователя
    required_headers = (
        HEADER_CLIENT_COOKIE,
        HEADER_CONSUMER_CLIENT_IP,
        HEADER_CLIENT_USER_AGENT,
        HEADER_CLIENT_HOST,
    )
    # Грант
    required_grants = ['family.create_child']

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode='register',
            track_id=self.track_id,
            ip=self.client_ip,
            user_agent=self.user_agent,
        )

    def process_request(self):
        self.process_basic_form()
        # Проверяем принятое пользовательское соглашение
        self.__check_eula()
        # Получаем аккаунт выполнивший запрос из сессии
        self.get_account_from_session(
            need_phones=True,
        )
        # Прочитали трек, нужно что бы достать авторизированный телефон
        self.read_track()
        self.__check_track_state()
        # Проверям что не требуется подтверждать телефон
        self.assert_secure_phone_confirmed_with_track()
        # Проверить, что пользователь - админ семьи, тут же проверяется что есть семья
        # Тут же загружаем family_info
        self.assert_is_family_admin(load_family_info=True)
        # Регистрируем детский аккаунт:
        # Создаем новый аккаунт с аттрибутом is_child, привязываем номер родителя, добавляем в семью
        return self.__register_child_account()

    def __check_track_state(self):
        if self.track.is_successful_registered:
            raise AccountAlreadyRegisteredError()

    def __register_child_account(self):
        """
        Регистрация портального аккаунта с выставлением аттрибута is_child
        Добавление в семью
        """
        self.validate_password()
        # Бронируем место
        # Если мест нет тут же кидаем ошибку
        place = self.get_family_member_free_place()

        new_account = default_account(
            self.form_values['login'],
            datetime.now(),
            self.form_values,
            build_default_person_registration_info(self.client_ip),
        )
        with CREATE(
            new_account,
            self.request.env,
            events={'action': 'child_account_register', 'consumer': self.consumer}
        ):
            new_account.content_rating_class = self.form_values['content_rating_class']
            save_secure_phone = self.build_save_secure_phone(
                aliasify=False,
                enable_search_alias=False,
                allow_to_take_busy_alias_from_any_account=False,
                language=self.track.language,
                account=new_account,
                phone_number=self.account.phones.secure,
                should_ignore_binding_limit=False,
                is_new_account=True,
            )
            save_secure_phone.submit()

        with UPDATE(
            self.family_info,
            self.request.env,
            events={'action': 'family_add_child', 'consumer': self.consumer}
        ):
            self.family_info.add_member_uid(new_account.uid, place)


        self.response_values.update(uid=new_account.uid)

    def __check_eula(self):
        if not self.form_values['eula_accepted']:
            raise EulaIsNotAcceptedError()
