# -*- coding: utf-8 -*-

import logging

from passport.backend.api.common.logs import setup_log_prefix
from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.constants import (
    AUTHENTICATION_MEDIA_SESSION,
    AUTHENTICATION_MEDIA_USER_TICKET,
    FAMILY_MANAGE_SCOPE,
    SESSIONID_SCOPE,
)
from passport.backend.api.views.bundle.exceptions import (
    RequestCredentialsAllMissingError,
    ValidationFailedError,
)
from passport.backend.api.views.bundle.family.forms import CreateKiddishForm
from passport.backend.api.views.bundle.mixins.account import (
    BundleAccountGetterMixin,
    BundleAccountResponseRendererMixin,
)
from passport.backend.api.views.bundle.mixins.common import BundleTvmUserTicketMixin
from passport.backend.api.views.bundle.mixins.family import BundleFamilyMixin
from passport.backend.api.views.bundle.mixins.kiddish import BundleKiddishMixin
from passport.backend.core.runner.context_managers import (
    CREATE,
    UPDATE,
)


log = logging.getLogger(__name__)


class CreateKiddishView(
    BundleKiddishMixin,
    BundleFamilyMixin,
    BundleAccountResponseRendererMixin,
    BundleAccountGetterMixin,
    BundleTvmUserTicketMixin,
    BaseBundleView,
):
    basic_form = CreateKiddishForm
    required_grants = ['family.create_kiddish']
    family_id_grant = 'family.create_kiddish_by_family_id'

    def process_request(self):
        self.process_basic_form()

        try:
            self.get_family_admin_account()
            family_id = self.account.family_info.family_id
        except RequestCredentialsAllMissingError:
            if not self.form_values['family_id']:
                raise
            self.check_grant(self.family_id_grant)
            family_id = self.form_values['family_id']
            log.debug('Use specified family id')

        self.load_family_info_by_family_id(family_id)
        place = self.get_family_kid_free_place()

        kwargs = [
            'avatar_id',
            'birthday',
            'content_rating_class',
            'display_name',
            'firstname',
            'gender',
            'lastname',
            'music_content_rating_class',
            'video_content_rating_class',
        ]
        kwargs = {k: self.form_values.get(k) for k in kwargs}
        kiddish = self.create_kiddish(**kwargs)

        with CREATE(
            kiddish,
            self.request.env,
            events={'action': 'kiddish_create', 'consumer': self.consumer}
        ):
            pass

        self.account = kiddish
        setup_log_prefix(kiddish)

        with UPDATE(
            self.family_info,
            self.request.env,
            events={'action': 'family_add_kiddish', 'consumer': self.consumer},
        ):
            self.family_info.kids.add_member_uid(kiddish.uid, place)

        self.fill_response_with_kiddish()

    def get_family_admin_account(self):
        self.get_account_from_available_media(
            enabled_media=(
                AUTHENTICATION_MEDIA_SESSION,
                AUTHENTICATION_MEDIA_USER_TICKET,
            ),
            get_family_info=True,
            multisession_uid=self.form_values.get('multisession_uid'),
            required_scope=(FAMILY_MANAGE_SCOPE, SESSIONID_SCOPE),
        )
        self.assert_allowed_to_manage_kiddish()
        self.check_family_id_match_if_specified(self.account.family_info.family_id)

    def check_family_id_match_if_specified(self, family_id):
        if self.form_values['family_id'] and self.form_values['family_id'] != family_id:
            raise ValidationFailedError(['family_id.invalid'])
