# -*- coding: utf-8 -*-

import logging

from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.constants import (
    AUTHENTICATION_MEDIA_SESSION,
    AUTHENTICATION_MEDIA_UID,
    AUTHENTICATION_MEDIA_USER_TICKET,
    FAMILY_MANAGE_SCOPE,
    SESSIONID_SCOPE,
)
from passport.backend.api.views.bundle.exceptions import (
    AccountInvalidTypeError,
    InternalTemporaryError,
)
from passport.backend.api.views.bundle.family.forms import DeleteKiddishForm
from passport.backend.api.views.bundle.mixins.account import BundleAccountGetterMixin
from passport.backend.api.views.bundle.mixins.common import BundleTvmUserTicketMixin
from passport.backend.api.views.bundle.mixins.family import BundleFamilyMixin
from passport.backend.api.views.bundle.mixins.kiddish import BundleKiddishMixin
from passport.backend.core.runner.context_managers import DELETE


log = logging.getLogger(__name__)


class DeleteKiddishView(
    BundleKiddishMixin,
    BundleFamilyMixin,
    BundleAccountGetterMixin,
    BundleTvmUserTicketMixin,
    BaseBundleView,
):
    basic_form = DeleteKiddishForm
    required_grants = ['family.delete_kiddish']
    by_uid_grant = 'family.delete_kiddish_by_uid'

    def process_request(self):
        self.process_basic_form()

        self.get_account_from_available_media(
            by_uid_grant=self.by_uid_grant,
            enabled_media=(
                AUTHENTICATION_MEDIA_SESSION,
                AUTHENTICATION_MEDIA_UID,
                AUTHENTICATION_MEDIA_USER_TICKET,
            ),
            get_family_info=True,
            multisession_uid=self.form_values.get('multisession_uid'),
            required_scope=(FAMILY_MANAGE_SCOPE, SESSIONID_SCOPE),
        )

        self.assert_allowed_to_manage_kiddish()
        self.load_family_info_by_family_id(self.account.family_info.family_id)

        self.get_account_by_uid(
            self.form_values.get('kiddish_uid'),
            enabled_required=False,
            get_family_info=True,
            ip=self.client_ip,
        )

        try:
            self.check_kiddish_consistent(self.account)
        except AccountInvalidTypeError:
            # Это состояние возможно, если в момент запроса инфы о семье
            # аккаунт был ещё ребёнкишем, а в момент запроса инфы о ребёнкише
            # он стал взрослым участником семьи.
            log.debug('Family kids collection contains not-kiddish account')
            raise InternalTemporaryError()

        with DELETE(
            self.account,
            self.request.env,
            events={'action': 'kiddish_delete', 'consumer': self.consumer}
        ):
            pass
