# -*- coding: utf-8 -*-

from datetime import datetime
import logging
import time

from passport.backend.api.common.ip import get_ip_autonomous_system
from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.constants import (
    AUTHENTICATION_MEDIA_SESSION,
    AUTHENTICATION_MEDIA_UID,
    AUTHENTICATION_MEDIA_USER_TICKET,
)
from passport.backend.api.views.bundle.exceptions import RateLimitExceedError
from passport.backend.api.views.bundle.family.forms import FamilyCreateInviteForm
from passport.backend.api.views.bundle.headers import HEADER_CONSUMER_CLIENT_IP
from passport.backend.api.views.bundle.mixins import BundleAccountGetterMixin
from passport.backend.api.views.bundle.mixins.common import BundleTvmUserTicketMixin
from passport.backend.api.views.bundle.mixins.family import BundleFamilyMixin
from passport.backend.api.views.bundle.mixins.kolmogor import KolmogorMixin
from passport.backend.api.views.bundle.mixins.phone import YASMS_EXCEPTIONS_MAPPING
from passport.backend.core.builders.antifraud import (
    BaseAntifraudApiError,
    get_antifraud_api,
    ScoreAction,
)
from passport.backend.core.builders.yasms import exceptions as yasms_errors
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.loggers import StatboxLogger
from passport.backend.core.models.family import FamilyInvite
from passport.backend.core.types.email.email import mask_email_for_statbox
from passport.backend.core.types.phone_number.phone_number import mask_phone_number
from passport.backend.core.utils.decorators import cached_property
from passport.backend.core.ydb.processors.family_invite import (
    delete_family_invite,
    insert_family_invite,
)
from passport.backend.utils.string import smart_text
from passport.backend.utils.time import get_unixtime


log = logging.getLogger('passport.api.view.bundle.family.issue_invite')
event_log = logging.getLogger('historydb.event')


class IssueInviteView(
    BaseBundleView,
    BundleAccountGetterMixin,
    BundleFamilyMixin,
    KolmogorMixin,
    BundleTvmUserTicketMixin,
):
    basic_form = FamilyCreateInviteForm

    required_headers = (
        HEADER_CONSUMER_CLIENT_IP,
    )

    required_grants = ['family.issue_invite']
    by_uid_grant = 'family.issue_invite_by_uid'

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode='family',
            action='family_issue_invite',
            consumer=self.consumer,
            ip=self.client_ip,
            user_agent=self.user_agent,
        )

    def _assert_counters(self, family_id, uid):
        per_family_key_template = settings.FAMILY_INVITE_ISSUE_PER_FAMILY_COUNTER
        per_family_counter = self.build_counter(
            keyspace=settings.FAMILY_INVITE_KOLMOGOR_KEY_SPACE,
            name=per_family_key_template % family_id,
            limit=settings.COUNTERS[per_family_key_template],
        )
        per_uid_key_template = settings.FAMILY_INVITE_ISSUE_PER_UID_COUNTER
        per_uid_counter = self.build_counter(
            keyspace=settings.FAMILY_INVITE_KOLMOGOR_KEY_SPACE,
            name=per_uid_key_template % uid,
            limit=settings.COUNTERS[per_uid_key_template],
        )
        counters = [per_family_counter, per_uid_counter]
        self.failsafe_check_kolmogor_counters(counters)
        self.failsafe_inc_kolmogor_counters(counters)

    def _score_invite_sent_via_email(self, email):
        af_api = get_antifraud_api()
        features = dict(
            channel='pharma',
            subchannel='family_invite_email',
            t=get_unixtime() * 1000,

            AS=get_ip_autonomous_system(self.client_ip),
            email=email,
            external_id='track-{}'.format(self.track_id),
            family_id=self.family_info.family_id,
            ip=str(self.client_ip),
            uid=self.account.uid,
            user_agent=self.user_agent,
        )
        try:
            af_response = af_api.score(features)
        except BaseAntifraudApiError as e:
            log.debug('Antifraud API request ended with error: {}. Allowing to send invite'.format(str(e)))
            return

        if af_response.action != ScoreAction.ALLOW:
            log.debug('Antifraud API rejected sending invite (action={})'.format(af_response.action))
            self.statbox.log(status='error', error='antifraud_score_deny')
            raise RateLimitExceedError()

    def process_request(self):
        self.process_basic_form()

        # Вычислить send method
        if self.form_values['sms_phone']:
            send_method = FamilyInvite.SEND_METHOD_SMS
            contact = self.form_values['sms_phone']
            contact_for_db = mask_phone_number(str(contact))
        elif self.form_values['email']:
            send_method = FamilyInvite.SEND_METHOD_EMAIL
            contact = self.form_values['email']
            contact_for_db = mask_email_for_statbox(contact)
        else:
            send_method = 0
            contact = ''
            contact_for_db = ''

        # Загрузить аккаунт
        self.get_account_from_available_media(
            enabled_media=(
                AUTHENTICATION_MEDIA_UID,
                AUTHENTICATION_MEDIA_USER_TICKET,
                AUTHENTICATION_MEDIA_SESSION,
            ),
            by_uid_grant=self.by_uid_grant,
            multisession_uid=self.form_values['multisession_uid'],
            enabled_required=True,
            get_family_info=True,
            need_display_name=bool(send_method),
        )
        self.statbox.bind(uid=self.account.uid)

        # Проверить, что пользователь - админ семьи
        self.assert_is_family_admin(load_family_info=True)

        # Проверить, что в семье есть свободные места
        self.assert_family_has_empty_slots()

        # Проверить счётчики для отправки инвайтов
        self._assert_counters(
            family_id=self.family_info.family_id,
            uid=self.account.uid,
        )

        # Спросить в АФ, можно ли слать инвайт
        if send_method == FamilyInvite.SEND_METHOD_EMAIL and settings.FAMILY_INVITES_VIA_EMAIL_CHECK_ANTIFRAUD_SCORE:
            # Отправку по смс не скорим: она уже скорится на стороне YaSMS
            # Выдачи инвайта без отправки не скорим: нет угрозы фишинга
            self._score_invite_sent_via_email(email=contact)

        # Создать инвайт
        create_time = datetime.now()
        invite = FamilyInvite.generate(
            family_id=self.family_info.family_id,
            send_method=send_method,
            contact=contact_for_db,
            uid=self.account.uid,
            create_time=time.mktime(create_time.timetuple())
        )
        insert_family_invite(invite)

        log.debug('uid=%s created invite %s for family family_id=%s' % (
            self.account.uid,
            invite.invite_id,
            self.family_info.family_id,
        ))

        # Отправить инвайт, если требуется
        if send_method == FamilyInvite.SEND_METHOD_SMS:
            if settings.FAMILY_DISABLE_SMS_INVITE:
                log.debug(
                    'Invite %s is not sent to phone %s by uid %s: disabled in settings',
                    invite.invite_id,
                    contact,
                    self.account.uid,
                )
                delete_family_invite(invite.invite_id)
                return
            else:
                try:
                    self.send_invite_by_sms(invite, contact)
                except yasms_errors.YaSmsError as err:
                    log.debug(
                        'Failed to send invite %s by SMS: %s' %
                        (invite.invite_id, err),
                    )
                    delete_family_invite(invite.invite_id)
                    try:
                        exc_class = YASMS_EXCEPTIONS_MAPPING[err.__class__]
                    except KeyError:
                        raise
                    raise exc_class()
        elif send_method == FamilyInvite.SEND_METHOD_EMAIL:
            self.send_invite_by_email(invite, contact)

        # Записать о выпуске инвайта в statbox
        self.statbox.log(
            status='ok',
            scenario=self.form_values['scenario'],
            **self.invite_to_statbox(invite)
        )

        # Сделать запись об инвайте в History DB
        self.write_history_db(
            uid=self.account.uid,
            create_time=create_time,
            action='family_issue_invite',
            invite_id=invite.invite_id,
            family_id=self.family_info.family_id,
            send_method=FamilyInvite.send_method_to_text(send_method),
            contact=smart_text(contact),
            consumer=self.consumer,
        )

        # Вернуть invite id
        self.response_values.update(
            invite_id=invite.invite_id,
        )
