# -*- coding: utf-8 -*-
import logging

from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.exceptions import (
    AccountIsChildError,
    FamilyIsNotAMember,
)
from passport.backend.api.views.bundle.family.forms import FamilyBasicForm
from passport.backend.api.views.bundle.headers import HEADER_CONSUMER_CLIENT_IP
from passport.backend.api.views.bundle.mixins import BundleAccountGetterMixin
from passport.backend.api.views.bundle.mixins.family import BundleFamilyMixin
from passport.backend.core.runner.context_managers import UPDATE


log = logging.getLogger('passport.api.view.bundle.family.remove_member')


class LeaveView(BaseBundleView,
                BundleAccountGetterMixin,
                BundleFamilyMixin):
    basic_form = FamilyBasicForm

    required_headers = (
        HEADER_CONSUMER_CLIENT_IP,
    )

    required_grants = ['family.leave']
    by_uid_grant = 'family.leave_by_uid'

    def process_request(self):
        self.process_basic_form()
        # Загрузить аккаунт
        self.get_account_from_uid_or_session(
            by_uid_grant=self.by_uid_grant,
            multisession_uid=self.form_values['multisession_uid'],
            enabled_required=True,
            get_family_info=True,
        )

        # Проверить, что пользователь - не админ семьи
        self.assert_is_not_family_admin()

        if self.account.is_child:
            raise AccountIsChildError()

        # Загрузить информацию о семье
        # TODO в принципе, тут можно создать фиктивный объект семьи,
        #  и добавлять в него, не ходя в ЧЯ, но это костыль
        self.load_family_info_by_family_id(self.account.family_info.family_id)
        if self.account.uid not in self.family_info:
            message = 'User was removed from a family %s during request' % \
                      self.family_info.family_id
            raise FamilyIsNotAMember(message)
        assert self.account.uid != self.family_info.admin_uid

        # Выйти из семьи
        events = {'action': 'family_leave', 'consumer': self.consumer}
        with UPDATE(self.family_info, self.request.env, events):
            self.family_info.remove_member_uid(self.account.uid)
            if self.account.can_manage_children:
                with UPDATE(self.account, self.request.env, events):
                    self.account.can_manage_children = False

        log.debug('Removed uid=%s from family family_id=%s' % (
            self.account.uid,
            self.family_info.family_id,
        ))

        self.response_values.update(
            family_id=self.family_info.family_id,
        )
