# -*- coding: utf-8 -*-
import logging

from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.exceptions import (
    AccountIsChildError,
    FamilyInvalidFamily,
    FamilyInvalidPlace,
    FamilyIsAdmin,
)
from passport.backend.api.views.bundle.family.forms import FamilyRemoveMemberForm
from passport.backend.api.views.bundle.headers import HEADER_CONSUMER_CLIENT_IP
from passport.backend.api.views.bundle.mixins import BundleAccountGetterMixin
from passport.backend.api.views.bundle.mixins.family import BundleFamilyMixin
from passport.backend.core.exceptions import UnknownUid
from passport.backend.core.models.account import Account
from passport.backend.core.runner.context_managers import UPDATE


log = logging.getLogger('passport.api.view.bundle.family.remove_member')


class RemoveMemberView(BaseBundleView,
                       BundleAccountGetterMixin,
                       BundleFamilyMixin):
    basic_form = FamilyRemoveMemberForm

    required_headers = (
        HEADER_CONSUMER_CLIENT_IP,
    )

    required_grants = ['family.remove_member']
    by_uid_grant = 'family.remove_member_by_uid'

    def assert_member_is_not_child(self, member_uid):
        bb_response = self.blackbox.userinfo(uid=member_uid, ip=self.client_ip)
        try:
            member_account = Account().parse(bb_response)
        except UnknownUid:
            # эта ситуация должна быть недостижимой, но неконсистентность - не повод мешать выкидыванию из семьи
            return

        if member_account.is_child:
            raise AccountIsChildError()

    def process_request(self):
        self.process_basic_form()

        # Загрузить аккаунт
        self.get_account_from_uid_or_session(
            by_uid_grant=self.by_uid_grant,
            multisession_uid=self.form_values['multisession_uid'],
            enabled_required=True,
            get_family_info=True,
        )

        # Проверить, что пользователь - админ семьи
        self.assert_is_family_admin(load_family_info=True)

        # Получить uid удаляемого члена семьи
        if self.form_values['member_uid'] is not None:
            remove_uid = self.form_values['member_uid']
            dbg_message = 'by uid'
        else:
            parts = self.form_values['place_id'].split(':')
            family_id = parts[0]
            remove_place = int(parts[1])
            if self.family_info.family_id != family_id:
                message = (
                    'Requested family_id %s does not match actual %s' %
                    (family_id, self.family_info)
                )
                log.debug(message)
                raise FamilyInvalidFamily(message)
            dbg_message = 'by place=%s' % remove_place
            remove_uid = self.family_info.get_member_uid_by_place(remove_place)
            if remove_uid is None:
                message = (
                    'Family %s doesn\'t have a member with place %s' %
                    (self.family_info.family_id, remove_place)
                )
                log.debug(message)
                raise FamilyInvalidPlace(message)

        # Проверить, что удаляемый аккаунт - не текущий пользователь
        # (если так, то он - админ семьи, вернуть ошибку об этом)
        if self.account.uid == remove_uid:
            message = (
                'User being removed is current user and admin of family %s'
                % self.family_info.family_id
            )
            log.debug(message)
            raise FamilyIsAdmin(message)

        self.assert_has_family_member(remove_uid)
        self.assert_member_is_not_child(remove_uid)

        # Удалить аккаунт из семьи
        events = {'action': 'family_remove_member', 'consumer': self.consumer}
        with UPDATE(self.family_info, self.request.env, events):
            self.family_info.remove_member_uid(remove_uid)

        log.debug('Removed uid=%s %s from family family_id=%s' % (
            remove_uid,
            dbg_message,
            self.account.family_info.family_id,
        ))

        self.response_values.update(
            family_id=self.family_info.family_id,
        )
