# -*- coding: utf-8 -*-
from datetime import datetime
import logging

from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.exceptions import FamilyInvalidInvite
from passport.backend.api.views.bundle.family.forms import FamilyInviteForm
from passport.backend.api.views.bundle.headers import (
    HEADER_CLIENT_COOKIE,
    HEADER_CLIENT_HOST,
    HEADER_CONSUMER_CLIENT_IP,
)
from passport.backend.api.views.bundle.mixins import BundleAccountGetterMixin
from passport.backend.api.views.bundle.mixins.family import BundleFamilyMixin
from passport.backend.core.logging_utils.loggers import StatboxLogger
from passport.backend.core.models.family import FamilyInvite
from passport.backend.core.utils.decorators import cached_property


log = logging.getLogger('passport.api.view.bundle.family.revoke_invite')


class RevokeInviteView(BaseBundleView,
                       BundleAccountGetterMixin,
                       BundleFamilyMixin):
    basic_form = FamilyInviteForm

    required_headers = (
        HEADER_CLIENT_HOST,
        HEADER_CONSUMER_CLIENT_IP,
        HEADER_CLIENT_COOKIE,
    )

    required_grants = ['family.revoke_invite']

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode='family',
            consumer=self.consumer,
            ip=self.client_ip,
            user_agent=self.user_agent,
        )

    def process_request(self):
        self.process_basic_form()

        # Загрузить аккаунт
        self.get_account_from_session(
            multisession_uid=self.form_values['multisession_uid'],
            enabled_required=True,
            get_family_info=True,
        )

        # Проверить, что пользователь - админ семьи
        self.assert_is_family_admin(load_family_info=True)

        # Загрузить инвайт
        invite = self.find_invite(self.form_values['invite_id'])
        if invite is None:
            message = 'Invite %s not found' % self.form_values['invite_id']
            log.debug(message)
            raise FamilyInvalidInvite(message)

        # Проверить, что инвайт относится к семье пользователя
        if invite.family_id != self.account.family_info.family_id:
            message = (
                'Invite %s family id %s != user %s family_id %s' %
                (
                    invite.invite_id,
                    invite.family_id,
                    self.account.uid,
                    self.account.family_info.family_id,
                ),
            )
            log.debug(message)
            message = 'Invite %s not found' % self.form_values['invite_id']
            raise FamilyInvalidInvite(message)

        # Удалить инвайт
        self.delete_invite(invite.invite_id)

        log.debug('uid=%s deleted invite %s for family family_id=%s' % (
            self.account.uid,
            invite.invite_id,
            self.family_info.family_id,
        ))

        # Записать об отзыве инвайта в statbox
        self.statbox.log(
            action='family_revoke_invite',
            uid=str(self.account.uid),
            **self.invite_to_statbox(invite)
        )

        # Сделать запись об удалении в HistoryDb
        self.write_history_db(
            uid=self.account.uid,
            create_time=datetime.now(),
            action='family_revoke_invite',
            invite_id=invite.invite_id,
            family_id=invite.family_id,
            consumer=self.consumer,
            send_method=FamilyInvite.send_method_to_text(invite.send_method),
            contact=invite.contact,
        )
