# -*- coding: utf-8 -*-
from passport.backend.api.common import json_response
from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.exceptions import (
    BlackboxUnavailableError,
    DatabaseUnavailableError,
    UnhandledBundleError,
)


CONSUMER = 'idm'
GRANT = 'idm.manage_roles'


class BaseIDMView(BaseBundleView):
    """
    https://wiki.yandex-team.ru/intranet/idm/API/
    """
    required_grants = [GRANT]

    def process_root_form(self):
        self.consumer = CONSUMER

    @property
    def idm_request_id(self):
        return self.headers.get('X-IDM-Request-Id')

    def ok_response(self, **response_values):
        return json_response(
            _code=200,  # http status code
            code=0,  # код ответа для IDM, обязательное поле
            **response_values
        )

    def error_response(self, errors, code=None, error_message=None, **kwargs):
        error = errors[0]
        need_retry = error in (
            BlackboxUnavailableError.error,
            DatabaseUnavailableError.error,
            UnhandledBundleError.error,
        )
        error_type = 'error' if need_retry else 'fatal'
        if code is None:
            code = 500 if need_retry else 400

        if error_message is not None:
            error_message = '%s (%s)' % (errors[0], error_message)
        else:
            error_message = errors[0]

        return json_response(**{
            '_code': 200,  # http status code
            'code': code,  # код ответа для IDM, обязательное поле
            error_type: error_message,
        })
