# -*- coding: utf-8 -*-
import logging

from passport.backend.api.views.bundle.idm.base import BaseIDMView
from passport.backend.api.views.bundle.idm.exceptions import (
    AltdomainAliasExistsError,
    InvalidRoleError,
)
from passport.backend.api.views.bundle.idm.forms import RoleForm
from passport.backend.api.views.bundle.mixins import BundleAccountGetterMixin
from passport.backend.core.conf import settings
from passport.backend.core.models.alias import AltDomainAlias
from passport.backend.core.runner.context_managers import UPDATE


log = logging.getLogger('passport.api.view.bundle.idm')


class MailRoleConfig(object):
    # TODO: заменить на dataclass после перехода на py3
    def __init__(self, altdomain, hide_yandex_domains_emails=False):
        self.altdomain = altdomain
        self.hide_yandex_domains_emails = hide_yandex_domains_emails


MAIL_ALTDOMAIN_ROLES = {
    'ext': MailRoleConfig(altdomain='ext.yandex.ru', hide_yandex_domains_emails=True),
    'k50': MailRoleConfig(altdomain='k50.ru'),
    'openyard': MailRoleConfig(altdomain='openyard.ru'),
    'yango': MailRoleConfig(altdomain='yango.com'),
    'cloudil': MailRoleConfig(altdomain='cloudil.co.il'),
    'meteum': MailRoleConfig(altdomain='meteum.ai'),
    'srb.tech': MailRoleConfig(altdomain='srb.tech'),
    'userver.tech': MailRoleConfig(altdomain='userver.tech'),
    'legal.direct': MailRoleConfig(altdomain='legal.direct'),
}


class IDMInfoView(BaseIDMView):
    def process_request(self):
        self.response_values.update(
            roles={
                'slug': 'mail',
                'name': u'Корпоративная почта',
                'values': {
                    role: {
                        'name': {
                            'ru': u'В домене %s' % role_config.altdomain,
                            'en': u'At %s' % role_config.altdomain,
                        },
                    }
                    for role, role_config in sorted(MAIL_ALTDOMAIN_ROLES.items())
                },
            },
        )


class BaseManageRoleView(BaseIDMView, BundleAccountGetterMixin):
    basic_form = RoleForm
    action = None
    require_enabled_account = True

    def modify_account(self, role):
        raise NotImplementedError()

    def process_request(self):
        log.debug('IDM request id: %s', self.idm_request_id or '-')
        self.process_basic_form()

        role = self.form_values['role']
        if not role.get('mail') or role['mail'] not in MAIL_ALTDOMAIN_ROLES.keys():
            raise InvalidRoleError()

        self.get_account_by_login(self.form_values['login'], enabled_required=self.require_enabled_account)

        events = {
            'action': self.action,
            'consumer': self.consumer,
        }
        with UPDATE(self.account, self.request.env, events):
            self.modify_account(role)


class IDMAddRoleView(BaseManageRoleView):
    action = 'add_role'

    def modify_account(self, role):
        if not role.get('mail'):
            return

        role_config = MAIL_ALTDOMAIN_ROLES[role['mail']]
        altdomain = role_config.altdomain
        altdomain_id = settings.ALT_DOMAINS[altdomain]
        if self.account.altdomain_alias and self.account.altdomain_alias.domain_id != altdomain_id:
            # Сейчас аккаунт не может иметь два разных altdomain-алиаса
            raise AltdomainAliasExistsError()
        self.account.altdomain_alias = AltDomainAlias(
            parent=self.account,
            login='%s@%s' % (self.account.login, altdomain),
        )
        if role_config.hide_yandex_domains_emails:
            self.account.hide_yandex_domains_emails = True


class IDMRemoveRoleView(BaseManageRoleView):
    action = 'remove_role'
    require_enabled_account = False

    def modify_account(self, role):
        if not role.get('mail'):
            return

        role_config = MAIL_ALTDOMAIN_ROLES[role['mail']]
        altdomain_id = settings.ALT_DOMAINS[role_config.altdomain]

        if self.account.altdomain_alias and self.account.altdomain_alias.domain_id == altdomain_id:
            # Удаляем алиас только если он с тем же доменом
            self.account.altdomain_alias = None

        self.account.hide_yandex_domains_emails = False
