# -*- coding: utf-8 -*-

from datetime import datetime
import json
import logging
from os import path
import subprocess

from flask import request
from passport.backend.api.common.account import (
    build_default_person_registration_info,
    default_account,
)
from passport.backend.api.common.logbroker import get_api_logbroker_writer
from passport.backend.api.forms.base import UidForm
from passport.backend.api.validators import Password as PasswordValidator
from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.email.exceptions import EmailNotFoundError
from passport.backend.api.views.bundle.exceptions import (
    Account2FAEnabledError,
    AccountDeletionOperationNotFound,
    ValidationFailedError,
    YaKeyBackupNotFoundError,
)
from passport.backend.api.views.bundle.headers import (
    HEADER_CLIENT_USER_AGENT,
    HEADER_CONSUMER_CLIENT_IP,
)
from passport.backend.api.views.bundle.internal.exceptions import (
    AccountDeletionFailed,
    AccountDeletionTemporaryError,
    AccountNotTestAccountError,
    AttributeMustBeLessOrEqualThan,
    AttributeMustBeNull,
    EmailNotUpdatableError,
    PhoneNotFoundError,
    PhoneNotTestNumberError,
    PhoneNotUpdatableError,
    SecurePhoneAlreadyExistsError,
    SecurePhoneBoundAndConfirmedError,
)
from passport.backend.api.views.bundle.internal.forms import (
    AmPushMessageTestForm,
    ConfirmAndBindPhoneForm,
    DeleteYakeyBackupForm,
    LiteAccountRegisterForm,
    LogbrokerTestForm,
    SetPasswordForm,
    UpdateAccountDeletionOperationForm,
    UpdateAccountRegistrationDatetimeForm,
    UpdateEmailForm,
    UpdatePhoneForm,
)
from passport.backend.api.views.bundle.mixins import (
    BundleAccountGetterMixin,
    BundlePasswordChangeMixin,
    BundlePasswordValidationMixin,
    BundlePhoneMixin,
    BundleVerifyPasswordMixin,
)
from passport.backend.core import validators
from passport.backend.core.am_pushes.push_request import (
    AppTargetingTypes,
    Platforms,
    PushRequest,
    PushRequestRecipient,
)
from passport.backend.core.builders.blackbox.constants import BLACKBOX_YAKEY_BACKUP_NOT_FOUND_STATUS
from passport.backend.core.builders.blackbox.utils import (
    add_phone_arguments,
    get_attribute,
)
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.helpers import mask_sensitive_fields
from passport.backend.core.logging_utils.loggers import StatboxLogger
from passport.backend.core.models.phones.phones import SECURITY_IDENTITY
from passport.backend.core.models.yakey_backup import YaKeyBackup
from passport.backend.core.protobuf.challenge_pushes.challenge_pushes_pb2 import ChallengePushRequest
from passport.backend.core.protobuf.logbroker_test.logbroker_test_pb2 import LogbrokerTestMessage
from passport.backend.core.runner.context_managers import (
    CREATE,
    DELETE,
    UPDATE,
)
from passport.backend.core.services import get_service
from passport.backend.core.subscription import add_subscription
from passport.backend.core.types.login.login import is_test_yandex_login
from passport.backend.core.types.phone_number.phone_number import (
    FakePhoneNumber,
    InvalidPhoneNumber,
    PhoneNumber,
)
from passport.backend.core.utils.decorators import cached_property


log = logging.getLogger('passport.api.view.bundle.test')


class LiteAccountRegister(BaseBundleView):
    required_headers = (
        HEADER_CONSUMER_CLIENT_IP,
    )

    required_grants = ['internal.register_lite']

    basic_form = LiteAccountRegisterForm

    require_track = False

    def process_request(self):

        self.process_basic_form()

        try:
            # не используем self.validate_password, так как у нас нет трека
            validation_result = PasswordValidator().to_python(
                {'password': self.form_values['password']},
                validators.State(self.request.env),
            )
            self.form_values['quality'] = validation_result['quality']
        except validators.Invalid as e:
            raise ValidationFailedError.from_invalid(e)

        person_info = build_default_person_registration_info(request.env.user_ip)
        person_info.update(
            {
                'firstname': self.form_values['login'] if self.form_values['firstname'] is None else self.form_values['firstname'],
                'lastname': self.form_values['login'] if self.form_values['lastname'] is None else self.form_values['lastname'],
            }
        )

        with CREATE(
            default_account(
                self.form_values['login'],
                datetime.now(),
                self.form_values,
                person_info,
                alias_type='lite',
            ),
            request.env,
            {'action': 'account_register_lite', 'consumer': self.consumer}
        ) as account:
            add_subscription(account, get_service(sid=33))

        self.response_values.update(uid=account.uid)


class GetPhones(BaseBundleView, BundleAccountGetterMixin):
    required_grants = ['internal.get_phones']
    basic_form = UidForm

    def process_request(self):
        self.process_basic_form()

        kwargs = {'uid': self.form_values['uid']}
        kwargs = add_phone_arguments(**kwargs)
        userinfo = self.blackbox.userinfo(**kwargs)

        # Вызываем для обработки исключительных ситуаций, например аккаунт не
        # найден.
        self.parse_account(userinfo, enabled_required=False)

        phones = userinfo['phones'].values()

        for binding in userinfo['phone_bindings']:
            binding['phone_number'] = binding['phone_number'].e164

        response = {
            'phones': phones,
            'phone_bindings': userinfo['phone_bindings'],
            'phone_operations': userinfo['phone_operations'].values(),
        }

        secure_id = get_attribute(userinfo, 'phones.secure')
        if secure_id is not None:
            response['secure_id'] = int(secure_id)

        default_id = get_attribute(userinfo, 'phones.default')
        if default_id is not None:
            response['default_id'] = int(default_id)

        phone_numbers = []
        for phone in phones:
            attrs = phone['attributes']
            if 'number' in attrs:
                phone_numbers.append(attrs['number'])
        phone_bindings = self.blackbox.phone_bindings(phone_numbers=phone_numbers)
        for binding in phone_bindings:
            binding['phone_number'] = binding['phone_number'].e164
        response['phone_bindings'] = phone_bindings

        if not is_test_yandex_login(userinfo['login']):
            self._mask_real_phone_info(response)

        self.response_values.update(response)

    def _mask_real_phone_info(self, response):
        real_phone_ids = set()
        for phone in response['phones']:
            attrs = phone['attributes']
            if 'number' in attrs and not self._is_fake_number(attrs['number']):
                real_phone_ids.add(phone['id'])
                self._mask_attrs(attrs)
                self._mask_binding(phone['binding'])
                self._mask_operation(phone['operation'])

        for operation in response['phone_operations']:
            security_identity = operation['security_identity']
            security_identity_is_real_number = (
                security_identity is not SECURITY_IDENTITY and
                not self._is_fake_number('+%s' % security_identity)
            )
            operation_is_on_real_number = (
                operation['phone_id'] in real_phone_ids or
                operation['phone_id2'] in real_phone_ids
            )
            if security_identity_is_real_number or operation_is_on_real_number:
                self._mask_operation(operation)

        for binding in response['phone_bindings']:
            if not self._is_fake_number(binding['phone_number']):
                self._mask_binding(binding)

    def _is_fake_number(self, phone_number):
        try:
            phone_number = PhoneNumber.parse(phone_number)
        except InvalidPhoneNumber:
            return False
        return isinstance(phone_number, FakePhoneNumber)

    def _mask_attrs(self, attrs):
        attrs.update(mask_sensitive_fields(attrs, ['number']))

    def _mask_binding(self, binding):
        if binding is not None:
            binding.update(mask_sensitive_fields(binding, ['phone_number']))

    def _mask_operation(self, operation):
        if operation is not None:
            operation.update(mask_sensitive_fields(operation, ['code_value']))
            operation['security_identity'] = -1


class UpdatePhone(BaseBundleView, BundleAccountGetterMixin):
    required_grants = ['internal.update_phone']
    basic_form = UpdatePhoneForm

    UPDATABLE_TIMESTAMP_ATTRIBUTES = ['created', 'bound', 'secured', 'admitted', 'confirmed']

    def _get_account(self):
        self.get_account_by_uid(self.form_values['uid'], enabled_required=False, need_phones=True)
        return self.account

    def _check_phone_updatable(self, account, phone):
        if (
            not is_test_yandex_login(account.login) and
            not isinstance(phone.number, FakePhoneNumber)
        ):
            raise PhoneNotUpdatableError()

    def _update_phone_attributes(self, account, phone, attrs):
        for name in self.UPDATABLE_TIMESTAMP_ATTRIBUTES:
            new = attrs[name]
            if new:
                old = getattr(phone, name)
                if not old:
                    raise AttributeMustBeNull(name)
                setattr(phone, name, new)

    def _check_phone_consistent(self, p):
        self._check_phone_attr_less_or_equal_than(p, 'created', 'bound')
        self._check_phone_attr_less_or_equal_than(p, 'created', 'secured')
        self._check_phone_attr_less_or_equal_than(p, 'created', 'admitted')
        self._check_phone_attr_less_or_equal_than(p, 'created', 'confirmed')
        self._check_phone_attr_less_or_equal_than(p, 'bound', 'secured')
        self._check_phone_attr_less_or_equal_than(p, 'bound', 'confirmed')
        self._check_phone_attr_less_or_equal_than(p, 'secured', 'confirmed')

    def _check_phone_attr_less_or_equal_than(self, phone, lesser_attr, greater_attr):
        lesser = getattr(phone, lesser_attr)
        greater = getattr(phone, greater_attr)
        if lesser and greater:
            if not (lesser <= greater):
                raise AttributeMustBeLessOrEqualThan(lesser_attr, greater_attr)

    def process_request(self):
        self.process_basic_form()
        account = self._get_account()

        if not account.phones.has_id(self.form_values['phone_id']):
            raise PhoneNotFoundError()
        phone = account.phones.by_id(self.form_values['phone_id'])

        self._check_phone_updatable(account, phone)
        with UPDATE(account, request.env, {'action': 'update_phone_attributes', 'consumer': self.consumer}):
            self._update_phone_attributes(account, phone, self.form_values)
            self._check_phone_consistent(phone)


class SetPassword(BaseBundleView, BundleAccountGetterMixin, BundlePasswordChangeMixin, BundlePasswordValidationMixin):
    required_grants = ['internal.set_password']
    basic_form = SetPasswordForm

    def process_request(self):
        self.process_basic_form()
        self.get_account_by_uid(self.form_values['uid'], enabled_required=False, need_phones=True, emails=True)

        if self.account.totp_secret.is_set:
            raise Account2FAEnabledError()

        with UPDATE(self.account, self.request.env, {'action': 'set_password_internal'}):
            password, password_quality = self.validate_password(
                is_strong_policy=self.account.is_strong_password_required,
                old_password_hash=self.account.password.serialized_encrypted_password,
                emails=set([email.address for email in self.account.emails.native]),
                phone_number=self.account.phones.secure.number if self.account.phones.secure else None,
            )
            self.change_password(password, quality=password_quality, global_logout=False)


class DeleteYakeyBackup(BaseBundleView):
    required_grants = ['internal.delete_yakey_backup']
    basic_form = DeleteYakeyBackupForm

    def process_request(self):
        self.process_basic_form()

        number = self.form_values['number']

        response = self.blackbox.yakey_backup(number.digital)

        if response['status'] == BLACKBOX_YAKEY_BACKUP_NOT_FOUND_STATUS:
            raise YaKeyBackupNotFoundError()

        with DELETE(YaKeyBackup().parse(response['yakey_backup']), self.request.env, {}):
            pass


class FinishAccountDeletion(BaseBundleView):
    basic_form = UidForm

    required_grants = ['internal.finish_account_deletion']

    def process_request(self):
        self.process_basic_form()

        if not path.exists(settings.ACCOUNT_DELETER_PATH):
            log.error('Account deleter not found')
            raise AccountDeletionFailed()

        process = subprocess.Popen(
            [
                settings.ACCOUNT_DELETER_PATH,
                '--uids',
                str(self.form_values['uid']),
                '--',
                'step_last',
            ],
            shell=False,
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE,
        )

        out, err = process.communicate()
        if process.returncode != 0:
            err = err.replace('\n', r'\n')
            log.debug('Account deleter failed: %s' % err)
            raise AccountDeletionFailed()

        if not out:
            # Не получилось захватить лок, так как скрипт уже запущен
            raise AccountDeletionTemporaryError('Deletion process already running')

        self.response_values['uids'] = json.loads(out)


class UpdateAccountDeletionOperation(BaseBundleView, BundleAccountGetterMixin):
    required_grants = ['internal.update_account_deletion_operation']
    basic_form = UpdateAccountDeletionOperationForm

    def _get_account(self):
        self.get_account_by_uid(
            self.form_values['uid'],
            enabled_required=False,
        )
        return self.account

    def process_request(self):
        self.process_basic_form()

        account = self._get_account()
        if not account.deletion_operation:
            raise AccountDeletionOperationNotFound()

        with UPDATE(account, request.env, {'action': 'update_account_deletion_operation', 'consumer': self.consumer}):
            if self.form_values['started_at'] is not None:
                account.deletion_operation.started_at = self.form_values['started_at']


class UpdateAccountRegistrationDatetime(BaseBundleView, BundleAccountGetterMixin):
    required_grants = ['internal.update_account_registration_datetime']
    basic_form = UpdateAccountRegistrationDatetimeForm

    def process_request(self):
        self.process_basic_form()

        self.get_account_by_uid(self.form_values['uid'])

        events = {'action': 'update_account_registration_datetime', 'consumer': self.consumer}
        with UPDATE(self.account, request.env, events):
            self.account.registration_datetime = self.form_values['registration_datetime']


class UpdateEmail(BaseBundleView, BundleAccountGetterMixin):
    required_grants = ['internal.update_email']
    basic_form = UpdateEmailForm

    UPDATABLE_TIMESTAMP_ATTRIBUTES = ['created_at', 'bound_at', 'confirmed_at']

    def _get_account(self):
        self.get_account_by_uid(self.form_values['uid'], enabled_required=False, email_attributes='all')
        return self.account

    def _check_email_updatable(self, account):
        if not is_test_yandex_login(account.login):
            raise EmailNotUpdatableError()

    def _update_email_attributes(self, email, attrs):
        for name in self.UPDATABLE_TIMESTAMP_ATTRIBUTES:
            new = attrs[name]
            if new:
                old = getattr(email, name)
                if not old:
                    raise AttributeMustBeNull(name)
                setattr(email, name, new)

    def _check_email_consistent(self, e):
        self._check_email_attr_less_or_equal_than(e, 'created_at', 'bound_at')
        self._check_email_attr_less_or_equal_than(e, 'created_at', 'confirmed_at')
        self._check_email_attr_less_or_equal_than(e, 'bound_at', 'confirmed_at')

    def _check_email_attr_less_or_equal_than(self, email, lesser_attr, greater_attr):
        lesser = getattr(email, lesser_attr)
        greater = getattr(email, greater_attr)
        if lesser and greater:
            if not (lesser <= greater):
                raise AttributeMustBeLessOrEqualThan(lesser_attr, greater_attr)

    def process_request(self):
        self.process_basic_form()
        account = self._get_account()

        if self.form_values['email'] not in account.emails:
            raise EmailNotFoundError()
        email = account.emails[self.form_values['email']]

        self._check_email_updatable(account)
        with UPDATE(account, request.env, {'action': 'update_email_attributes', 'consumer': self.consumer}):
            self._update_email_attributes(email, self.form_values)
            self._check_email_consistent(email)


class ConfirmAndBindPhone(BaseBundleView, BundlePhoneMixin, BundleAccountGetterMixin, BundleVerifyPasswordMixin):
    require_track = False

    basic_form = ConfirmAndBindPhoneForm

    required_grants = ['internal.bind_test_phone']

    required_headers = (
        HEADER_CLIENT_USER_AGENT,
        HEADER_CONSUMER_CLIENT_IP,
    )

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode='bind_test_phone',
            track_id=self.track.track_id,
            uid=self.track.uid or '-',
            ip=self.client_ip,
            consumer=self.consumer,
            user_agent=self.user_agent,
        )

    @property
    def number(self):
        return PhoneNumber.parse(self.form_values['number'])

    def assert_secure_phone_absense(self):
        if self.secure_number is not None:
            raise SecurePhoneAlreadyExistsError()

    def assert_number_is_not_bound_as_secure(self):
        if self.secure_number and self.secure_number == self.number:
            raise SecurePhoneBoundAndConfirmedError()

    def process_request(self):
        self.process_basic_form()
        self.read_or_create_track('register')
        uid = self.form_values['uid']
        self.set_uid_to_track(uid)

        is_fake_phone = isinstance(self.number, FakePhoneNumber)
        if not is_fake_phone:
            raise PhoneNotTestNumberError('Not a test phone number transmitted', self.form_values['number'])

        self.get_account_by_uid(uid, enabled_required=True, need_phones=True)

        if not (
            settings.IS_BIND_TEST_PHONE_TO_NONTEST_ACCOUNT_ALLOWED or
            is_test_yandex_login(self.account.login)
        ):
            raise AccountNotTestAccountError()

        self.verify_password(uid, self.form_values['password'], need_phones=True)

        if self.form_values['secure']:
            self.assert_secure_phone_absense()
        else:
            self.assert_number_is_not_bound_as_secure()
        self._bind_phone_to_account()

    def _bind_phone_to_account(self):
        is_secure = self.form_values['secure']
        action = 'confirm_and_bind_secure' if is_secure else 'confirm_and_bind'
        if is_secure:
            save_phone = self.build_save_secure_phone(
                account=self.account,
                phone_number=self.number,
            )
        else:
            save_phone = self.build_save_simple_phone(
                account=self.account,
                phone_number=self.number,
            )
        save_phone.submit()

        with UPDATE(
            self.account,
            self.request.env,
            {'action': action, 'consumer': self.consumer},
        ):
            save_phone.commit()

        save_phone.after_commit()


class LogbrokerTest(BaseBundleView):
    basic_form = LogbrokerTestForm()
    required_grants = ['internal.logbroker_test']

    @cached_property
    def logbroker(self):
        return get_api_logbroker_writer('logbroker_test')

    def process_request(self):
        self.process_basic_form()
        message = LogbrokerTestMessage(
            id=self.form_values['id'],
            test_data=self.form_values['test_data'],
            sub_message=LogbrokerTestMessage.SubMessage(
                int=self.form_values['sub_message_int'],
                str=self.form_values['sub_message_str'],
            ),
        )
        log.debug('Sending test message id={}: {}'.format(message.id, message))
        self.logbroker.send(payload=message)


class AMPushMessage(BaseBundleView):
    basic_form = AmPushMessageTestForm()
    required_grants = ['internal.am_push_message_test']

    @cached_property
    def logbroker(self):
        return get_api_logbroker_writer('challenge_pushes')

    def process_request(self):
        self.process_basic_form()
        push_request = PushRequest(
            push_service=self.form_values['push_service'],
            event_name=self.form_values['event_name'],
            recipients=PushRequestRecipient(
                uid=self.form_values['uid'],
                app_targeting_type=(
                    getattr(AppTargetingTypes, self.form_values['app_targeting_type'])
                    if self.form_values['app_targeting_type'] else None
                ),
                custom_app_priority=self.form_values['custom_app_priority'] or None,
                required_am_capabilities=self.form_values['am_capabilities'] or None,
                required_platforms=(
                    [
                        Platforms[platform]
                        for platform in self.form_values['required_platforms']
                    ]
                    if self.form_values['required_platforms'] else None
                ),
                device_ids=self.form_values['device_ids'] or None,
            ),
            title=self.form_values['title'],
            body=self.form_values['body'],
            subtitle=self.form_values['subtitle'],
            webview_url=self.form_values['webview_url'],
            require_web_auth=self.form_values['require_web_auth'],
            push_id=self.form_values['push_id'],
        )
        log.debug('Sending test message {} for uid {}'.format(
            push_request.push_id,
            self.form_values['uid'],
        ))
        self.logbroker.send(
            ChallengePushRequest(
                push_message_request=push_request.to_proto(),
            ),
        )


__all__ = (
    'AMPushMessage',
    'ConfirmAndBindPhone',
    'DeleteYakeyBackup',
    'FinishAccountDeletion',
    'GetPhones',
    'LiteAccountRegister',
    'SetPassword',
    'UpdateAccountDeletionOperation',
    'UpdateAccountRegistrationDatetime',
    'UpdateEmail',
    'UpdatePhone',
    'LogbrokerTest',
)
