# -*- coding: utf-8 -*-
from passport.backend.api.forms import UidForm
from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.exceptions import ActionNotRequiredError
from passport.backend.api.views.bundle.headers import HEADER_CONSUMER_CLIENT_IP
from passport.backend.api.views.bundle.mixins import BundleAccountGetterMixin
from passport.backend.core.runner.context_managers import UPDATE
from passport.backend.core.types.totp_secret import TotpSecretType

from .forms import TotpEditForm


class TotpBase(BaseBundleView,
               BundleAccountGetterMixin):

    required_headers = (
        HEADER_CONSUMER_CLIENT_IP,
    )

    required_grants = ['internal.totp_edit']

    require_track = False

    def _process_request(self):
        raise NotImplementedError()  # pragma: no cover

    def process_request(self):

        self.process_basic_form()

        self.get_account_by_uid(self.form_values['uid'])
        self._process_request()
        self.response_values.update(uid=self.account.uid)


class TotpEdit(TotpBase):

    basic_form = TotpEditForm

    def _process_request(self):
        events = {
            'consumer': self.consumer,
            'action': 'test_totp_edit',
        }

        with UPDATE(self.account, self.request.env, events):
            self.account.totp_secret.set(TotpSecretType(encrypted_pin_and_secret=self.form_values['totp_secret']))


class TotpDelete(TotpBase):

    basic_form = UidForm

    def _process_request(self):
        # Сейчас эта ручка восстановит фактически вебовский пароль
        events = {
            'consumer': self.consumer,
            'action': 'test_totp_delete',
        }
        if not self.account.totp_secret.is_set:
            raise ActionNotRequiredError()
        with UPDATE(self.account, self.request.env, events):
            self.account.totp_secret = None


class TotpGenerate(BaseBundleView):

    required_grants = ['internal.totp_edit']

    def process_request(self):
        totp_secret = TotpSecretType.generate()
        if self.track_id:
            self.read_track()
            with self.track_transaction.commit_on_error():
                self.track.totp_pin = totp_secret.pin
                self.track.totp_app_secret = totp_secret.human_readable_app_secret

        self.response_values.update(
            pin=totp_secret.pin,
            app_secret=totp_secret.human_readable_app_secret,
        )


__all__ = (
    'TotpEdit',
    'TotpDelete',
    'TotpGenerate',
)
