# -*- coding: utf-8 -*-
from passport.backend.api.common.processes import ALL_PROCESSES
from passport.backend.api.forms import UidForm
from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.headers import HEADER_CONSUMER_CLIENT_IP
from passport.backend.api.views.bundle.mixins import BundleAccountGetterMixin
from passport.backend.core.types.login.login import is_test_yandex_login
from passport.backend.utils.string import mask_postfix

from .forms import SaveTrackForm


class GetTrackView(BaseBundleView):

    required_headers = (
        HEADER_CONSUMER_CLIENT_IP,
    )

    required_grants = ['internal.track']

    require_track = True

    allowed_processes = ALL_PROCESSES

    def process_request(self):
        self.read_track()
        self.response_values.update(self.track._data)
        self.response_values.update(self.track._lists)
        self.response_values.update(self.track._counters)


class SaveTrackView(BaseBundleView):

    required_headers = (
        HEADER_CONSUMER_CLIENT_IP,
    )

    required_grants = ['internal.track']

    require_track = True

    basic_form = SaveTrackForm

    allowed_processes = ALL_PROCESSES

    def process_request(self):
        self.read_track()
        self.process_basic_form()
        with self.track_transaction.rollback_on_error() as track:
            track.parse(self.form_values)


class GetPersistentTracks(BaseBundleView, BundleAccountGetterMixin):
    required_grants = ['internal.get_persistent_tracks']
    basic_form = UidForm

    def process_request(self):
        self.process_basic_form()
        uid = self.form_values['uid']

        self.get_account_by_uid(uid, enabled_required=False)

        tracks = self.blackbox.get_all_tracks(uid)
        if not is_test_yandex_login(self.account.login):
            for track in tracks:
                track['track_id'] = mask_postfix(track['track_id'], len(track['track_id']) - 4)
                track['content'] = {
                    field: value if field == 'type' else '*' for (field, value) in track['content'].items()
                }

        self.response_values['persistent_tracks'] = tracks


__all__ = (
    'GetTrackView',
    'SaveTrackView',
    'GetPersistentTracks',
)
