# -*- coding: utf-8 -*-
from passport.backend.api.common import log_internal_error
from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.mixins import BundleAccountGetterMixin
from passport.backend.core.builders.historydb_api import (
    BaseHistoryDBApiError,
    HistoryDBApiTemporaryError,
)
from passport.backend.core.mail_subscriptions.services_manager import get_mail_subscription_services_manager
from passport.backend.core.runner.context_managers import UPDATE
from passport.backend.core.utils.decorators import cached_property

from .forms import (
    GetMailSubscriptionsForm,
    SetMailSubscriptionsForm,
)


MAX_HISTORYDB_AGE = 6 * 30 * 24 * 3600  # 6 месяцев


class BaseMailSubscriptionView(BaseBundleView, BundleAccountGetterMixin):
    @cached_property
    def mail_subscription_services_manager(self):
        return get_mail_subscription_services_manager()


class GetCurrentSubscriptions(BaseMailSubscriptionView):
    basic_form = GetMailSubscriptionsForm
    required_grants = ['mail_subscriptions.get']

    def process_request(self):
        self.process_basic_form()

        self.get_account_from_session(
            multisession_uid=self.form_values['uid'],
            get_user_ticket=True,
        )
        all_services = self.mail_subscription_services_manager.get_all_services()

        try:
            list_id_to_ts = self.historydb_api.last_letter(
                user_ticket=self.tvm_user_ticket,
                max_age=MAX_HISTORYDB_AGE,
            )
        except HistoryDBApiTemporaryError:
            list_id_to_ts = None
        except BaseHistoryDBApiError as e:
            log_internal_error(e)
            list_id_to_ts = None

        response_struct = dict()
        for service in all_services:
            if service.id in self.account.unsubscribed_from_maillists:
                is_subscribed = False
            else:
                is_subscribed = True

            if list_id_to_ts is not None:
                email_sent_ts = max(
                    [
                        list_id_to_ts.get(list_id, 0)
                        for list_id in service.external_list_ids
                    ] or [0]
                )
            else:
                email_sent_ts = None

            response_struct[service.id] = {
                'slug': service.slug,
                'is_subscribed': is_subscribed,
                'email_sent_ts': email_sent_ts,
            }
        self.response_values['services'] = response_struct


class SetCurrentSubscriptions(BaseMailSubscriptionView):
    basic_form = SetMailSubscriptionsForm
    required_grants = ['mail_subscriptions.set']

    def process_request(self):
        self.process_basic_form()

        self.get_account_from_session(
            multisession_uid=self.form_values['uid'],
        )

        subscriptions_json = self.form_values['subscriptions_json']

        possible_services = {s.id for s in self.mail_subscription_services_manager.get_all_services()}
        actual_subscriptions = {
            s_id: subscriptions_json.get(s_id, True) for s_id in possible_services
        }

        with UPDATE(self.account, self.request.env, {'action': 'mail_subscriptions'}):
            self.account.unsubscribed_from_maillists.set_by_dict(actual_subscriptions)
