# -*- coding: utf-8 -*-

import collections
import random
import uuid

from passport.backend.api.common.authorization import (
    bb_response_to_session,
    build_auth_cookies,
    build_cookie_i,
    build_cookie_mda2_beacon,
    build_cookie_my,
    build_cookie_yandex_login,
    build_cookie_yandexuid,
    build_cookies_noauth,
    build_cookie_yandex_gid,
    build_cookies_yx,
    get_session_policy_by_ttl,
    is_user_session_valid,
    try_update_cookie_lah,
    update_cookie_mda2_domains,
)
from passport.backend.api.env import APIEnvironment
from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.exceptions import ValidationFailedError
from passport.backend.api.views.bundle.headers import (
    HEADER_CLIENT_COOKIE,
    HEADER_CLIENT_HOST,
    HEADER_CONSUMER_CLIENT_IP,
)
from passport.backend.api.views.bundle.mda2.container import MDA2Container
from passport.backend.api.views.bundle.mda2.exceptions import (
    ContainerInvalidError,
    HostNotSlaveError,
    TargetHostInvalidError,
    TargetHostNotSlaveError,
)
from passport.backend.api.views.bundle.mda2.forms import (
    BuildContainerForm,
    UseContainerForm,
)
from passport.backend.api.views.bundle.mixins import (
    BundleAccountGetterMixin,
    CookieCheckStatus,
)
from passport.backend.api.views.bundle.utils import assert_valid_host
from passport.backend.core.conf import settings
from passport.backend.core.cookies.consts import (
    I_COOKIE_NAME,
    MDA2_BEACON_COOKIE_NAME,
    MY_COOKIE_NAME,
    SESSION_ID_COOKIE_NAME,
    YANDEX_GID_COOKIE_NAME,
    YANDEXUID_COOKIE_NAME,
    YP_COOKIE_NAME,
    YS_COOKIE_NAME,
)
from passport.backend.core.cookies.cookie_y import (
    PermanentCookieYContainer,
    SessionCookieYContainer,
)
from passport.backend.core.cookies.utils import MAX_COOKIE_EXPIRATION_TIMESTAMP
from passport.backend.core.encrypted_container import EncryptedContainerInvalid
from passport.backend.core.exceptions import WrongHostError
from passport.backend.core.logging_utils.loggers import StatboxLogger
from passport.backend.core.utils.decorators import cached_property
from passport.backend.core.utils.domains import (
    get_cookie_domain_by_host,
    get_keyspace_by_host,
)
from passport.backend.core.utils.experiments import is_experiment_enabled
from passport.backend.utils.common import deep_merge
from passport.backend.utils.string import (
    always_str,
    smart_text,
)
from six.moves.http_cookies import SimpleCookie
from six.moves.urllib.parse import urlparse


BUILD_CONTAINER_GRANT = 'mda2.build_container'
USE_CONTAINER_GRANT = 'mda2.use_container'

MAX_COOKIE_CHECK_VALUE = 2 ** 32 - 1

AUTH_STATUS_EMPTY = 'empty'
AUTH_STATUS_NOAUTH = 'noauth'
AUTH_STATUS_AUTH = 'auth'


class BaseMda2View(BaseBundleView, BundleAccountGetterMixin):
    required_headers = (
        HEADER_CLIENT_HOST,
        HEADER_CONSUMER_CLIENT_IP,
        HEADER_CLIENT_COOKIE,
    )

    statbox_action = None

    @property
    def statbox_event(self):
        raise NotImplementedError()  # pragma: no cover

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode='mda2',
            consumer=self.consumer,
            action=self.statbox_action,
            event=self.statbox_event,
            ip=self.client_ip,
            user_agent=self.user_agent,
            yandexuid=self.cookies.get(YANDEXUID_COOKIE_NAME),
            process_uuid=self.form_values['process_uuid'],
        )

    def make_cookie_check_value(self):
        return str(random.SystemRandom().randint(1, MAX_COOKIE_CHECK_VALUE))

    def make_script_nonce(self):
        return uuid.uuid4().get_hex()

    def build_csp_header(self, host, script_nonce=None):
        # Разрешаем открывать эту страницу в iframe указанного домена
        header = "default-src 'none'; frame-ancestors https://*.%s" % get_keyspace_by_host(host)
        if script_nonce is not None:
            # Разрешаем JS-скрипты, но лишь сопровождаемые данным nonce
            header += "; connect-src 'self'; script-src 'nonce-%s'" % script_nonce
        return header

    def get_domain_settings(self, target_host):
        target_domain = get_keyspace_by_host(target_host)
        return deep_merge(
            settings.BASE_MDA2_SLAVE_DOMAIN_CONFIG,
            settings.CUSTOM_MDA_DOMAIN_CONFIGS.get(target_domain, {}),
        )

    def validate_target_host(self, target_host):
        try:
            keyspace = get_keyspace_by_host(target_host)
        except WrongHostError:
            raise TargetHostInvalidError()

        if keyspace not in settings.MDA2_SLAVE_DOMAINS:
            raise TargetHostNotSlaveError()

        return target_host

    @cached_property
    def value_for_experiment(self):
        yandexuid = self.cookies.get(YANDEXUID_COOKIE_NAME)
        if not yandexuid:
            return
        try:
            yandexuid = int(yandexuid)
        except (TypeError, ValueError):
            return
        return yandexuid

    def experimental_cookie_enabled(self):
        return (
            self.value_for_experiment is not None and
            is_experiment_enabled(
                self.value_for_experiment,
                settings.MDA_EXPERIMENTAL_COOKIE_DENOMINATOR,
            )
        )

    def process_request(self):
        assert_valid_host(self.request.env)
        self.process_basic_form()
        self._process()

    def _process(self):
        raise NotImplementedError()  # pragma: no cover


class BuildContainerView(BaseMda2View):
    """
    Выполняется на корневом домене, собирает контейнер с куками для подчинённого домена
    """
    required_grants = [BUILD_CONTAINER_GRANT]
    basic_form = BuildContainerForm
    statbox_action = 'build_container'

    def __init__(self):
        super(BuildContainerView, self).__init__()

        # Срок жизни куки session_id для подчиненного домена
        # Используем для установки времени куки mda2_beacon
        # Определяем в build_auth_cookies_for_host
        self.target_session_expires = None

    @property
    def statbox_event(self):
        return 'pull' if self.form_values['is_background'] else 'push'

    def build_custom_env(self, host=None, ignore_ys=False):
        env = self.request.env
        cookies = env.cookies
        if ignore_ys:
            # Особый случай: кука ys используется нами, в ответе она окажется в любом случае. А вот учитывать
            # значение куки ys текущего окружения нужно не всегда.
            cookies.pop(YS_COOKIE_NAME, None)
        return APIEnvironment(
            cookies=cookies,
            cookies_all=cookies,
            consumer_ip=env.consumer_ip,
            user_ip=env.user_ip,
            user_agent=env.user_agent,
            host=host or env.host,
            accept_language=env.accept_language,
            referer=env.referer,
            authorization=env.authorization,
            service_ticket=env.service_ticket,
            request_id=env.request_id,
        )

    def build_auth_cookies_for_host(self, session_info, target_env=None):
        env = target_env or self.request.env
        is_master_domain = target_env is None

        bb_response = session_info.response
        if is_master_domain:
            session, _ = bb_response_to_session(bb_response, env)
            if not session:
                # Куки свежие, подновления не требуют
                return []
        else:
            resigned_cookies = bb_response['resigned_cookies'][env.host]
            session, _ = bb_response_to_session(resigned_cookies, env)
            # Куку sessionid2 ставим только на корневой домен: на подчинённом достаточно Session_Id
            session.pop('sslsession', None)
            # Sessguard на подчинённом не нужен. Сейчас ЧЯ его и не отдаёт, но перестрахуемся.
            session.pop('sessguard', None)
        cookies = build_auth_cookies(env, session)

        cookies.append(
            build_cookie_mda2_beacon(
                env,
                expires=session['session']['expires'],
            ),
        )

        if target_env is not None:
            self.target_session_expires = session['session']['expires']

        if is_user_session_valid(bb_response):
            # Если сессия дефолтного пользователя тоже валидна - надо выставить куку yandex_login
            # (срок её жизни обычно совпадает со сроком жизни session_id) и подновить куку lah
            self.parse_account(
                bb_response,
                enabled_required=False,
            )
            cookies.append(
                build_cookie_yandex_login(
                    env,
                    human_readable_login=self.account.human_readable_login,
                ),
            )
            if is_master_domain:
                # Кука lah бессмысленна на подчинённых доменах - там нет паспорта
                cookie_lah = try_update_cookie_lah(
                    env=env,
                    uid=self.account.uid,
                    auth_method=None,
                    authorization_session_policy=get_session_policy_by_ttl(session_info.ttl),
                )
                if cookie_lah:
                    cookies.append(cookie_lah)

        return cookies

    def build_non_auth_cookies_for_host(self, cookie_check_value, target_env=None):
        env = target_env or self.request.env
        is_master_domain = target_env is None

        if is_master_domain or not self.experimental_cookie_enabled():
            need_yp = False
        else:
            need_yp = self.get_domain_settings(env.host)['cookies'].get(YP_COOKIE_NAME)
        cookies = build_cookies_yx(
            env,
            cookie_check_value=cookie_check_value,
            display_name=self.account.person.display_name if self.account else None,
            need_yp=need_yp,
        )

        if not is_master_domain:
            # Трекинговые куки на мастер-домене подновлять не будем, это не паспортная зона ответственности
            if env.cookies.get(I_COOKIE_NAME):
                cookies.append(build_cookie_i(env))
            if env.cookies.get(YANDEXUID_COOKIE_NAME):
                cookies.append(build_cookie_yandexuid(env))

            if self.experimental_cookie_enabled():
                if self.get_domain_settings(env.host)['cookies'].get(MY_COOKIE_NAME):
                    cookies.append(build_cookie_my(env))
                if self.get_domain_settings(env.host)['cookies'].get(YANDEX_GID_COOKIE_NAME):
                    cookies.append(build_cookie_yandex_gid(env))

        return cookies

    def postprocess_slave_cookies(self, cookies, target_host):
        slave_domain_settings = self.get_domain_settings(target_host)
        processed_cookies = []
        for cookie in cookies:
            cookie_name, _, cookie_value = cookie.partition('=')
            target_cookie_name = slave_domain_settings['cookies'].get(cookie_name)
            if not target_cookie_name:
                if cookie_name == YS_COOKIE_NAME:
                    # Кука ys используется нами, в ответе она должна оказаться в любом случае, независимо от настроек
                    target_cookie_name = cookie_name
                else:
                    continue
            processed_cookie = '='.join([target_cookie_name, cookie_value])
            processed_cookies.append(processed_cookie)
        return processed_cookies

    def _process(self):
        target_host = self.validate_target_host(self.form_values['target_host'])
        target_domain_settings = self.get_domain_settings(target_host)
        ignore_ys = not target_domain_settings['cookies'].get(YS_COOKIE_NAME)
        target_env = self.build_custom_env(host=target_host, ignore_ys=ignore_ys)

        script_nonce = self.make_script_nonce()
        self.response_values.update(
            headers={
                'Content-Security-Policy': self.build_csp_header(
                    host=target_host,
                    script_nonce=script_nonce,
                ),
            },
            script_nonce=script_nonce,
        )

        cookies_for_current_host = []
        cookies_for_container = []

        session_info = self.check_session_cookie(
            prolong_cookies=True,
            resign_for_domains=[target_host],
        )
        is_session_cookie_valid = session_info.cookie_status in (
            CookieCheckStatus.Valid,
            CookieCheckStatus.NeedReset,
        )

        # Синхронизируем авторизационные куки
        if is_session_cookie_valid:
            cookies_for_container += self.build_auth_cookies_for_host(session_info, target_env=target_env)
            cookies_for_current_host += self.build_auth_cookies_for_host(session_info, target_env=None)
            auth_status = AUTH_STATUS_AUTH
        else:
            cookies_for_container += build_cookies_noauth(target_env)
            auth_status = AUTH_STATUS_NOAUTH if self.request.env.cookies else AUTH_STATUS_EMPTY
            if not self.request.env.cookies.get(MDA2_BEACON_COOKIE_NAME):
                # PASSP-26181 Если на головном домене нет сессии и куки mda2_beacon, то ставим куку на головной домен
                cookies_for_current_host.append(
                    build_cookie_mda2_beacon(self.request.env),
                )

        # Синхронизируем общепортальные куки
        cookie_check_value = self.make_cookie_check_value()
        cookies_for_container += self.build_non_auth_cookies_for_host(cookie_check_value, target_env=target_env)
        cookies_for_current_host += self.build_non_auth_cookies_for_host(cookie_check_value, target_env=None)

        # Запишем текущий подчинённый домен в куку (пригодится в mda 2.1)
        cookies_for_current_host.append(
            update_cookie_mda2_domains(self.request.env, target_host),
        )

        # Поправим куки для подчинённого домена (некоторые может потребоваться выкинуть, некоторые - переименовать)
        cookies_for_container = self.postprocess_slave_cookies(cookies_for_container, target_host=target_host)

        # Добавляем куку mda2_beacon для подчиненного домена
        cookies_for_container.append(
            build_cookie_mda2_beacon(
                target_env,
                domain=get_cookie_domain_by_host(target_host),
                expires=self.target_session_expires if is_session_cookie_valid else MAX_COOKIE_EXPIRATION_TIMESTAMP,
            ),
        )

        dst_domain = get_keyspace_by_host(target_host)
        container = MDA2Container(
            data=dict(
                src_domain=get_keyspace_by_host(self.request.env.host),
                dst_domain=dst_domain,
                auth_cookie_valid=auth_status == AUTH_STATUS_AUTH,
                cookies=cookies_for_container,
                cookie_check_value=cookie_check_value,
            ),
        )
        self.response_values.update(
            cookies=cookies_for_current_host,
            container=container.pack(),
            auth_status=auth_status,
        )
        if not self.request.env.cookies:
            # Если в запросе не было кук - возможно, они заблокированы. Фронту стоит проверить это сразу,
            # не дожидаясь второго шага.
            self.response_values.update(cookie_check_value=cookie_check_value)

        self.statbox.log(
            status='ok',
            auth_status=auth_status,
            host=dst_domain,
        )


class UseContainerView(BaseMda2View):
    """
    Выполняется на подчинённом домене, выставляет куки из переданного контейнера
    """
    required_grants = [USE_CONTAINER_GRANT]
    basic_form = UseContainerForm
    statbox_action = 'use_container'

    def __init__(self):
        super(UseContainerView, self).__init__()

        self.domain_settings = None

    @property
    def statbox_event(self):
        return 'inject' if self.form_values['is_background'] else 'install'

    def validate_retpath(self, retpath, container):
        # Дополнительно проверяем, что retpath находится на том же домене 2 уровня,
        # что и текущий host и хост из контейнера
        keyspaces = {
            get_keyspace_by_host(urlparse(retpath).hostname),
            get_keyspace_by_host(self.request.env.host),
            container['dst_domain'],
        }
        if len(keyspaces) > 1:
            raise ValidationFailedError(['retpath.invalid'])
        if keyspaces.pop() not in settings.MDA2_SLAVE_DOMAINS:
            raise HostNotSlaveError()
        return retpath

    def _get_old_auth_cookie_status(self):
        target_host = self.validate_target_host(self.request.env.host)
        target_domain_settings = self.get_domain_settings(target_host)
        session_id = self.cookies.get(target_domain_settings['cookies'][SESSION_ID_COOKIE_NAME])

        session_info = self._check_session_cookie(
            session_cookie_body=session_id,
            ssl_session_cookie_body=None,
            sessguard_body=None,
            host=target_host,
        )
        if session_info.cookie_status in (CookieCheckStatus.Valid, CookieCheckStatus.NeedReset):
            return (
                AUTH_STATUS_AUTH,
                session_info,
            )

        return (
            AUTH_STATUS_NOAUTH if self.request.env.cookies else AUTH_STATUS_EMPTY,
            session_info,
        )

    def _shake_cookies(self, container):
        cookies = collections.OrderedDict()
        for serialized_cookie in container['cookies']:
            cookie = SimpleCookie()
            cookie.load(always_str(serialized_cookie))
            for cookie_name in cookie:
                cookies.setdefault(cookie_name, cookie[cookie_name])

        for cookie_name in [
            MY_COOKIE_NAME,
            YANDEX_GID_COOKIE_NAME,
            YP_COOKIE_NAME,
        ]:
            self._ignore_cookie_if_not_configured(cookies, cookie_name)

        self._merge_ys_cookie_if_not_configured(cookies)

        container['cookies'] = [c.OutputString() for c in cookies.values()]

    def _ignore_cookie_if_not_configured(self, cookies, cookie_name):
        # Игнорирование куки равносильно оставлению её старого значения
        if (
            self.domain_settings['cookies'].get(cookie_name) is None or
            not self.experimental_cookie_enabled()
        ):
            cookies.pop(cookie_name, None)

    def _merge_ys_cookie_if_not_configured(self, cookies):
        if (
            self.domain_settings['cookies'].get(YS_COOKIE_NAME) is None and
            self.request.env.cookies.get(YS_COOKIE_NAME)
        ):
            self._merge_yx_cookies(YS_COOKIE_NAME, cookies)

    def _merge_yx_cookies(self, cookie_name, container_cookies):
        if cookie_name == YS_COOKIE_NAME:
            CookieYContainer = SessionCookieYContainer
        elif cookie_name == YP_COOKIE_NAME:
            CookieYContainer = PermanentCookieYContainer
        else:
            raise NotImplementedError()

        merged_cookie = CookieYContainer()

        for _, v in CookieYContainer().parse(self.request.env.cookies.get(cookie_name)):
            merged_cookie.insert(**v)

        for _, v in CookieYContainer().parse(container_cookies[cookie_name].value):
            merged_cookie.insert(**v)

        serialized_merged_cookie = merged_cookie.serialize()
        container_cookies[cookie_name].set(cookie_name, serialized_merged_cookie, smart_text(serialized_merged_cookie))

    def _process(self):
        script_nonce = self.make_script_nonce()
        self.response_values.update(
            headers={
                'Referrer-Policy': 'origin',  # отправлять в Referer не весь урл, а лишь его домен
                'Content-Security-Policy': self.build_csp_header(
                    host=self.request.env.host,
                    script_nonce=script_nonce,
                ),
            },
            script_nonce=script_nonce,
        )

        try:
            container = MDA2Container.unpack(self.form_values['container'])
        except EncryptedContainerInvalid as e:
            raise ContainerInvalidError(e)

        # Для install валидируем retpath всегда, для inject - только если фронт его передал
        if not self.form_values['is_background'] or self.form_values['retpath']:
            retpath = self.validate_retpath(retpath=self.form_values['retpath'], container=container)
            self.response_values.update(retpath=retpath)

        self.domain_settings = self.get_domain_settings(container['dst_domain'])

        self._shake_cookies(container)

        self.response_values.update(
            origin=container['src_domain'],
            current_domain=container['dst_domain'],
            auth_cookie_valid=container['auth_cookie_valid'],
            cookies=container['cookies'],
            cookie_check_value=container['cookie_check_value'],
        )

        old_auth_status, old_auth_bb_session_info = self._get_old_auth_cookie_status()

        self.statbox.log(
            status='ok',
            auth_status=AUTH_STATUS_AUTH if container['auth_cookie_valid'] else AUTH_STATUS_NOAUTH,
            old_auth_status=old_auth_status,
            old_auth_bb_status=str(old_auth_bb_session_info.error),
            host=container['dst_domain'],
        )
