# -*- coding: utf-8 -*-
from datetime import datetime
import time

from passport.backend.api.views.bundle.exceptions import (
    AccountNotFoundError,
    DomainInvalidTypeError,
)
from passport.backend.api.views.bundle.mdapi.base import BasePddView
from passport.backend.api.views.bundle.mdapi.exceptions import (
    DomainAliasAlreadyExistsError,
    DomainAliasNotFoundError,
    DomainAlreadyExistsError,
    DomainMasterIsAliasError,
    DomainNotFoundError,
    DomainUsersRemainingError,
)
from passport.backend.api.views.bundle.mdapi.forms import (
    PddAddDomainAliasForm,
    PddAddDomainForm,
    PddAliasToMasterForm,
    PddDeleteDomainAliasForm,
    PddDeleteDomainForm,
    PddEditDomainForm,
)
from passport.backend.api.views.bundle.mixins import BundleFederalMixin
from passport.backend.core.models.account import Service
from passport.backend.core.models.domain import Domain
from passport.backend.core.models.swap_domains_batch import SwapDomainsBatch
from passport.backend.core.runner.context_managers import (
    CREATE,
    DELETE,
    UPDATE,
)
from passport.backend.core.subscription import (
    add_subscription,
    delete_subscription,
)


DOMAIN_ADD_GRANT = 'domain.add'
DOMAIN_EDIT_GRANT = 'domain.edit'
DOMAIN_DELETE_GRANT = 'domain.delete'


class PddAddDomainView(BasePddView):
    basic_form = PddAddDomainForm
    required_grants = [DOMAIN_ADD_GRANT]

    def does_domain_exist(self, domain_or_id):
        try:
            self.get_domain(domain_or_id)
            return True
        except DomainNotFoundError:
            return False

    def process_request(self):
        self.process_basic_form()
        admin_uid = self.form_values['admin_uid']
        domain_to_add = self.form_values['domain']

        if self.does_domain_exist(domain_to_add):
            raise DomainAlreadyExistsError()

        # FIXME: согласовать с ПДД исправление такого поведения,
        # когда отключенный пользователь может добавлять домены
        self.get_account_by_uid(admin_uid, enabled_required=False)

        if not self.account.is_pdd_admin:
            with UPDATE(self.account, self.request.env, {'action': 'make_pdd_admin', 'consumer': self.consumer}):
                add_subscription(
                    self.account,
                    Service.by_slug('pddadmin'),
                )

        new_domain = Domain()
        with CREATE(new_domain, self.request.env, {'action': 'pdd_add_domain', 'consumer': self.consumer}):
            new_domain.domain = domain_to_add
            new_domain.admin_uid = admin_uid
            new_domain.is_yandex_mx = self.form_values['mx']
            new_domain.is_enabled = self.form_values['enabled']
            new_domain.registration_datetime = datetime.now()
            new_domain.master_domain = None
            new_domain.default_uid = 0

        self.statbox.log(
            action='pdd_add_domain',
            admin_uid=admin_uid,
            domain=new_domain.punycode_domain,
            domain_id=new_domain.id,
            mx=new_domain.is_yandex_mx,
            enabled=new_domain.is_enabled,
        )
        self.response_values.update({
            'domain': new_domain.domain,
            'domain_id': new_domain.id,
        })


class PddEditDomainView(BasePddView):
    basic_form = PddEditDomainForm
    required_grants = [DOMAIN_EDIT_GRANT]

    def process_request(self):
        self.process_basic_form()

        admin_uid = self.form_values['admin_uid']
        default_user = self.form_values['default']
        need_glogout = self.form_values['glogouted']
        display_master_id = self.form_values['display_master_id']
        organization_name = self.form_values['organization_name']
        mx = self.form_values['mx']
        is_enabled = self.form_values['enabled']
        can_users_change_password = self.form_values['can_users_change_password']
        default_uid = None
        old_admin = new_admin = None

        domain = self.get_domain(
            self.form_values['domain_id'],
            aliases=True,
        )

        if display_master_id:
            display_master = self.get_domain(display_master_id)
            # "Отображаемый" домен должен быть подчиненным редактируемому домену.
            if display_master.domain not in domain.aliases:
                raise DomainAliasNotFoundError()

        if default_user is not None:
            if default_user:
                self.get_account_by_login('%s@%s' % (default_user, domain.domain))
                default_uid = self.account.uid
            else:
                default_uid = 0

        # Логика смены администратора разбита по частям
        if admin_uid and domain.admin_uid != admin_uid:
            # У свежесозданных алиасов админ может быть не указан
            if domain.admin_uid:
                try:
                    self.get_account_by_uid(domain.admin_uid, enabled_required=False)
                    old_admin = self.account
                except AccountNotFoundError:
                    # Очень старым админам могли не выставлять блокирующий сид, так что они могли удалиться
                    pass

            self.get_account_by_uid(admin_uid, enabled_required=False)
            new_admin = self.account

            with UPDATE(new_admin, self.request.env, {'action': 'change_pdd_admin', 'consumer': self.consumer}):
                add_subscription(
                    new_admin,
                    Service.by_slug('pddadmin'),
                )

        with UPDATE(domain, self.request.env, {'action': 'pdd_edit_domain', 'consumer': self.consumer}):
            if is_enabled is not None:
                domain.is_enabled = is_enabled
            if can_users_change_password is not None:
                domain.can_users_change_password = can_users_change_password
            if display_master_id is not None:
                domain.display_master_id = display_master_id or None
            if organization_name is not None:
                domain.organization_name = organization_name or None
            if mx is not None:
                domain.is_yandex_mx = mx
            if default_uid is not None:
                domain.default_uid = default_uid
            if admin_uid:
                domain.admin_uid = admin_uid
            if need_glogout:
                domain.glogout_time = int(time.time())

        self.statbox.log(
            action='pdd_domain_edit',
            domain=domain.punycode_domain,
            domain_id=domain.id,
            admin_uid=domain.admin_uid,
            mx=domain.is_yandex_mx,
            enabled=domain.is_enabled,
            default_uid=domain.default_uid,
            display_master_id=domain.display_master_id or None,
            can_users_change_password=domain.can_users_change_password,
            glogout_time=domain.glogout_time or None,
        )

        if new_admin and old_admin:
            remaining = self.blackbox.hosted_domains(domain_admin=old_admin.uid)
            # Если удаляемый домен является последним из администрируемых
            # пользователем, то нужно убрать у него соответствующий SID.

            if not remaining['hosted_domains']:
                with UPDATE(old_admin, self.request.env, {'action': 'change_pdd_admin', 'consumer': self.consumer}):
                    delete_subscription(
                        old_admin,
                        Service.by_slug('pddadmin'),
                    )


class PddDeleteDomainView(BasePddView, BundleFederalMixin):
    basic_form = PddDeleteDomainForm
    required_grants = [DOMAIN_DELETE_GRANT]

    def process_request(self):
        self.process_basic_form()
        domain_id = self.form_values['domain_id']

        domain = self.get_domain(domain_id)
        # TODO: вернуть limit=0 после починки PASSP-21168
        pdd_accounts = self.blackbox.find_pdd_accounts(domain=domain.domain, limit=1)

        # Нельзя удалять домен, если на нем существует хотя бы один пользователь
        if pdd_accounts['total_count'] > 0:
            raise DomainUsersRemainingError()

        saml_settings = self.get_saml_settings(domain_id=domain.id)
        if saml_settings is not None:
            raise DomainInvalidTypeError()

        with DELETE(domain, self.request.env, {'action': 'delete_pdd_domain', 'consumer': self.consumer}):
            pass

        remaining = self.blackbox.hosted_domains(domain_admin=domain.admin_uid)
        # Если удаляемый домен является последним из администрируемых
        # пользователем, то нужно убрать у него соответствующий SID.
        if not remaining['hosted_domains']:
            try:
                self.get_account_by_uid(
                    domain.admin_uid,
                    enabled_required=False,
                )
            except AccountNotFoundError:
                pass
            else:
                admin = self.account

                with UPDATE(admin, self.request.env, {'action': 'remove_pdd_admin', 'consumer': self.consumer}):
                    delete_subscription(
                        admin,
                        Service.by_slug('pddadmin'),
                    )

        self.statbox.log(
            action='pdd_domain_delete',
            domain=domain.punycode_domain,
            domain_id=domain.id,
        )


class PddAddDomainAliasView(BasePddView):
    basic_form = PddAddDomainAliasForm
    required_grants = ['domain_alias.add']

    def process_request(self):
        self.process_basic_form()
        alias_host = self.form_values['alias']

        domain = self.get_domain(self.form_values['domain_id'])
        if domain.master_domain:
            raise DomainMasterIsAliasError()

        try:
            alias = self.get_domain(alias_host)
        except DomainNotFoundError:
            pass
        else:
            # Нужно выдавать разные ошибки в случае существования такого имени
            # в качестве алиаса и при существовании полноценного домена.
            if alias.master_domain:
                raise DomainAliasAlreadyExistsError()
            else:
                raise DomainAlreadyExistsError()

        with UPDATE(domain, self.request.env, {'action': 'pdd_domain_alias', 'consumer': self.consumer}):
            domain.aliases.append(alias_host)

        self.statbox.log(
            action='pdd_domain_alias',
            domain=domain.punycode_domain,
            domain_id=domain.id,
            alias=alias_host,
        )


class PddDeleteDomainAliasView(BasePddView):
    basic_form = PddDeleteDomainAliasForm
    required_grants = ['domain_alias.delete']

    def process_request(self):
        self.process_basic_form()

        alias_host = self.form_values['alias_id']
        domain = self.get_domain(
            self.form_values['domain_id'],
            aliases=True,
        )

        try:
            alias = self.get_domain(alias_host)
        except DomainNotFoundError:
            raise DomainAliasNotFoundError()

        if alias.domain not in domain.aliases:
            raise DomainAliasNotFoundError()

        with UPDATE(domain, self.request.env, {'action': 'pdd_domain_alias', 'consumer': self.consumer}):
            domain.alias_to_id_mapping[alias.domain] = alias.id
            domain.aliases.remove(alias.domain)

        self.statbox.log(
            action='pdd_domain_unalias',
            domain=domain.punycode_domain,
            alias=alias_host,
        )


class PddAliasToMasterView(BasePddView):
    basic_form = PddAliasToMasterForm
    required_grants = ['domain_alias.to_master']

    def process_request(self):
        self.process_basic_form()

        alias_id = self.form_values['alias_id']
        domain = self.get_domain(
            self.form_values['domain_id'],
            aliases=True,
        )

        try:
            alias = self.get_domain(alias_id)
        except DomainNotFoundError:
            raise DomainAliasNotFoundError()

        if alias.domain not in domain.aliases:
            raise DomainAliasNotFoundError()

        events = {'action': 'pdd_alias_to_master', 'consumer': self.consumer}
        with CREATE(SwapDomainsBatch(), self.request.env, events) as batch:
            batch.master_domain = domain
            batch.alias = alias

        self.statbox.log(
            action='pdd_alias_to_master',
            new_master_domain=alias.punycode_domain,
            new_alias=domain.punycode_domain,
        )
