# -*- coding: utf-8 -*-
from collections import namedtuple
from copy import deepcopy
from datetime import datetime
import logging
import re

from namedlist import namedlist
from passport.backend.api.common.account import unsubscribe_from_maillists_if_nessesary
from passport.backend.api.common.authorization import (
    is_user_in_multisession,
    is_user_session_valid,
    user_from_multisession,
    users_from_multisession,
)
from passport.backend.api.common.logs import setup_log_prefix
from passport.backend.api.common.suggest import get_countries_suggest
from passport.backend.api.templatetags import escape_percents
from passport.backend.api.views.bundle.constants import (
    AUTHENTICATION_MEDIA_SESSION,
    AUTHENTICATION_MEDIA_TOKEN,
    AUTHENTICATION_MEDIA_UID,
    AUTHENTICATION_MEDIA_USER_TICKET,
    SESSIONID_SCOPE,
    X_TOKEN_OAUTH_SCOPE,
)
from passport.backend.api.views.bundle.exceptions import (
    AccountDisabledError,
    AccountDisabledOnDeletionError,
    AccountGlobalLogoutError,
    AccountInvalidTypeError,
    AccountNotFoundError,
    AccountSms2FAEnabledError,
    AccountUidMismatchError,
    AccountWithoutPasswordError,
    AuthorizationHeaderError,
    InvalidTrackStateError,
    OAuthTokenValidationError,
    RequestCredentialsAllMissingError,
    RequestCredentialsSeveralPresentError,
    SessguardInvalidError,
    SessionidInvalidError,
    UidNotInSessionError,
    AccountIsChildError,
)
from passport.backend.api.views.bundle.headers import (
    HEADER_CLIENT_COOKIE,
    HEADER_CLIENT_HOST,
    HEADER_CONSUMER_AUTHORIZATION,
)
from passport.backend.api.views.bundle.mixins.kolmogor import KolmogorMixin
from passport.backend.api.views.bundle.phone.helpers import dump_number
from passport.backend.api.views.bundle.states import RedirectToSocialCompletion
from passport.backend.api.views.bundle.utils import is_user_disabled_on_deletion
from passport.backend.api.yasms import api as yasms_api
from passport.backend.api.yasms.exceptions import YaSmsError
from passport.backend.core import Undefined
from passport.backend.core.builders.blackbox.constants import (
    BLACKBOX_ERROR_INVALID_CHARACTERS_IN_LOGIN,
    BLACKBOX_ERROR_LOGIN_EMPTY_DOMAIN_PART,
    BLACKBOX_FIND_BY_PHONE_ALIAS_FORCE_ON,
    BLACKBOX_OAUTH_VALID_STATUS,
    BLACKBOX_SESSIONID_DISABLED_STATUS,
    BLACKBOX_SESSIONID_EXPIRED_STATUS,
    BLACKBOX_SESSIONID_INVALID_STATUS,
    BLACKBOX_SESSIONID_NEED_RESET_STATUS,
    BLACKBOX_SESSIONID_NOAUTH_STATUS,
    BLACKBOX_SESSIONID_VALID_STATUS,
    BLACKBOX_SESSIONID_WRONG_GUARD_STATUS,
)
from passport.backend.core.builders.blackbox.exceptions import (
    BaseBlackboxError,
    BlackboxInvalidParamsError,
)
from passport.backend.core.builders.blackbox.utils import add_phone_arguments
from passport.backend.core.builders.historydb_api.exceptions import BaseHistoryDBApiError
from passport.backend.core.builders.social_api import (
    BaseSocialApiError,
    get_social_api,
)
from passport.backend.core.conf import settings
from passport.backend.core.counters import auth_email
from passport.backend.core.dbmanager.exceptions import DBError
from passport.backend.core.geobase import (
    get_country_code_by_ip,
    get_geobase,
    is_valid_country_code,
    Region,
)
from passport.backend.core.logging_utils.helpers import mask_sessionid
from passport.backend.core.logging_utils.loggers import SocialBindingLogger
from passport.backend.core.logging_utils.loggers.social_binding import BindPhonishAccountByTrackStatboxEvent
from passport.backend.core.logging_utils.loggers.statbox import to_statbox
from passport.backend.core.mailer.utils import (
    get_tld_by_country,
    login_shadower,
    MailInfo,
    make_email_context,
    send_mail_for_account,
)
from passport.backend.core.models.account import (
    Account,
    ACCOUNT_DISABLED_ON_DELETION,
    get_preferred_language,
    Service,
    UnknownUid,
)
from passport.backend.core.models.domain import Domain
from passport.backend.core.models.email import Emails
from passport.backend.core.models.password import PASSWORD_CHANGING_REASON_HACKED
from passport.backend.core.portallib import is_yandex_server_ip
from passport.backend.core.runner.context_managers import UPDATE
from passport.backend.core.services import get_service
from passport.backend.core.subscription import (
    add_subscription,
    can_be_subscribed,
    SubscriptionError,
)
from passport.backend.core.types.account.account import (
    ACCOUNT_TYPE_LITE,
    ACCOUNT_TYPE_SOCIAL,
)
from passport.backend.core.types.email.email import unicode_email
from passport.backend.core.types.gender import Gender
from passport.backend.core.types.login.login import is_test_yandex_login
from passport.backend.core.types.mobile_device_info import get_app_id_from_track
from passport.backend.core.types.phone_number.phone_number import (
    get_alt_phone_numbers_of_phone_number,
    InvalidPhoneNumber,
    PhoneNumber,
)
from passport.backend.core.utils.blackbox import get_many_accounts_by_uids
from passport.backend.core.utils.domains import build_passport_domain
from passport.backend.core.utils.experiments import is_experiment_enabled_by_uid
from passport.backend.core.yasms.phonenumber_alias import is_phonenumber_alias_as_email_allowed
from passport.backend.core.yasms.utils import get_many_accounts_with_phones_by_uids
from passport.backend.utils.common import remove_none_values
from passport.backend.utils.string import (
    smart_bytes,
    smart_text,
)
from passport.backend.utils.time import (
    datetime_to_integer_unixtime,
    datetime_to_integer_unixtime_nullable,
    get_unixtime,
    unixtime_to_datetime,
)
from six import string_types
from six.moves.urllib.parse import quote


RE_BLACKBOX_INVALID_LOGIN_ERROR = re.compile(
    r'^(?:{})'.format('|'.join([
        BLACKBOX_ERROR_LOGIN_EMPTY_DOMAIN_PART,
        BLACKBOX_ERROR_INVALID_CHARACTERS_IN_LOGIN,
        ])
    ),
)

PDD_DISPLAY_TEMPLATE = 't:%pdd_username%@%display_domain%'
PDD_DOMAIN_TEMPLATE = 't:%pdd_username%@%pdd_domain%'

LANGUAGE_TO_LOCALE = {
    'ru': 'ru_RU',
    'en': 'en_US',
    'ua': 'ua_UA',
    'tr': 'tr_TR',
}

CookieCheckStatus = namedtuple(
    'CookieCheckStatus',
    ['Invalid', 'Foreign', 'Valid', 'NeedReset', 'Disabled', 'WrongGuard']
)._make([0, 1, 2, 3, 4, 5])

SessionInfo = namedlist(
    'SessionInfo',
    [
        ('uid', None),
        ('ttl', None),
        ('authid', None),
        ('ip', None),
        ('create_timestamp', None),
        ('age', None),
        ('response', None),
        ('status', None),  # статус дефолтного аккаунта(дефолтной сессии)
        ('session', None),
        ('ssl_session', None),
        ('cookie_status', None),  # статус всей куки
        ('error', None),  # поле error из ответа ЧЯ
        ('sessguard', None),
        ('login_id', None),
    ],
)

PASSWORD_STRENGTH_STATUS_WEAK = 0
PASSWORD_STRENGTH_STATUS_STRONG = 1
PASSWORD_STRENGTH_STATUS_UNKNOWN = -1

SECURITY_LEVEL_UNKNOWN = -1
SECURITY_LEVEL_WEAK = 4
SECURITY_LEVEL_NORMAL = 16
SECURITY_LEVEL_STRONG = 32

MUSIC_SID = 78
CHILD_UPDATE_RESTRICTION_FIELDS = [
    'firstname',
    'lastname',
    'birthday',
    'gender',
    'country',
    'city',
    'timezone',
    'contact_phone_number',
    'content_rating_class',
]

log = logging.getLogger('passport.api.view.bundle.mixins')


def mailish_email_to_social_provider(email):
    _username, _, domain = email.lower().partition('@')
    social_provider = settings.MAILISH_DOMAIN_TO_PROVIDER.get(domain)
    return social_provider


def filter_none(dct):
    return {k: v for k, v in dct.items() if v not in [None, Undefined]}


def clean_name(name=None):
    return smart_text(name or '').strip()


class BundleAccountGetterMixin(object):

    def is_account_type_allowed(self):
        """
        Проверка допустимости типа аккаунта - для переопределения в потомках.
        @return признак допустимости типа аккаунта
        """
        return True

    def parse_account(self, response, enabled_required=True,
                      check_disabled_on_deletion=False, expected_uid=None, user_ticket=None):
        """
        Получить аккаунт из ответа ЧЯ.
        @param response: ответ ЧЯ
        @param enabled_required: требуется ли проверять, заблокирован аккаунт или нет
        @param check_disabled_on_deletion: требуется ли отделять случай удаленного
        аккаунта, подписанного на блокирующие СИДы
        @param expected_uid: ожидаемый uid аккаунта, если не совпадает с uid из response,
        то self.account не устанавливается
        """
        try:
            account = Account().parse(response)
            if expected_uid and account.uid != expected_uid:
                return
            else:
                self.account = account
                self.tvm_user_ticket = user_ticket or response.get('user_ticket')
                setup_log_prefix(self.account)

            if enabled_required and not self.account.is_user_enabled:
                if (
                    check_disabled_on_deletion and
                    (self.account.disabled_status == ACCOUNT_DISABLED_ON_DELETION)
                ):
                    raise AccountDisabledOnDeletionError()
                else:
                    raise AccountDisabledError()
            if not self.is_account_type_allowed():
                raise AccountInvalidTypeError()
        except UnknownUid:
            raise AccountNotFoundError()

    def parse_account_with_domains(self, response, check_disabled_on_deletion=False):
        self.parse_account(
            response,
            enabled_required=False,
            check_disabled_on_deletion=check_disabled_on_deletion,
        )
        if self.account.is_pdd or self.account.is_federal:
            response = self.hosted_domains(self.account.domain.domain)
            self.account.parse(response)

    def get_account_by_login(self, login, enabled_required=True,
                             check_disabled_on_deletion=False, need_phones=False,
                             **blackbox_kwargs):
        """
        Получение аккаунта по заданному login.
        @param login:  login
        @param enabled_required: требуется ли проверять, заблокирован аккаунт или нет
        @param check_disabled_on_deletion: требуется ли отделять случай удаленного
        аккаунта, подписанного на блокирующие СИДы
        @param blackbox_kwargs: словарь дополнительных аргументов для вызова userinfo ЧЯ
        @return ответ метода userinfo ЧЯ, если проверки прошли успешно
        """
        blackbox_kwargs.setdefault('find_by_phone_alias', BLACKBOX_FIND_BY_PHONE_ALIAS_FORCE_ON)
        if 'country' not in blackbox_kwargs:
            blackbox_kwargs['country'] = get_country_code_by_ip(self.client_ip)

        if need_phones:
            blackbox_kwargs = add_phone_arguments(**blackbox_kwargs)

        blackbox_kwargs.setdefault('ip', self.client_ip)

        try:
            response = self.blackbox.userinfo(login=login, **blackbox_kwargs)
            self.parse_account(
                response,
                enabled_required=enabled_required,
                check_disabled_on_deletion=check_disabled_on_deletion,
            )
            return response

        except BlackboxInvalidParamsError as ex:
            # PASSP-11214, PASSP-26156, PASSP-26231 ЧЯ не принял переданный ему логин
            if RE_BLACKBOX_INVALID_LOGIN_ERROR.match(ex.message):
                raise AccountNotFoundError(ex.message)

            raise

    def get_account_by_alt_logins(self, login, allow_federal=False, **kwargs):
        """
        Получение аккаунта по данному login и его альтернативам, например,
        другим телефонам, когда в стране меняется телефонная нумерация.
        """
        error = None
        all_alt_logins = [login]

        country_list = get_countries_suggest() or [None]
        phone = None
        for country in country_list:
            try:
                phone = PhoneNumber.parse(login, country=country)
            except InvalidPhoneNumber as e:
                error = e

        if phone:
            alt_phones = get_alt_phone_numbers_of_phone_number(phone)
            all_alt_logins.extend([p.e164 for p in alt_phones])

        for alt_login in all_alt_logins:
            try:
                return self.get_account_by_login(alt_login, **kwargs)
            except AccountNotFoundError as e:
                error = e

        if settings.SAML_SSO_ENABLED and allow_federal and '@' in login:
            login_part, domain_part = login.rsplit('@', 1)
            domain_info = self.blackbox.hosted_domains(domain=domain_part)
            if domain_info['hosted_domains']:
                domain = Domain().parse(domain_info)
                saml_settings = self.get_saml_settings(domain_id=domain.id, only_enabled=True)
                if domain.is_enabled and saml_settings is not None:
                    federal_alias = '%s/%s' % (domain.id, login_part)
                    try:
                        self.get_account_by_login(federal_alias, sid='federal', **kwargs)
                        return self.account.parse(domain_info)
                    except AccountNotFoundError as e:
                        error = e

        raise error

    def get_account_by_uid(self, uid, enabled_required=True,
                           check_disabled_on_deletion=False, need_phones=False,
                           **blackbox_kwargs):
        """
        Получение аккаунта по заданному UID. Не проверяет гранты!
        @param uid: UID
        @param enabled_required: требуется ли проверять, заблокирован аккаунт или нет
        @param check_disabled_on_deletion: требуется ли отделять случай удаленного
        аккаунта, подписанного на блокирующие СИДы
        @param need_phones: необходимо ли запрашивать информацию о телефонах пользователя
        @param blackbox_kwargs: словарь дополнительных аргументов для вызова userinfo ЧЯ
        @return ответ метода userinfo ЧЯ, если проверки прошли успешно
        """
        if need_phones:
            blackbox_kwargs = add_phone_arguments(**blackbox_kwargs)

        blackbox_kwargs.setdefault('ip', self.client_ip)

        response = self.blackbox.userinfo(uid=uid, **blackbox_kwargs)

        self.parse_account(
            response,
            enabled_required=enabled_required,
            check_disabled_on_deletion=check_disabled_on_deletion,
        )
        return response

    def get_account_from_session(self, get_session_from_headers=True,
                                 check_host_header=True,
                                 enabled_required=True, check_disabled_on_deletion=False,
                                 multisession_uid=None,
                                 need_phones=False, **blackbox_kwargs):
        """
        Получение аккаунта из сессии.
        Записывает session_info в инстанс.
        @param get_session_from_headers: признак получения сессии из заголовков;
        при отрицательном значении сессия берется из формы
        @param enabled_required: требуется ли проверять, заблокирован аккаунт или нет
        @param check_disabled_on_deletion: требуется ли отделять случай удаленного
        аккаунта, подписанного на блокирующие СИДы
        @param multisession_uid: uid пользователя в мультисессии по которому
        строится объект учётной записи
        @param need_phones: необходимо ли запрашивать информацию о телефонах пользователя
        @param blackbox_kwargs: словарь дополнительных аргументов для вызова метода sessionid ЧЯ
        @return ответ метода sessionid ЧЯ, если проверки прошли успешно
        """
        if check_host_header:
            self.check_header(HEADER_CLIENT_HOST)
        if get_session_from_headers:
            self.check_header(HEADER_CLIENT_COOKIE)

        if need_phones:
            blackbox_kwargs = add_phone_arguments(**blackbox_kwargs)

        self.session_info = self.check_session_cookie(
            get_session_from_headers=get_session_from_headers,
            **blackbox_kwargs
        )
        self.login_id = self.session_info.login_id

        if self.session_info.cookie_status == CookieCheckStatus.WrongGuard:
            raise SessguardInvalidError()
        elif self.session_info.cookie_status not in (
            CookieCheckStatus.Valid,
            CookieCheckStatus.NeedReset,
        ):
            raise SessionidInvalidError()

        users = self.session_info.response.get('users')
        user_ticket = self.session_info.response.get('user_ticket')
        multisession_uid = multisession_uid or self.session_info.response['default_uid']
        try:
            user_session = users[multisession_uid]
        except KeyError:
            raise UidNotInSessionError(known_uids=users.keys())

        if user_session.get('status') == BLACKBOX_SESSIONID_DISABLED_STATUS:
            if is_user_disabled_on_deletion(user_session):
                raise AccountDisabledOnDeletionError()
            else:
                raise AccountDisabledError()

        if not is_user_session_valid(user_session):
            raise SessionidInvalidError()

        self.parse_account(
            user_session,
            enabled_required=enabled_required,
            check_disabled_on_deletion=check_disabled_on_deletion,
            user_ticket=user_ticket,
        )
        return self.session_info.response

    def get_account_from_uid_or_session(self, by_uid_grant=None, get_session_from_headers=True,
                                        enabled_required=True, check_disabled_on_deletion=False,
                                        need_phones=False, multisession_uid=None,
                                        ignore_grants=False, check_host_header=True,
                                        **blackbox_kwargs):
        """
        Получение аккаунта по UID из формы или из сессии.
        @param by_uid_grant: грант для получения аккаунта по UID из формы
        при положительном значении UID приоритетнее
        @param get_session_from_headers: признак получения сессии из заголовков;
        при отрицательном значении сессия берется из поля sessionid формы
        @param enabled_required: требуется ли проверять, заблокирован аккаунт или нет
        @param check_disabled_on_deletion: требуется ли отделять случай удаленного
        аккаунта, подписанного на блокирующие СИДы
        @param need_phones: необходимо ли запрашивать информацию о телефонах пользователя
        @param multisession_uid: uid пользователя в мультисессии по которому
        строится объект учётной записи
        @param blackbox_kwargs: словарь дополнительных аргументов для вызова методов ЧЯ (userinfo или sessionid)
        @return ответ ЧЯ, если проверки прошли успешно
        """
        uid = self.form_values.get('uid')
        session_cookie_body, _, _ = self.get_effective_session_and_host(
            get_session_from_headers=get_session_from_headers,
        )
        if uid is not None and session_cookie_body is not None:
            log.warn('uid, session present in request')
            raise RequestCredentialsSeveralPresentError()

        if uid is not None:
            if not ignore_grants:
                self.check_grant(by_uid_grant)
            return self.get_account_by_uid(
                uid,
                enabled_required=enabled_required,
                check_disabled_on_deletion=check_disabled_on_deletion,
                need_phones=need_phones,
                **blackbox_kwargs
            )
        elif session_cookie_body is not None:
            return self.get_account_from_session(
                check_disabled_on_deletion=check_disabled_on_deletion,
                get_session_from_headers=get_session_from_headers,
                need_phones=need_phones,
                multisession_uid=multisession_uid,
                check_host_header=check_host_header,
                **blackbox_kwargs
            )
        else:
            raise RequestCredentialsAllMissingError()

    def get_account_from_track(self, enabled_required=True,
                               check_disabled_on_deletion=False,
                               need_phones=False, **blackbox_kwargs):
        """
        Получение аккаунта из трека.
        @param enabled_required: требуется ли проверять, заблокирован аккаунт или нет
        @param check_disabled_on_deletion: требуется ли отделять случай заблокированного
        аккаунта, подписанного на Яндекс.Деньги
        @param need_phones: необходимо ли запрашивать информацию о телефонах пользователя
        @param blackbox_kwargs: словарь дополнительных аргументов для вызова userinfo ЧЯ
        @return ответ ЧЯ, если проверки прошли успешно
        """
        if not self.track.uid:
            raise InvalidTrackStateError()

        bb_response = self.get_account_by_uid(
            self.track.uid,
            enabled_required=enabled_required,
            check_disabled_on_deletion=check_disabled_on_deletion,
            need_phones=need_phones,
            **blackbox_kwargs
        )

        is_logout_occured = (
            not self.track.is_web_sessions_logout and
            self.account.is_logouted_after(
                unixtime_to_datetime(self.track.logout_checkpoint_timestamp),
            )
        )
        if is_logout_occured:
            raise AccountGlobalLogoutError()  # это может быть и отзыв веб-сессий, а не только глогаут

        return bb_response

    def get_account_from_track_or_uid_or_session(self, by_uid_grant=None,
                                                 get_session_from_headers=True,
                                                 enabled_required=True,
                                                 check_disabled_on_deletion=False,
                                                 need_phones=False,
                                                 multisession_uid=None,
                                                 **blackbox_kwargs):
        """
        Получение аккаунта по UID из трэка или из формы или из сессии.
        @param by_uid_grant: грант для получения аккаунта по UID из формы
        при положительном значении UID приоритетнее
        @param get_session_from_headers: признак получения сессии из заголовков;
        при отрицательном значении сессия берется из поля sessionid формы
        @param enabled_required: требуется ли проверять, заблокирован аккаунт или нет
        @param check_disabled_on_deletion: требуется ли отделять случай удаленного
        аккаунта, подписанного на блокирующие СИДы
        @param need_phones: необходимо ли запрашивать информацию о телефонах пользователя
        @param multisession_uid: uid пользователя в мультисессии по которому
        строится объект учётной записи
        @param blackbox_kwargs: словарь дополнительных аргументов для вызова методов ЧЯ (userinfo или sessionid)
        @return ответ ЧЯ, если проверки прошли успешно
        """
        if self.track.uid:
            return self.get_account_from_track(
                enabled_required=enabled_required,
                check_disabled_on_deletion=check_disabled_on_deletion,
                need_phones=need_phones,
                **blackbox_kwargs
            )
        else:
            return self.get_account_from_uid_or_session(
                by_uid_grant=by_uid_grant,
                check_disabled_on_deletion=check_disabled_on_deletion,
                get_session_from_headers=get_session_from_headers,
                enabled_required=enabled_required,
                need_phones=need_phones,
                multisession_uid=multisession_uid,
                **blackbox_kwargs
            )

    def is_device_id_matches_for_account(self, account, device_id):
        # Сделаем проверку, есть ли совпадения среди xtoken-ов по device_id
        # Сходим в ЧЯ получим токены по юиду и проверим их
        # device_id может быть пустым, тогда возвращаем False
        if not device_id:
            return False

        # Запрашиваем в том числе протухшие токены (full_info=True), потому что надо находить
        # совпадение по device_id даже с отозванными токенами (например после glogout),
        # чтобы владельцы фонишей залогинились уже в неофонишей и все данные склеились
        tokens = self.blackbox.get_oauth_tokens(uid=account.uid, xtoken_only=True, full_info=True)
        devices_ids = [token['oauth']['device_id'] for token in tokens if token['oauth'].get('device_id')]

        return device_id in devices_ids

    def get_account_by_phone_number(self, phone_number, phonish_namespace=None, device_id=None):
        """
        Аккаунт, который можно аутентифицировать через подтверждение данного
        номера.

        Также возвращается номер телефона по которому нашёлся аккаунта (данный
        или альтернативный номер). Если аккаунт не найден, то возвращается
        данный номер.
        """
        all_phone_numbers = [phone_number] + get_alt_phone_numbers_of_phone_number(phone_number)
        bindings = self.blackbox.phone_bindings(
            phone_numbers=[p.e164 for p in all_phone_numbers],
            need_current=True,
            need_history=False,
            need_unbound=False,
        )
        uids = {b['uid'] for b in bindings}
        accounts, _ = get_many_accounts_with_phones_by_uids(
            uids,
            self.blackbox,
        )

        # Список аккаунтов, которым достаточно обладать телефонном номером,
        # чтобы аутентифицировать учётную запись.
        candidates_lastauth = []
        candidates_deviceid = []
        candidates_whitelist = []
        for alt_phone_number in all_phone_numbers:
            for account in accounts:
                if not account.is_phonish:
                    continue

                if phonish_namespace != (account.phonish_namespace or None):
                    continue

                phone = account.phones.by_number(alt_phone_number)
                if not (phone and phone.confirmed and phone.bound):
                    continue

                # Если с момента последнего подтверждения номера прошло больше
                # периода отключения неактивного номера, то нельзя быть уверенным, что
                # официальный владелец номера не менялся.
                if datetime.now() - phone.confirmed >= settings.PERIOD_OF_PHONE_NUMBER_LOYALTY:
                    if self.is_device_id_matches_for_account(account, device_id):
                        candidates_deviceid.append(account)
                    continue

                # События lastauth происходят, когда пользователь удостоверяет свою
                # личность любым способом.
                lastauth = self.get_lastauth(account.uid)
                if lastauth:
                    lastauth = unixtime_to_datetime(lastauth['timestamp'])
                else:
                    # lastauth может припаздывать, так что время регистрация держим
                    # про запас.
                    lastauth = account.registration_datetime

                if datetime.now() - lastauth < settings.RECENT_ACCOUNT_USAGE_PERIOD:
                    candidates_lastauth.append((account, lastauth))
                elif self.is_device_id_matches_for_account(account, device_id):
                    candidates_deviceid.append(account)
                elif account.uid in settings.IGNORE_POSSIBLE_PHONE_OWNER_CHANGE_FOR_UIDS:
                    candidates_whitelist.append(account)

            if candidates_lastauth or candidates_deviceid or candidates_whitelist:
                # Если нашли аккаунты по введённому пользователем номеру, то
                # не продолжаем поиск по другим номерам. Пока у нас нет
                # дублефонишей, такая логика ни на что не влияет. Но, если
                # дублефониши появятся, такая логика поможет логинить
                # пользователя в аккаунт, от которого введён номер.
                break

        if candidates_lastauth:
            # Выберем аккаунт с самым поздним lastauth
            account, _ = max(candidates_lastauth, key=lambda c: c[1])
        elif candidates_deviceid:
            # Выберем первый
            account = candidates_deviceid[0]
        elif candidates_whitelist:
            # Выберем первый
            account = candidates_whitelist[0]
        else:
            account = None

        if account:
            # Т.к. мы могли найти аккаунт не с заданным телефоном, а с
            # альтернативным телефоном для данного, нам нужно сообщить клиенту
            # этот телефон, потому что иначе он не сможет найти его на аккаунте.
            for phone_number in all_phone_numbers:
                if account.phones.by_number(phone_number):
                    break
            else:
                raise NotImplementedError('На найденном по номеру телефона аккаунте не оказалось телефона')

        return account, phone_number

    def get_account_by_oauth_token(self, uid=None,
                                   enabled_required=True,
                                   check_disabled_on_deletion=False,
                                   need_phones=False,
                                   required_scope=X_TOKEN_OAUTH_SCOPE,
                                   whitelisted_client_ids=None,
                                   **blackbox_kwargs):
        """
        Получение аккаунта по oauth токену.
        Записывает session_info в инстанс равным None.
        @param uid: UID аккаунта для проверки совместимости данных в запросе;
        @param enabled_required: требуется ли проверять, заблокирован аккаунт или нет;
        @param check_disabled_on_deletion: требуется ли отделять случай удаленного
        аккаунта, подписанного на блокирующие СИДы;
        @param need_phones: необходимо ли запрашивать информацию о телефонах пользователя;
        @param required_scope: необходимый скоуп для авторизации по токену;
        @param whitelisted_client_ids: список приложений, токен которых принимается независимо от наличия required_scope
        @param blackbox_kwargs: словарь дополнительных аргументов для вызова метода oauth ЧЯ.
        """
        whitelisted_client_ids = whitelisted_client_ids or set()
        self.check_header(HEADER_CONSUMER_AUTHORIZATION)

        if not self.authorization or not self.oauth_token:
            raise AuthorizationHeaderError()

        if need_phones:
            blackbox_kwargs = add_phone_arguments(**blackbox_kwargs)

        blackbox_kwargs.setdefault('ip', self.client_ip)

        bb_response = self.blackbox.oauth(
            self.oauth_token,
            get_login_id=True,
            request_id=self.request.env.request_id,
            **blackbox_kwargs
        )
        if bb_response['status'] != BLACKBOX_OAUTH_VALID_STATUS:
            raise OAuthTokenValidationError()

        if isinstance(required_scope, string_types):
            required_scope = [required_scope]

        if not (
            any(s in bb_response['oauth']['scope'] for s in required_scope) or
            bb_response['oauth']['client_id'] in whitelisted_client_ids
        ):
            raise OAuthTokenValidationError()

        self.parse_account(
            bb_response,
            enabled_required=enabled_required,
            check_disabled_on_deletion=check_disabled_on_deletion,
        )
        if uid and self.account.uid != uid:
            raise AccountUidMismatchError(known_uids=self.account.uid)

        self.session_info = None
        self.oauth_info = bb_response['oauth']
        self.login_id = bb_response.get('login_id')
        return bb_response

    def get_account_by_user_ticket(self, uid=None,
                                   required_scope=(X_TOKEN_OAUTH_SCOPE, SESSIONID_SCOPE),
                                   **blackbox_kwargs):
        user_ticket = self.check_user_ticket(required_scope=required_scope)
        uid = self.get_uid_or_default_from_user_ticket(user_ticket, uid=uid)
        return self.get_account_by_uid(
            uid=uid,
            **blackbox_kwargs
        )

    def get_account_from_session_or_oauth_token(self, multisession_uid=None,
                                                required_scope=X_TOKEN_OAUTH_SCOPE,
                                                whitelisted_client_ids=None,
                                                get_session_from_headers=True,
                                                **kwargs):
        """
        Получение аккаунта из сессии или по oauth токену.
        Метод заполняет поля account и session_info инстанса.
        @param multisession_uid: uid пользователя в мультисессии, по которому
        строится объект учётной записи;
        @param required_scope: необходимый скоуп для авторизации по токену;
        @param get_session_from_headers: признак получения сессии из заголовков,
        при отрицательном значении сессия берется из поля sessionid формы;
        @param kwargs: словарь дополнительных аргументов для вызова методов get_account_from_session
        и get_account_by_oauth_token (enabled_required, check_disabled_on_deletion, need_phones),
        а также методов ЧЯ (userinfo или sessionid).
        @:returns ответ ЧЯ
        """
        media = self.select_authentication_media_name(
            get_session_from_headers=get_session_from_headers,
            enabled_media={AUTHENTICATION_MEDIA_SESSION, AUTHENTICATION_MEDIA_TOKEN},
        )
        if media == AUTHENTICATION_MEDIA_SESSION:
            return self.get_account_from_session(
                multisession_uid=multisession_uid,
                get_session_from_headers=get_session_from_headers,
                **kwargs
            )
        elif media == AUTHENTICATION_MEDIA_TOKEN:
            return self.get_account_by_oauth_token(
                uid=multisession_uid,
                required_scope=required_scope,
                whitelisted_client_ids=whitelisted_client_ids,
                **kwargs
            )
        else:
            raise NotImplementedError()  # pragma: no cover

    def get_account_from_uid_or_session_or_oauth_token(self,
                                                       by_uid_grant=None,
                                                       get_session_from_headers=True,
                                                       ignore_grants=False,
                                                       token_required_scope=X_TOKEN_OAUTH_SCOPE,
                                                       whitelisted_client_ids=None,
                                                       expected_uid=None,
                                                       **kwargs):
        return self.get_account_from_available_media(
            expected_uid=expected_uid,
            multisession_uid=expected_uid,
            enabled_media=(
                AUTHENTICATION_MEDIA_UID,
                AUTHENTICATION_MEDIA_SESSION,
                AUTHENTICATION_MEDIA_TOKEN,
            ),
            required_scope=token_required_scope,
            whitelisted_client_ids=whitelisted_client_ids,
            get_session_from_headers=get_session_from_headers,
            by_uid_grant=by_uid_grant,
            ignore_grants=ignore_grants,
            **kwargs
        )

    def get_account_from_available_media(
        self,
        expected_uid=None,
        multisession_uid=None,
        enabled_media=(
            AUTHENTICATION_MEDIA_UID,
            AUTHENTICATION_MEDIA_USER_TICKET,
            AUTHENTICATION_MEDIA_SESSION,
            AUTHENTICATION_MEDIA_TOKEN,
        ),
        required_scope=(X_TOKEN_OAUTH_SCOPE, SESSIONID_SCOPE),
        whitelisted_client_ids=None,
        get_session_from_headers=True,
        by_uid_grant=None,
        ignore_grants=False,
        **kwargs
    ):
        media = self.select_authentication_media_name(
            get_session_from_headers=get_session_from_headers,
            by_uid_grant=by_uid_grant,
            ignore_grants=ignore_grants,
            enabled_media=enabled_media,
        )
        if media == AUTHENTICATION_MEDIA_USER_TICKET:
            blackbox_response = self.get_account_by_user_ticket(
                uid=multisession_uid,
                required_scope=required_scope,
                **kwargs
            )
        elif media == AUTHENTICATION_MEDIA_SESSION:
            blackbox_response = self.get_account_from_session(
                multisession_uid=multisession_uid,
                get_session_from_headers=get_session_from_headers,
                **kwargs
            )
        elif media == AUTHENTICATION_MEDIA_TOKEN:
            blackbox_response = self.get_account_by_oauth_token(
                uid=multisession_uid,
                required_scope=required_scope,
                whitelisted_client_ids=whitelisted_client_ids,
                **kwargs
            )
        elif media == AUTHENTICATION_MEDIA_UID:
            actual_uid = self.form_values['uid']
            if expected_uid is not None and actual_uid != expected_uid:
                raise AccountUidMismatchError(known_uids=actual_uid)
            blackbox_response = self.get_account_by_uid(actual_uid, **kwargs)
        else:
            raise NotImplementedError()  # pragma: no cover
        return media, blackbox_response

    def select_authentication_media_name(self,
                                         uid=None,
                                         by_uid_grant=None,
                                         get_session_from_headers=True,
                                         ignore_grants=False,
                                         enabled_media=None):
        if enabled_media is None:
            enabled_media = {
                AUTHENTICATION_MEDIA_SESSION,
                AUTHENTICATION_MEDIA_TOKEN,
                AUTHENTICATION_MEDIA_USER_TICKET,
                AUTHENTICATION_MEDIA_UID,
            }

        provided_media = dict()

        if AUTHENTICATION_MEDIA_UID in enabled_media:
            provided_media[AUTHENTICATION_MEDIA_UID] = uid or self.form_values.get('uid')

        if AUTHENTICATION_MEDIA_SESSION in enabled_media:
            session_cookie_body, _, _ = self.get_effective_session_and_host(
                get_session_from_headers=get_session_from_headers,
            )
            provided_media[AUTHENTICATION_MEDIA_SESSION] = session_cookie_body

        if AUTHENTICATION_MEDIA_TOKEN in enabled_media:
            provided_media[AUTHENTICATION_MEDIA_TOKEN] = self.authorization

        if AUTHENTICATION_MEDIA_USER_TICKET in enabled_media:
            provided_media[AUTHENTICATION_MEDIA_USER_TICKET] = self.request.env.user_ticket

        provided_media = remove_none_values(provided_media)
        provided_media = {k: v for k, v in provided_media.items() if k in enabled_media}

        if len(provided_media) > 1:
            log.warn('%s present in request' % ', '.join(provided_media.keys()))
            raise RequestCredentialsSeveralPresentError()

        selected_media = None if not provided_media else list(provided_media.keys())[0]

        if selected_media == AUTHENTICATION_MEDIA_UID:
            if ignore_grants:
                pass
            elif by_uid_grant is not None:
                self.check_grant(by_uid_grant)
            else:
                selected_media = None

        if not selected_media:
            raise RequestCredentialsAllMissingError()

        return selected_media

    def get_tracked_uid(self):
        if not self.track.uid:
            raise InvalidTrackStateError()
        return int(self.track.uid)

    def get_pinned_account_from_session(self, **blackbox_kwargs):
        return self.get_account_from_session(
            multisession_uid=self.get_tracked_uid(),
            **blackbox_kwargs
        )

    def get_pinned_account_from_session_or_oauth_token(self, **blackbox_kwargs):
        return self.get_account_from_session_or_oauth_token(
            multisession_uid=self.get_tracked_uid(),
            **blackbox_kwargs
        )

    def _get_session_from_headers_and_host(self):
        """
        В зависимости от типа пользователя действуют разные куки
        :return: Тело сессионной куки для текущего пользователя
        """

        sessionid_cookie = self.cookies.get('Session_id')
        ssl_sessionid_cookie = self.cookies.get('sessionid2')
        return sessionid_cookie, ssl_sessionid_cookie, self.host

    def _get_session_from_form_and_host(self):
        """
        Костыль для случая передачи куки Session_id в форме
        """
        sessionid_cookie = self.form_values.get('sessionid')
        return sessionid_cookie, None, self.host

    def get_effective_session_and_host(self, get_session_from_headers=True):
        if get_session_from_headers:
            return self._get_session_from_headers_and_host()
        else:
            return self._get_session_from_form_and_host()

    def get_sessguard(self):
        return self.cookies.get('sessguard')

    def check_session_cookie(
        self,
        get_session_from_headers=True,
        extra_sessguard_host=None,
        **blackbox_kwargs
    ):
        """
        Проверить сессионную куку.
        @param get_session_from_headers: признак получения сессии из заголовков;
        при отрицательном значении сессия берется из поля sessionid формы
        @param extra_sessguard_host: дополнительный хост для запроса sessguard в ЧЯ
        @param blackbox_kwargs: словарь дополнительных аргументов для вызова метода sessionid ЧЯ
        @return объект типа SessionInfo
        """
        session_cookie_body, ssl_session_cookie_body, host = self.get_effective_session_and_host(
            get_session_from_headers=get_session_from_headers,
        )
        sessguard_body = self.get_sessguard()
        # Текущий домен может быть бескуковым. Так что если сессионной куки нет - и домен для сесгарда
        # можно не формировать.
        guard_hosts = [build_passport_domain(host)] if session_cookie_body else []
        if extra_sessguard_host:
            guard_hosts.append(extra_sessguard_host)
        return self._check_session_cookie(
            session_cookie_body,
            ssl_session_cookie_body,
            sessguard_body=sessguard_body,
            host=host,
            guard_hosts=guard_hosts,
            **blackbox_kwargs
        )

    def _check_session_cookie(self, session_cookie_body, ssl_session_cookie_body, sessguard_body, host,
                              guard_hosts=None, **blackbox_kwargs):
        """
        Проверим сессионную куку на валидность в ЧЯ.

        Пустой список в dbfields существенно сокращает вес запроса к ЧЯ,
        но делает невозможным парсинг полноценной модели:
        не будет доступна информация из профиля пользователя и пароль.
        Поэтому сохраняем интерфейс как в builders.blackbox: передаем dbfields=None,
        тогда используем список запрашиваемых полей по умолчанию. При явной передаче параметра
        будем использовать его значение.

        uid, логин и домен (для ПДДшника) доступны всегда!

        :return: Результат валидации кук
        :return: UID пользователя, на которого зарегистрирована кука
        """

        # Нет смысла посылать в ЧЯ пустые куки
        if not session_cookie_body:
            return SessionInfo(
                status=CookieCheckStatus.Invalid,
                cookie_status=CookieCheckStatus.Invalid,
            )

        to_statbox({
            'mode': 'check_cookies',
            'host': host,
            'consumer': self.consumer,
            'have_sessguard': sessguard_body is not None,
            'sessionid': mask_sessionid(session_cookie_body),
        })
        # Проверим валидность кук в ЧЯ
        bb_response = self.blackbox.sessionid(
            sessionid=session_cookie_body,
            ip=self.client_ip,
            host=host,
            sslsessionid=ssl_session_cookie_body,
            multisession=True,
            guard_hosts=guard_hosts,
            sessguard=sessguard_body,
            request_id=self.request.env.request_id,
            get_login_id=True,
            **blackbox_kwargs
        )

        # В первую очередь проверим статус всей куки
        cookie_status = bb_response['cookie_status']
        if cookie_status in (
            BLACKBOX_SESSIONID_NOAUTH_STATUS,
            BLACKBOX_SESSIONID_INVALID_STATUS,
            BLACKBOX_SESSIONID_EXPIRED_STATUS,
        ):
            return SessionInfo(
                status=CookieCheckStatus.Invalid,
                cookie_status=CookieCheckStatus.Invalid,
                error=bb_response['error'],
            )
        elif cookie_status == BLACKBOX_SESSIONID_WRONG_GUARD_STATUS:
            return SessionInfo(
                status=CookieCheckStatus.WrongGuard,
                cookie_status=CookieCheckStatus.WrongGuard,
                error=bb_response['error'],
            )

        # Кука в целом валидна, работаем с сессией дефолтного аккаунта
        status = bb_response['status']

        common_session_info = dict(
            response=bb_response,
            error=bb_response['error'],
            session=session_cookie_body,
            ssl_session=ssl_session_cookie_body,
            sessguard=sessguard_body,
            authid=bb_response['authid']['id'],
            ip=bb_response['authid']['ip'],
            create_timestamp=bb_response['authid']['time'],
            ttl=bb_response['ttl'],
            age=bb_response['age'],
            login_id=bb_response.get('login_id'),
        )
        # В первую очередь проверим не удален ли аккаунт
        if status == BLACKBOX_SESSIONID_INVALID_STATUS and 'uid' not in bb_response:
            return SessionInfo(
                status=CookieCheckStatus.Invalid,
                cookie_status=CookieCheckStatus.Valid,
                **common_session_info
            )
        elif status == BLACKBOX_SESSIONID_DISABLED_STATUS:
            log.info('Account is disabled for this session cookie')
            return SessionInfo(
                status=CookieCheckStatus.Disabled,
                cookie_status=CookieCheckStatus.Valid,
                **common_session_info
            )

        # Если пользователь прислал логин и пароль - мы уже знаем его uid
        # проверим что кука выписана для этого самого пользователя
        session_info = SessionInfo(
            uid=bb_response['uid'],
            cookie_status=CookieCheckStatus.Valid,
            **common_session_info
        )

        # Дефолтная сессия протухла или невалидная?
        if status not in (
            BLACKBOX_SESSIONID_VALID_STATUS,
            BLACKBOX_SESSIONID_NEED_RESET_STATUS,
        ):
            session_info.status = CookieCheckStatus.Invalid
            return session_info

        if self.account and self.account.uid:
            is_uid_match = is_user_in_multisession(session_info, self.account.uid)
            if not is_uid_match:
                session_info.status = CookieCheckStatus.Foreign
                return session_info

        if status == BLACKBOX_SESSIONID_NEED_RESET_STATUS:
            # После авторизации нужно переправить пользователя в
            # МДА/resign
            session_info.status = CookieCheckStatus.NeedReset
            return session_info

        session_info.status = CookieCheckStatus.Valid
        return session_info

    def assert_allowed_to_get_cookie(self):
        if (self.account.is_phonish or
            self.account.is_mailish or
            self.account.is_kolonkish or
            self.account.is_kinopoisk
        ):
            # Фонишам, мейлишам, колонкишам и Кинопоиску куку не даём -
            # пусть живут в своём приложении и наружу не лезут
            raise AccountInvalidTypeError()


class BundleAccountResponseRendererMixin(object):

    def pdd_account_domain_info(self, account):
        return {
            'punycode': account.domain.punycode_domain,
            'unicode': account.domain.unicode_domain,
        }

    def account_short_info(self, account):
        account_dict = {
            'uid': account.uid,
            'login': account.login if account.login else '',
            'display_login': account.display_login if account.display_login else '',
        }
        if account.person and account.person.display_name:
            account_dict['display_name'] = account.person.display_name.as_dict()
            account_dict['display_name']['default_avatar'] = account.person.default_avatar or ''
        if account.public_id:
            account_dict['public_id'] = account.public_id
        return account_dict

    def get_multisession_accounts(self, cookie_session_info, exclude_uid=None):
        accounts = []
        session_users = {}
        if cookie_session_info.response is not None:
            session_users = users_from_multisession(cookie_session_info)
        for user in session_users.values():
            try:
                account = Account().parse(user)
                if not account.uid:
                    # Возможна ситуация, когда ЧЯ не возвращает данные о пользователе вообще,
                    # например если это сессия удаленного пользователя
                    continue
                if exclude_uid and exclude_uid == account.uid:
                    continue
                # Берем актуальную информацию по текущему аккаунту.
                if self.account and self.account.uid == account.uid:
                    account = self.account
                accounts.append(self.account_short_info(account))
            except UnknownUid:
                pass
        if self.account and self.account.uid not in session_users:
            accounts.append(self.account_short_info(self.account))
        return accounts

    def account_to_response(self, account, personal_data_required,
                            account_info_required, display_name_required=False, additional_person_data=False,
                            with_display_names=False, with_phones=False, with_emails=False,
                            with_social_profiles=False, with_question=False, additional_account_data=False,
                            with_rfc_2fa=False, with_family_info=False, add_avatar_with_secret_url=False):
        account_dict = {
            'display_login': account.display_login or '',
            'login': account.login or '',
            'uid': account.uid,
        }

        if account_info_required:
            account_dict['is_yandexoid'] = account.is_yandexoid
            account_dict['is_2fa_enabled'] = bool(account.totp_secret.is_set)  # Это свойство бывает Undefined
            account_dict['is_workspace_user'] = bool(account.is_pdd_workspace_user)
        if account_info_required or with_rfc_2fa:
            account_dict['is_rfc_2fa_enabled'] = bool(account.rfc_totp_secret.is_set)
        if account.person and personal_data_required:
            account_dict['person'] = {
                'firstname': account.person.firstname,
                'lastname': account.person.lastname,
            }
            if account.person.birthday:
                account_dict['person']['birthday'] = str(account.person.birthday)
            if account.person.gender:
                account_dict['person']['gender'] = account.person.gender
            if account.person.language:
                account_dict['person']['language'] = account.person.language
            if account.person.country and is_valid_country_code(account.person.country):
                account_dict['person']['country'] = account.person.country
        if account.person and (personal_data_required or display_name_required):
            account_dict['display_name'] = account.person.display_name.as_dict()
            account_dict['display_name']['default_avatar'] = account.person.default_avatar or ''
        if account.person and additional_person_data:
            if account.person.city:
                account_dict['person']['city'] = account.person.city
            account_dict['person']['timezone'] = str(account.person.timezone)

        if account.is_pdd:
            account_dict['domain'] = self.pdd_account_domain_info(account)
            if additional_account_data:
                # Некоторые ПДД-пользователи не имеют права изменять свой пароль и устанавливать 2fa
                response = self.blackbox.hosted_domains(domain=account.domain.domain)
                account.parse(response)
                account_dict['can_change_password'] = account.domain.can_users_change_password

        social_profiles = []
        if with_social_profiles or with_display_names:
            try:
                social_profiles = get_social_api().get_profiles_by_uid(
                    account.uid,
                    subscriptions=True,
                    person=True,
                    expand_provider=True,
                )
            except BaseSocialApiError:
                log.warning('Failed to get social profiles for uid %s' % account.uid)

        if with_display_names:
            account_dict['display_names'] = self.get_display_name_variants(social_profiles)

        if with_social_profiles:
            account_dict['profiles'] = self.social_profiles_response(social_profiles, expand_provider=True)

        if with_emails:
            account_dict['emails'] = self.emails_response() if account.emails else {}

        if with_phones:
            account_dict['phones'] = self.phones_response()

        if with_question and account.hint.is_set:
            account_dict['question'] = dict(
                id=account.hint.question.id,
                text=account.hint.question.text,
            )

        if additional_account_data:
            account_dict['app_passwords_enabled'] = bool(account.enable_app_password)
            # Внимание! Если historydb временно недоступна, отдаем пустой словарь
            # вместо данных о последней авторизации
            account_dict['lastauth'] = self.last_auth_response()
            account_dict['security_level'] = self.get_security_level()
            account_dict['password_info'] = self.get_password_info()

        if with_family_info:
            if account.has_family:
                account_dict['family_info'] = {
                    'family_id': account.family_info.family_id,
                    'admin_uid': account.family_info.admin_uid,
                }
        if account.public_id:
            account_dict['public_id'] = account.public_id
        if account.plus.has_plus:
            account_dict['has_plus'] = account.plus.has_plus
        if account.is_child:
            account_dict['is_child'] = account.is_child
        if account.content_rating_class:
            account_dict['content_rating_class'] = account.content_rating_class

        if add_avatar_with_secret_url:
            avatar_with_secret_url = self.get_avatar_with_secret_url(account)
            if avatar_with_secret_url:
                account_dict['avatar_url'] = avatar_with_secret_url

        return account_dict

    def get_password_info(self):
        strength = PASSWORD_STRENGTH_STATUS_UNKNOWN
        last_update = None
        if self.account.password.is_set:
            last_update = datetime_to_integer_unixtime_nullable(
                self.account.password.update_datetime,
            )
            if self.account.password.is_weak:
                strength = PASSWORD_STRENGTH_STATUS_WEAK
            else:
                strength = PASSWORD_STRENGTH_STATUS_STRONG
        return {
            'strength': strength,
            'last_update': last_update,
            'strong_policy_on': self.account.is_strong_password_required,
        }

    def get_security_level(self):
        has_phone = bool(self.account_phones.suitable_for_restore)
        has_email = bool(self.account.emails.suitable_for_restore)
        has_phone_or_email = has_phone or has_email
        has_hint = self.account.hint.is_set
        has_password = self.account.password.is_set
        weak_password = has_password and self.account.password.is_weak
        strong_password = has_password and not weak_password
        has_2fa = bool(self.account.totp_secret.is_set)

        if has_2fa or (strong_password and has_phone_or_email):
            return SECURITY_LEVEL_STRONG

        if (weak_password and has_phone_or_email) or (strong_password and has_hint):
            return SECURITY_LEVEL_NORMAL

        if (weak_password and has_hint) or (has_password and not (has_hint or has_phone_or_email)):
            return SECURITY_LEVEL_WEAK

        return SECURITY_LEVEL_UNKNOWN

    def last_auth_response(self):
        auths = []
        try:
            auths = self.get_auths_aggregated(
                uid=self.account.uid,
                limit=50,
                password_auths=True,
            )
        except BaseHistoryDBApiError:
            log.warning('Failed to get last auth for uid %s due to HistoryDB failure' % self.account.uid)

        if auths:
            latest_auth = None
            latest_timestamp = 0
            for auth in auths:
                last_timestamp = max(a['timestamp'] for a in auth['authentications'])
                if last_timestamp > latest_timestamp:
                    latest_auth = auth.get('auth')
                    latest_timestamp = last_timestamp
            if latest_auth and latest_timestamp:
                latest_auth.update(timestamp=latest_timestamp)
                return latest_auth
        return {}

    def emails_response(self):
        emails = {
            'default': self.account.emails.default.address if self.account.emails.default else None,
            'native': [email.address for email in self.account.emails.native],
            'confirmed_external': [email.address for email in self.account.emails.confirmed_external],
            'external': [email.address for email in self.account.emails.external],
            'suitable_for_restore': [email.address for email in self.account.emails.suitable_for_restore],
        }

        if (
            self.account.phonenumber_alias and
            self.account.phonenumber_alias.enable_search and
            self.account.emails.native
        ):
            emails.update(
                phonenumber_aliases=['%s@%s' % (self.account.phonenumber_alias.alias, email.domain)
                                     for email in self.account.emails.native],
            )
        return emails

    def phones_response(self):
        phones_data = {}
        for phone in self.account.phones.all().values():
            phone_data = {
                'id': phone.id,
                'number': dump_number(phone.number),
                'is_alias': self.account.phonenumber_alias == phone.number,
            }

            alias = {
                'login_enabled': False,
                'email_enabled': False,
                'email_allowed': is_phonenumber_alias_as_email_allowed(self.account),
            }
            if self.account.phonenumber_alias == phone.number:
                alias['login_enabled'] = True
                if self.account.phonenumber_alias.enable_search:
                    alias['email_enabled'] = True
            phone_data['alias'] = alias

            for key in ['created', 'bound', 'confirmed', 'admitted', 'secured']:
                value = getattr(phone, key)
                if not value:
                    continue
                phone_data[key] = datetime_to_integer_unixtime_nullable(value)

            phone_data['need_admission'] = phone.need_admission
            phone_data['is_default'] = phone == self.account.phones.default

            if phone.operation:
                op = phone.operation
                logical_op = phone.get_logical_operation(self.statbox)
                phone_data['operation'] = filter_none({
                    'id': op.id,
                    'type': op.type,
                    'is_secure_phone_operation': op.is_secure,
                    'started': datetime_to_integer_unixtime(op.started),
                    'code': filter_none({
                        'send_count': op.code_send_count,
                        'last_sent': datetime_to_integer_unixtime_nullable(op.code_last_sent),
                        'checks_count': op.code_checks_count,
                        'confirmed': datetime_to_integer_unixtime_nullable(op.code_confirmed),
                    }),
                    'password_verified': datetime_to_integer_unixtime_nullable(op.password_verified),
                    'phone_id2': op.phone_id2 or None,
                    'finished': datetime_to_integer_unixtime_nullable(op.finished),
                    'does_user_admit_phone': op.does_user_admit_phone,
                    'in_quarantine': logical_op.in_quarantine,
                })

            phones_data[phone.id] = phone_data

        return phones_data

    def social_profiles_response(self, profiles, expand_provider=False):
        social_profiles = deepcopy(profiles)

        for social_profile in social_profiles:
            provider = social_profile['provider']['name'] if expand_provider else social_profile['provider']
            social_api_sids = [subscription['sid'] for subscription in social_profile['subscriptions']]

            social_profile['subscriptions'] = []  # запишем подписки с учетом настроек

            # не показываем флажок возможности социальной авторизации, если он не доступен для провайдера
            if provider not in settings.SOCIAL_AUTH_PROVIDERS:
                del social_profile['allow_auth']

            for service in settings.SOCIAL_DEFAULT_SUBSCRIPTION:
                if provider in service['providers']:
                    social_profile['subscriptions'].append({
                        'sid': service['sid'],
                        'checked': service['sid'] in social_api_sids,
                    })

        yandex_profiles = []
        for profile in social_profiles[:]:
            if profile['provider_code'] == 'ya':
                yandex_profiles.append(profile)
                social_profiles.remove(profile)

        uid_to_yandex_profile = {int(p['userid']): p for p in yandex_profiles}
        accounts, _ = get_many_accounts_by_uids(uid_to_yandex_profile.keys(), self.blackbox)

        phonish_profiles = []
        for account in accounts:
            if account.is_phonish:
                profile = uid_to_yandex_profile[account.uid]
                profile['phonish'] = dict(
                    display_name=account.person.display_name.as_dict(),
                    registration_timestamp=datetime_to_integer_unixtime(account.registration_datetime),
                )
                phonish_profiles.append(profile)

        social_profiles += phonish_profiles

        return social_profiles

    def get_display_name_variants(self, social_profiles):
        all_display_names = {}

        # Сперва сгенерим общие для всех варианты. Они имеют общий префикс, который показывает, является ли аккаунт
        # недорегистрированным социальщиком (если да, то префикс содержит знание о соцпровайдере и profile_id,
        # с которыми социальщик регистрировался).
        # social_profiles могут отсутствовать в том числе тогда, когда запрос в social_api завершился неудачей. В
        # таком случае отдадим имена с префиксом "p": инфа о соцпровайдере всё равно не перетрётся.
        if self.account.is_social and social_profiles:
            main_profile = next(p for p in social_profiles if p['allow_auth'])  # такой всегда существует и единственен
            prefix = 's:{profile_id}:{provider_code}:'.format(
                profile_id=main_profile['profile_id'],
                provider_code=main_profile['provider']['code'],
            )
        else:
            prefix = 'p:'

        passport_variants = {
            self.account.person.firstname,
            self.account.person.lastname,
            u'%s %s' % (clean_name(self.account.person.firstname), clean_name(self.account.person.lastname)),
        }
        if not self.account.is_social:
            passport_variants.add(self.account.login)
        passport_variants = {clean_name(n) for n in passport_variants if n}
        all_display_names.update({name: prefix + name for name in passport_variants if name})

        for profile in social_profiles:
            person = profile.get('person', {})
            social_variants = {
                profile.get('username', ''),
                person.get('firstname', ''),
                person.get('lastname', ''),
                u'%s %s' % (clean_name(person.get('firstname', '')), clean_name(person.get('lastname', ''))),
            }
            social_variants = {clean_name(n) for n in social_variants if n}
            for variant in social_variants:
                if variant:
                    all_display_names.setdefault(variant, prefix + variant)

        # Затем добавим варианты для ПДД и Директории (они имеют особый формат)
        if self.account.is_pdd:
            all_display_names.update(
                {
                    u'%s@%s' % (
                        clean_name(self.account.normalized_login),
                        clean_name(self.account.domain.domain),
                    ): PDD_DOMAIN_TEMPLATE,
                },
            )

        return all_display_names

    def get_avatar_with_secret_url(self, account=None):
        account = account or self.account
        secret_value = ':'.join([
            str(account.uid),
            str(datetime_to_integer_unixtime(account.password.update_datetime or datetime.now())),
        ])
        try:
            secret = self.blackbox.sign(
                value=secret_value,
                ttl=settings.AVATAR_SECRET_TTL,
                sign_space=settings.AVATAR_SECRET_SIGN_SPACE,
            )
        except BaseBlackboxError as ex:
            log.warning('Could not sign secret in blackbox: %s', ex)
            return None

        return settings.GET_AVATAR_WITH_SECRET_URL % (
            quote(account.person.default_avatar or settings.DEFAULT_AVATAR_KEY),
            secret['signed_value']
        )

    def fill_response_with_account(self, personal_data_required=True, account_info_required=False,
                                   additional_person_data=False, with_display_names=False,
                                   with_phones=False, with_emails=False, with_social_profiles=False,
                                   with_question=False, additional_account_data=False,
                                   with_rfc_2fa=False, with_family_info=False, add_avatar_with_secret_url=False):
        self.response_values['account'] = self.account_to_response(
            account=self.account,
            personal_data_required=personal_data_required,
            account_info_required=account_info_required,
            additional_person_data=additional_person_data,
            with_display_names=with_display_names,
            with_phones=with_phones,
            with_emails=with_emails,
            with_social_profiles=with_social_profiles,
            with_question=with_question,
            with_rfc_2fa=with_rfc_2fa,
            additional_account_data=additional_account_data,
            with_family_info=with_family_info,
            add_avatar_with_secret_url=add_avatar_with_secret_url,
        )

    def get_alias_type(self, account=None, can_handle_neophonish=True):
        account = account or self.account
        alias_type = account.type
        # Наследие: у некоторых социальщиков нет пароля,
        # но есть портальный логин.
        if (
            account.social_alias and
            account.portal_alias and
            not account.have_password
        ):
            alias_type = ACCOUNT_TYPE_SOCIAL
        # Для совместимости со старыми АМами маскируем неофонишей под лайтов
        if not can_handle_neophonish and account.is_neophonish:
            alias_type = ACCOUNT_TYPE_LITE
        return alias_type

    def fill_response_with_short_account_info(self, avatar_size, avatar_with_secret=False):
        self.response_values.update(
            uid=self.account.uid,
            primary_alias_type=self.get_alias_type(can_handle_neophonish=False),
        )

        # Отдадим display_login: нужно для совместимости со старыми версиями АМ
        if self.account.is_mailish and self.account.emails.default:
            self.response_values.update(
                display_login=self.account.emails.default.address,
            )
        elif self.account.is_neophonish:
            self.response_values.update(
                display_login=self.account.machine_readable_login,
            )
        elif self.account.display_login:
            self.response_values.update(
                display_login=self.account.display_login,
            )

        if self.account.person.display_name:
            self.response_values.update(
                display_name=self.account.person.display_name.name,
                public_name=self.account.person.display_name.public_name,
            )
            if avatar_with_secret:
                avatar_url = self.get_avatar_with_secret_url() or \
                             settings.GET_AVATAR_URL % (
                                 self.account.person.default_avatar,
                                 avatar_size,
                             )
            else:
                avatar_url = settings.GET_AVATAR_URL % (
                    self.account.person.default_avatar,
                    avatar_size,
                )
            self.response_values.update(avatar_url=avatar_url)
            if self.account.person.display_name.is_social:
                self.response_values.update(social_provider=self.account.person.display_name.provider)

        if self.account.is_mailish:
            # дефолтный email должен быть у всех, но на всякий случай смотрим и на алиас
            if self.account.emails.default:
                email = self.account.emails.default.address
            else:
                email = self.account.mailish_alias.alias
            social_provider = mailish_email_to_social_provider(email)
            if social_provider:
                self.response_values.update(social_provider=social_provider)
        if self.account.person.firstname:
            self.response_values.update(firstname=self.account.person.firstname)
        if self.account.person.lastname:
            self.response_values.update(lastname=self.account.person.lastname)
        if self.account.person.birthday:
            self.response_values.update(birthday=str(self.account.person.birthday))
        if self.account.person.gender:
            self.response_values.update(gender=Gender.to_char(self.account.person.gender))
        if self.account.person.is_avatar_empty:
            self.response_values.update(is_avatar_empty=True)
        if self.account.emails.default and self.account.emails.default.is_native:
            self.response_values.update(native_default_email=unicode_email(self.account.emails.default.address))
        if self.account.is_child:
            self.response_values.update(is_child=True)
        if self.account.is_betatester:
            self.response_values.update(is_beta_tester=True)
        if self.account.yandexoid_alias:
            self.response_values.update(yandexoid_login=self.account.yandexoid_alias.alias)
        if self.account.have_password:
            self.response_values.update(has_password=True)
        if self.account.plus.has_plus:
            self.response_values.update(has_plus=True)
        if self.account.has_sid(MUSIC_SID):
            self.response_values.update(has_music_subscription=True)
        if self.account.login and not (self.account.is_social or self.account.is_phonish or self.account.is_neophonish):
            if self.account.is_mailish and self.account.emails.default:
                normalized_display_login = self.account.emails.default.address.lower()
            elif self.account.is_pdd:
                normalized_display_login = self.account.human_readable_login
            else:
                normalized_display_login = self.account.machine_readable_login
            self.response_values.update(normalized_display_login=normalized_display_login)
        if self.account.public_id:
            self.response_values.update(public_id=self.account.public_id)


class BundleAccountSubscribeMixin(object):

    def subscribe_if_allow_and_update_account(self, account, service=None):
        service = service or self.load_service_from_track()
        if service is None:
            return False

        try:
            with UPDATE(account, self.request.env, {'action': 'subscribe', 'from': service.slug}):
                self.subscribe_if_allow(account, service)
        except DBError:
            log.warning('Error while subscribing user to sid: %s', service.sid)

    def load_service_from_track(self):
        service_slug = self.track.service
        if not service_slug:
            return None
        return get_service(slug=service_slug)

    def subscribe_if_allow(self, account, service=None):
        service = service or self.load_service_from_track()
        if service is None:
            return False

        if account.is_subscribed(service):
            return False

        try:
            can_be_subscribed(account, service)
        except SubscriptionError:
            return False

        # Для galatasaray такой костыль
        if service.sid == 61 and not account.is_normal:
            return False
        if service.internal_subscribe:
            return False
        add_subscription(account, service)
        return True


class BundleAccountPropertiesMixin(object):

    def _check_child_update_restrictions(self):
        if not self.account.is_child:
            return

        if any(item in self.form_values.keys() for item in CHILD_UPDATE_RESTRICTION_FIELDS):
            raise AccountIsChildError()

    def check_have_password(self):
        if not self.account.have_password:
            if self.account.type == ACCOUNT_TYPE_SOCIAL or self.account.is_subscribed(Service.by_slug('social')):
                return RedirectToSocialCompletion()
            raise AccountWithoutPasswordError()

    def check_sms_2fa_disabled(self):
        if self.account.sms_2fa_on:
            # Вместе с ошибкой выдадим и информацию, может ли пользователь отключить смс-2фа,
            # или же ему остаётся только страдать
            self.response_values.update(
                can_disable_sms_2fa=not self.account.forbid_disabling_sms_2fa,
            )
            raise AccountSms2FAEnabledError()

    def check_has_recovery_method(self):
        """
        Проверка, есть ли у пользователя средство восстановления (КВ/КО и/или защищенный телефон).
        """
        return bool(self.account.hint.question or self.account.phones.secure)

    def assert_uid_in_track_is_valid(self):
        if str(self.account.uid) != self.track.uid:
            raise InvalidTrackStateError(
                'Different accounts in track and in session: %s & %s' % (
                    self.track.uid,
                    self.account.uid,
                ),
            )

    def is_password_verification_required(self, uid=None, max_age_threshold=None):
        """
        Данный метод проверяет время последнего ввода пароля:
            * Сначала смотрим на время в куке. Если вводили давно, то
            * проверим время последнего ввода по треку
        """
        threshold = max_age_threshold or settings.PASSWORD_VERIFICATION_MAX_AGE

        if getattr(self, 'session_info', None):
            user = user_from_multisession(self.session_info, int(uid or self.account.uid))
            password_age = user.get('auth', dict()).get('password_verification_age', -1)
        else:
            password_age = -1

        password_required = password_age < 0 or password_age > threshold

        if password_required and self.track:
            password_age = get_unixtime() - int(self.track.password_verification_passed_at or 0)
            password_required = password_age > threshold

        return password_required

    def can_restore_disabled_account(self, account):
        """
        Заблокированный аккаунт можно попытаться автоматически восстановить, если он был заблокирован при удалении,
        но не более 30 дней назад.
        Проверка возвращает осмысленный результат только для заблокированного аккаунта.
        """
        if account.is_pdd and not account.domain.is_enabled:
            return False
        if account.disabled_status == ACCOUNT_DISABLED_ON_DELETION and account.deletion_operation:
            deletion_op_age = datetime.now() - account.deletion_operation.started_at
            max_deletion_op_age = settings.ACCOUNT_DELETION_RESTORE_POSSIBLE_PERIOD
            return deletion_op_age <= max_deletion_op_age
        return False


class BundleAccountFlushMixin(object):

    def drop_phones(self, drop_bank_phone=True, unsecure_bank_phone=False):
        # Телефонный алиас, если он есть, удаляется ручкой
        yasms = yasms_api.Yasms(self.blackbox, self.yasms, self.request.env, self)
        try:
            yasms.remove_userphones(
                self.account,
                self.statbox,
                self.consumer,
                drop_bank_phone=drop_bank_phone,
                unsecure_bank_phone=unsecure_bank_phone,
            )
        except YaSmsError as e:
            log.warning('Unable to remove userphones from uid=%s: %s', self.account.uid, e)

    def drop_emails(self):
        self.account.emails = Emails()

    def drop_social_profiles(self):
        try:
            self.social_api.delete_all_profiles_by_uid(self.account.uid)
        except BaseSocialApiError as e:
            log.warning('Unable to remove social profiles from uid=%s: %s', self.account.uid, e)


class UserMetaDataMixin(object):
    def get_regions(self):
        return get_geobase().regions(smart_bytes(self.client_ip))

    def get_map_url(self, language):
        regions = self.get_regions()
        if regions:
            url_template = (
                'https://static-maps.yandex.ru/1.x/?'
                'll={lon:.6f},{lat:.6f}&'
                'size={width},{height}&'
                'z=13&l=map&'
                'lang={lang}&'
                'pt={lon:.6f},{lat:.6f},pm2rdm'.format(
                    width=450,
                    height=450,
                    lon=regions[0]['longitude'],
                    lat=regions[0]['latitude'],
                    lang=LANGUAGE_TO_LOCALE.get(language, 'en_US'),
                )
            )
            return url_template

    def get_browser(self):
        ua_info = self.request.env.user_agent_info
        browser_name = ua_info.get('BrowserName')
        browser_version = ua_info.get('BrowserVersion')
        os_name = ua_info.get('OSName', ua_info.get('OSFamily'))

        if not browser_name:
            return
        browser = browser_name
        if browser_version:
            browser = '%s %s' % (browser, browser_version)
        if os_name:
            browser = '%s (%s)' % (browser, os_name)
        return browser

    def get_location(self, language=None):
        if language is None:
            language = get_preferred_language(self.account)

        regions = self.get_regions()
        if regions:
            region_id = regions[0]['id']
            region = Region(id=region_id)
            try:
                location = region.linguistics(language).nominative
            except RuntimeError:
                location = regions[0]['en_name']
            if location:
                return smart_text(location)


class BundleAuthNotificationsMixin(UserMetaDataMixin, KolmogorMixin):
    def _get_challenge_email_data(self, language, device_name=None):
        """Возвращает template_name, info, context"""
        translations = settings.translations.NOTIFICATIONS[language]
        template_name = 'mail/auth_challenge_notification.html'
        info = MailInfo(
            subject=translations['auth_challenge.subject'],
            from_=translations['email_sender_display_name'],
            tld=get_tld_by_country(self.account.person.country),
        )
        context = make_email_context(
            language=language,
            account=self.account,
            context={
                'browser': self.get_browser(),
                'location': self.get_location(language),
                'device_name': escape_percents(device_name or ''),
            },
        )
        return template_name, info, context

    def _get_auth_email_data(self, language, device_name=None):
        """Возвращает template_name, info, context"""
        translations = settings.translations.NOTIFICATIONS[language]
        template_name = 'mail/auth_notification.html'
        info = MailInfo(
            subject=translations['auth_challenge.lite.subject'],
            from_=translations['email_sender_display_name'],
            tld=get_tld_by_country(self.account.person.country),
        )
        context = make_email_context(
            language=language,
            account=self.account,
            context={
                'browser': self.get_browser(),
                'location': self.get_location(language),
                'device_name': escape_percents(device_name or ''),
            },
        )
        return template_name, info, context

    def _get_statbox(self, device_name=None, is_challenged=False):
        statbox = None
        if self.statbox:
            statbox = self.statbox.get_child(
                action='auth_notification',
                counter_exceeded=False,
                email_sent=False,
                device_name=device_name,
                is_challenged=bool(is_challenged),
                uid=self.account.uid,
            )

        return statbox

    def _send_auth_emails(self, is_challenged=False, statbox=None, **kwargs):
        language = get_preferred_language(self.account)

        if is_challenged:
            email_data = self._get_challenge_email_data(language, **kwargs)
        else:
            email_data = self._get_auth_email_data(language, **kwargs)

        template_name, info, context = email_data
        send_mail_for_account(template_name, info, context, self.account, login_shadower, send_to_native=True)

        if statbox:
            statbox.bind_context(
                email_sent=True,
            )

    def _should_notify(self, is_challenged):
        if settings.NOTIFY_TESTER_ONLY:
            notifications_allowed = (
                self.account.login and
                is_test_yandex_login(self.account.login)
            )
        else:
            notifications_allowed = True

        return (
            notifications_allowed and
            not is_yandex_server_ip(str(self.client_ip)) and  # не хотим слать письма о входе из Сасово
            not (
                # если аккаунт принужден к смене, значит пользователь и так в курсе
                self.account.password.is_set and
                self.account.password.forced_changing_reason == PASSWORD_CHANGING_REASON_HACKED
            ) and (
                is_challenged or
                is_experiment_enabled_by_uid(self.account.uid, settings.EMAIL_NOTIFICATIONS_DENOMINATOR)
            )
        )

    def try_send_auth_notifications(self, device_name=None, is_challenged=False, **kwargs):
        statbox = self._get_statbox(device_name=device_name, is_challenged=is_challenged)

        try:
            if self._should_notify(is_challenged):
                if auth_email.incr_counter_and_check_limit(self.account.uid):
                    if statbox:
                        statbox.bind_context(counter_exceeded=True)
                    log.debug('User notification rejected: counter exceeded')
                    return False

                self._send_auth_emails(
                    device_name=device_name,
                    is_challenged=is_challenged,
                    statbox=statbox,
                    **kwargs
                )
                return True

            return False
        finally:
            if statbox:
                statbox.log()


class BindRelatedPhonishAccountMixin(object):
    def try_bind_related_phonish_account(self):
        if not (
            settings.IS_BIND_PHONISH_TO_CURRENT_ACCOUNT_ENABLED and
            self.is_secure_phone_confirmed_in_track(allow_by_flash_call=True)
        ):
            return

        app_id = get_app_id_from_track(self.track)
        if app_id not in settings.BIND_RELATED_PHONISH_ACCOUNT_APP_IDS:
            return

        BindPhonishAccountByTrackStatboxEvent(
            ip=self.client_ip,
            track_id=self.track.track_id,
            uid=self.account.uid,
        ).log(SocialBindingLogger())


class MailSubscriptionsMixin(object):
    def unsubscribe_from_maillists_if_nessesary(self):
        unsubscribe_from_maillists_if_nessesary(
            account=self.account,
            form_values=self.form_values,
            track=self.track,
        )
