# -*- coding: utf-8 -*-

from passport.backend.api.views.bundle.change_avatar.exceptions import (
    InvalidAvatarSizeError,
    InvalidFileSizeError,
    InvalidUrlError,
    InvalidYapicResponse,
    MDSTemporaryError,
)
from passport.backend.api.views.bundle.exceptions import RateLimitExceedError
from passport.backend.core.avatars import (
    delete_avatar,
    upload_avatar,
)
from passport.backend.core.builders.avatars_mds_api import (
    AvatarsMdsApiBadImageFormatError,
    AvatarsMdsApiImageNotFoundError,
    AvatarsMdsApiInvalidFileSizeError,
    AvatarsMdsApiInvalidImageSizeError,
    AvatarsMdsApiTemporaryError,
    BaseAvatarsMdsApiError,
)
from passport.backend.core.conf import settings
from passport.backend.core.counters.yapic_upload_ip_uid_counter import (
    is_limit_delete_exceeded,
    is_limit_upload_exceeded,
)

from passport.backend.core.runner.context_managers import UPDATE


class BundleAvatarMixin(object):
    def check_delete_counter(self):
        is_exceed = is_limit_delete_exceeded(self.client_ip, self.account.uid)
        if is_exceed:
            self.statbox.log(
                status='error',
                reason='rate_delete_limit',
            )
            raise RateLimitExceedError()

    def check_upload_counter(self):
        is_exceed = is_limit_upload_exceeded(self.client_ip, self.account.uid)
        if is_exceed:
            self.statbox.log(
                status='error',
                reason='rate_upload_limit',
            )
            raise RateLimitExceedError()

    def update_response_with_avatars(self):
        if self.account.person and self.account.person.default_avatar and not self.account.person.is_avatar_empty:
            url = settings.GET_AVATAR_URL_SCHEMALESS % self.account.person.default_avatar
            default_avatar_data = {
                'timestamp': 0,
                'default': True,
                'url': url,
                'id': self.account.person.default_avatar.split('/', 1)[1],
            }
            self.response_values.update(avatars=[default_avatar_data])
        else:
            self.response_values.update(avatars=[])

    def delete(self):
        delete_avatar(self.account.person.default_avatar)

        events = {
            'action': 'delete_default_avatar',
            'consumer': self.consumer,
        }
        with UPDATE(self.account, self.request.env, events):
            self.account.person.default_avatar = None

    def upload(self, url=None, file_obj=None, is_temp=False):
        if url is None and file_obj is None:
            raise ValueError('Specify one of arguments: url or file_obj')
        self.check_upload_counter()

        try:
            if url is not None:
                avatar_id = upload_avatar(
                    self.account.uid,
                    avatar_url=url,
                )
                self.statbox.log(action='upload_from_url')
            else:
                avatar_id = upload_avatar(
                    self.account.uid,
                    avatar_file=file_obj.stream,
                )
                self.statbox.log(action='upload_from_file')

            if not is_temp:
                events = {
                    'action': 'set_default_avatar',
                    'consumer': self.consumer,
                }
                with UPDATE(self.account, self.request.env, events):
                    self.account.person.default_avatar = avatar_id

            return avatar_id
        except AvatarsMdsApiInvalidImageSizeError:
            raise InvalidAvatarSizeError()
        except AvatarsMdsApiInvalidFileSizeError:
            raise InvalidFileSizeError()
        except (AvatarsMdsApiImageNotFoundError, AvatarsMdsApiBadImageFormatError):
            raise InvalidUrlError()
        except AvatarsMdsApiTemporaryError:
            raise MDSTemporaryError()
        except BaseAvatarsMdsApiError as e:
            # Из соображений совместимости отдаём старую ошибку
            raise InvalidYapicResponse(e)
