# -*- coding: utf-8 -*-
from collections import namedtuple
from datetime import (
    datetime,
    timedelta,
)
import json
import logging

from passport.backend.api.common import account_manager
from passport.backend.api.common.account import is_known_device
from passport.backend.api.common.authorization import log_auth_challenge_shown
from passport.backend.api.common.ip import is_ip_blacklisted
from passport.backend.api.common.profile.estimate import (
    estimate,
    EstimateResult,
)
from passport.backend.api.common.profile.utils import extract_mobile_params_from_track
from passport.backend.api.views.bundle.challenge.controllers import (
    AF_TAG_TO_CHALLENGE_MAP,
    challenges_available,
)
from passport.backend.api.views.bundle.exceptions import (
    CaptchaRequiredError,
    InternalTemporaryError,
    PasswordNotMatchedError,
)
from passport.backend.api.views.bundle.mixins.account import BundleAuthNotificationsMixin
from passport.backend.api.views.bundle.mixins.common import BundleAssertCaptchaMixin
from passport.backend.api.views.bundle.mixins.phone import BundlePhoneMixin
from passport.backend.api.views.bundle.states import AuthChallenge
from passport.backend.core.authtypes import AUTH_TYPE_OAUTH_CREATE
from passport.backend.core.builders.antifraud import (
    BaseAntifraudApiError,
    get_antifraud_api,
    ScoreAction,
    SCORE_TAGS_ALLOWED,
    ScoreResponse,
)
from passport.backend.core.builders.blackbox import (
    BaseBlackboxError,
    BLACKBOX_CHECK_SIGN_STATUS_OK,
)
from passport.backend.core.builders.kolmogor import get_kolmogor
from passport.backend.core.builders.messenger_api import (
    BaseMessengerApiError,
    get_mssngr_fanout_api,
)
from passport.backend.core.builders.oauth import (
    get_fast_oauth,
    OAuthPermanentError,
    OAuthTemporaryError,
)
from passport.backend.core.conf import settings
from passport.backend.core.cookies.cookie_lah import try_parse_cookie_lah
from passport.backend.core.counters.antifraud import (
    incr_af_calls_counter,
    is_af_limit_exceeded,
)
from passport.backend.core.counters.profile_fails import is_profile_broken
from passport.backend.core.env_profile.loader import load_ufo_profile
from passport.backend.core.env_profile.metric import EnvDistance
from passport.backend.core.logging_utils.helpers import trim_message
from passport.backend.core.portallib import (
    is_yandex_ip,
    is_yandex_server_ip,
)
from passport.backend.core.types.login.login import (
    is_test_yandex_login,
    normalize_login,
)
from passport.backend.core.utils.decorators import cached_property
from passport.backend.core.utils.experiments import is_experiment_enabled_by_uid
from passport.backend.utils.common import (
    MixinDependencyChecker,
    noneless_dict,
)
from passport.backend.utils.time import get_unixtime


log = logging.getLogger(__name__)


MobileDeviceStatus = namedtuple('MobileDeviceStatus', ['new', 'old', 'unknown'])._make([0, 1, 2])

MOBILE_DEVICE_STATUS_TO_STR = {v: k for k, v in MobileDeviceStatus._asdict().items()}

# некоторые челленджи (3ds, dictation, webauthn) предназначены для показа только в ЧААС, не проверяем возможность их показа при входе
AF_TAG_TO_CHALLENGE_MAP_FILTERED = {k: v for k, v in AF_TAG_TO_CHALLENGE_MAP.items() if k in SCORE_TAGS_ALLOWED}

AF_ALL_CHALLENGES = list(set(AF_TAG_TO_CHALLENGE_MAP_FILTERED.values()))

AF_TAG_TO_CHALLENGE_NAME_MAP = {k: v.name for k, v in AF_TAG_TO_CHALLENGE_MAP_FILTERED.items()}
CHALLENGE_NAME_TO_AF_TAG_MAP = dict({v: k for k, v in AF_TAG_TO_CHALLENGE_NAME_MAP.items() if v != 'phone_confirmation'}, phone_confirmation='phone_confirmation')


class MobilePasswordSource(object):
    LOGIN = 'login'
    SMARTLOCK = 'smartlock'
    CAPTCHA = 'captcha'
    AUTO_LOGIN = 'autologin'

    all_sources = {LOGIN, SMARTLOCK, CAPTCHA, AUTO_LOGIN}


class DecisionSource(object):
    AM_SMARTLOCK = 'am_smartlock'
    AM_SMARTLOCK_FRAUD = 'am_smartlock_fraud'
    BLACKLISTED_AS = 'blacklisted_AS'
    DEVICE_STATUS_FAILED = 'device_status_failed'
    FORCED_CHALLENGE = 'forced_challenge'
    MOBILE_FULL_PROFILE = 'mobile_full_profile'
    MOBILE_KNOWN_DEVICE = 'mobile_known_device'
    HAS_CARDS = 'has_cards'
    RANDOM = '8-ball'
    TEST_LOGIN = 'test_login'
    ANTIFRAUD_API = 'antifraud_api'
    ANTIFRAUD_FALLBACK = 'antifraud_fallback'
    UFO = 'ufo'
    UFO_FAILED = 'ufo_failed'
    CAPTCHA_ALREADY_PASSED = 'captcha_already_passed'
    CAPTCHA_ALREADY_PASSED_LUCKY = 'captcha_already_passed_lucky'
    SETTINGS = 'settings'
    ACCOUNT_IS_SHARED = 'account_is_shared'
    PASSWORD_CHANGING_REQUIRED = 'password_changing_required'
    TOTP_SECRET_IS_SET = 'totp_secret_is_set'
    AUTH_BY_X_TOKEN = 'auth_by_x_token'
    YANDEX_IP = 'yandex_ip'
    ALL_CHALLENGES_PASSED = 'all_challenges_passed'
    ANTIFRAUD_INVALID_TAGS = 'antifraud_invalid_tags'
    FRESH_ACCOUNT = 'fresh_account'


class MobileProfile(object):
    def __init__(self):
        self.device_status = MobileDeviceStatus.unknown
        self.device_name = None
        self.password_source = None
        self.app_id = None

    def setup(self, track, uid):
        self.app_id = track.device_application

        device_info = account_manager.track_to_oauth_params(account_manager.get_device_params_from_track(track))
        try:
            device_status = get_fast_oauth().device_status(uid=uid, **device_info)
        except (OAuthTemporaryError, OAuthPermanentError) as e:
            log.warning('OAuth device status failed: {}'.format(repr(e)))
            # Ручка не ответила, берём имя устройства из трека, ничего лучше нету
            self.device_name = track.device_name
            return
        else:
            # Берём device_name из ответа ручки, а не из трека, так как в OAuth есть свои эвристики
            # для вычисления явно не переданного имени
            self.device_name = device_status['device_name']

        if device_status['has_auth_on_device']:
            self.device_status = MobileDeviceStatus.old
            return
        else:
            self.device_status = MobileDeviceStatus.new


YANDEX_PROFILE_TEST_LOGIN_PREFIX = 'yndx-profile-test-'
YANDEX_LUCKY_TEST_LOGIN_PREFIX = 'yndx-force-challenge-'
YANDEX_PROFILE_TEST_LOGIN_NEW_CHALLENGE_PREFIX = 'yndx-force-new-challenge-'
YANDEX_YDB_PROFILE_TEST_LOGIN_PREFIX = 'yndx-ydb-profile-test-'


NO_ESTIMATE = EstimateResult(status=None, estimate=None, model=None, threshold=None)


def is_yandex_ydb_profile_test_login(login):
    return normalize_login(login).startswith(YANDEX_YDB_PROFILE_TEST_LOGIN_PREFIX)


def is_yandex_profile_test_login(login):
    return normalize_login(login).startswith(YANDEX_PROFILE_TEST_LOGIN_PREFIX)


def is_yandex_lucky_test_login(login):
    return normalize_login(login).startswith(YANDEX_LUCKY_TEST_LOGIN_PREFIX)


def is_yandex_force_new_challenge_test_login(login):
    return normalize_login(login).startswith(YANDEX_PROFILE_TEST_LOGIN_NEW_CHALLENGE_PREFIX)


def is_lucky_uid(uid, timestamp, chance, buckets, period):
    lucky_buckets = int(buckets * chance)

    if not lucky_buckets:
        return False

    uid_bucket = uid % buckets
    start = (int(timestamp / period) * lucky_buckets) % buckets
    end = (start + lucky_buckets) % buckets

    if start < end:
        return start <= uid_bucket < end
    elif end <= start:
        return start <= uid_bucket < buckets or 0 <= uid_bucket < end


def get_emails_for_challenge(account):
    return [
        email
        for email in account.emails.confirmed_external
        if not (account.lite_alias and email.address == account.lite_alias.alias)
    ]


def extract_full_profile_features(full_profile, name, period):
    feature_name = name + '_' + period
    data = dict(full_profile.get(feature_name, []))
    return data


def extract_full_profile_as_list(full_profile, period):
    # обычные фичи - авторизация, su-фичи - session update (ЧЯ), it - issue token (OAuth)
    # т.к. нас интересует только то, что человек в принципе светился с каких-то мест, нам подойдут все виды фич
    data = [dict(full_profile.get(feature_name + '_' + period, [])).keys() for feature_name in ['as_list_freq', 'su_as_list_freq', 'it_as_list_freq']]
    return set(sum(data, []))


class AntifraudFeatures(object):
    def __init__(
        self,
        unixtime,
        uid,
        track_id,
        client_ip,
        input_login,
        user_agent,
        retpath,
        traffic_fp,
        badauth_counts,
        login_id=None,
        surface=None,
    ):
        self.features = {
            't': unixtime * 1000,
            'uid': uid,
            'external_id': 'track-{}'.format(track_id),
            'channel': 'auth',
            'sub_channel': settings.ANTIFRAUD_AUTH_SUB_CHANNEL,
            'ip': str(client_ip),
            'input_login': input_login,
            'user_agent': user_agent,
            'retpath': retpath,
            'request': 'auth',
            'traffic_fp': traffic_fp,
            'badauth_counts': badauth_counts,
            'login_id': login_id,
            'surface': surface,
        }

    def __setitem__(self, key, value):
        self.features[key] = value

    def as_dict(self):
        return noneless_dict(self.features)

    def add_mobile_profile(self, mobile_profile):
        if mobile_profile:
            self.features.update({
                'known_device': MOBILE_DEVICE_STATUS_TO_STR[mobile_profile.device_status],
                'password_source': mobile_profile.password_source,
            })

    @staticmethod
    def profile_as_dict(cur_profile, **kwargs):
        profile = {
            'AS': cur_profile.AS_list,
            'am_version': cur_profile.am_version,
            'browser_id': cur_profile.browser_id,
            'city_id': cur_profile.city_id,
            'country_id': cur_profile.country_id,
            'device_id': cur_profile.device_id,
            'has_cloud_token': bool(cur_profile.cloud_token),
            'is_mobile': cur_profile.is_mobile,
            'os_id': cur_profile.os_id,
            'yandexuid_timestamp': cur_profile.yandexuid_timestamp,
        }
        profile.update(kwargs)
        return profile

    def add_device_params_from_track(self, track):
        self.features.update(account_manager.get_device_params_from_track(track))

    def add_cur_profile(self, cur_profile):
        self.features.update(self.profile_as_dict(cur_profile))

    def add_account(self, account):
        if account.registration_datetime:
            self.features['reg_date'] = account.registration_datetime.strftime('%Y-%m-%d %H:%M:%S')
        else:
            self.features['reg_date'] = None
        self.features['sms_2fa'] = bool(account.sms_2fa_on)

    def add_profile(self, profile_result_load):
        self.features['profile_loaded'] = profile_result_load.success
        if profile_result_load.success:
            self.features['trusted_device_ids'] = list(profile_result_load.content.trusted_device_ids)
            self.add_full_profile(profile_result_load.content.full_profile or dict())
            self.add_fresh_profiles(profile_result_load.content.fresh_profiles or [])
        else:
            self.features['trusted_device_ids'] = None
            self.add_full_profile({})
            self.add_fresh_profiles([])

    def add_fresh_profiles(self, fresh_profiles):
        if not fresh_profiles:
            return
        self.features['fresh_profiles'] = [self.profile_as_dict(fp,
                                                                ip=fp.raw_env['ip'],
                                                                timestamp=fp.timestamp) for fp in fresh_profiles]

    def add_full_profile(self, full_profile):
        profile_keys = [
            'as_list_freq',
            'browser_freq',
            'browser_name_freq',
            'browser_os_freq',
            'city_freq',
            'country_freq',
            'ip_freq',
            'os_family_freq',
            'os_name_freq',
            'it_am_version_freq',
            'it_as_list_freq',
            'it_city_freq',
            'it_country_freq',
            'it_ip_freq',
            'su_as_list_freq',
            'su_city_freq',
            'su_country_freq',
            'su_ip_freq',
            'yandexuid_freq',
        ]
        for profile_key in profile_keys:
            self.features[profile_key] = extract_full_profile_features(full_profile, profile_key, '6m')
        self.features.update({
            'has_cards': full_profile.get('has_cards'),
            'has_cards_by_linking': full_profile.get('has_cards_by_linking'),
            'has_own_cards': full_profile.get('has_own_cards'),
        })

    def add_social_auth_features(
        self,
        social_provider_code,
        social_userid,
    ):
        self.features.update(
            social_provider_code=social_provider_code,
            social_userid=str(social_userid) if social_userid is not None else None,
        )

    def add_available_challenges(self, available_challenges, additional_challenges=()):
        """
        :param available_challenges: Массив имен челленджей для преобразования в понятные Антифроду
        :param additional_challenges: дополнительные названия челленджей, не требующие какого-то преобразования, например, 'captcha' не есть отдельный челлендж
        """
        result = list(additional_challenges)
        for challenge in set(available_challenges):
            result.append(CHALLENGE_NAME_TO_AF_TAG_MAP[challenge])

        self.features['available_challenges'] = result

    def add_lah_cookie_uids(self, cookie_lah):
        auth_history_container = try_parse_cookie_lah(cookie_lah)
        uids = list({auth_history_item.uid for auth_history_item in auth_history_container})
        self.features['lah_uids'] = uids


class BundleChallengeMixin(object):
    __metaclass__ = MixinDependencyChecker

    mixin_dependencies = [
        BundleAssertCaptchaMixin,
        BundleAuthNotificationsMixin,
        BundlePhoneMixin,
    ]

    # В этом атрибуте кешируем профиль пользователя, полученный из UfoApi
    profile_load_result = None
    # Сюда набиваем фичи для отправки в антифродное API
    antifraud_features = None
    # Кэш решения антифродного API
    antifraud_action = None
    antifraud_tags = None
    antifraud_auth_type = None

    # челленджи которые доступны аккаунту (не учитывая пройдены или нет)
    account_challenges_enabled = None

    @cached_property
    def kolmogor(self):
        return get_kolmogor()

    @staticmethod
    def is_lucky_person(uid):
        """
        Здесь принимается решение, можно ли пользователю показать челлендж, чтобы набрать больше данных для
        обучения модели.

        Суть такова: каждый час меняется диапазон uid'ов, которым надо показать челлендж.
        Это сделано для того, чтобы нельзя было избавиться от челленджа просто меняя track_id, хотя бы в течение часа.
        """
        challenge_time = settings.FORCED_CHALLENGE_PERIOD_LENGTH
        buckets = 10000

        return is_lucky_uid(
            uid,
            get_unixtime(),
            settings.FORCED_CHALLENGE_CHANCE,
            buckets,
            challenge_time,
        )

    @staticmethod
    def load_profile(account, is_mobile, track, statbox):
        """
        Загрузка fresh и full профилей. Выполняется для всех сценариев - мобильного и вебного.
        """
        is_ydb_profile_test_login = is_yandex_ydb_profile_test_login(account.login)
        profile_load_result = load_ufo_profile(
            account.uid,
            account.global_logout_datetime,
            statbox=statbox,
            statbox_context={
                'uid': account.uid,
                'track_id': track.track_id,
                'action': 'ufo_profile_compared',
                'mode': 'any_auth',
                'is_mobile': is_mobile,
            },
            force_ydb=is_ydb_profile_test_login,
        )
        log.debug('Got profile data from {}. Success: {}'.format(
            profile_load_result.source,
            profile_load_result.success,
        ))
        return profile_load_result

    def is_fresh_account(self, is_profile_test_login, is_lucky_test_login):
        is_fresh_account = False

        if self.account.registration_datetime and not is_profile_test_login and not is_lucky_test_login:
            timedelta_since_registration = datetime.now() - self.account.registration_datetime
            if timedelta_since_registration <= timedelta(seconds=settings.PROFILE_TRIAL_PERIOD):
                is_fresh_account = True
        return is_fresh_account

    @staticmethod
    def web_estimate(cur_profile, profile, account, retpath, referer):
        ufo_closest, ufo_distance = profile.content.find_closest(cur_profile)

        estimate_result = estimate(
            account.uid,
            fresh_profile=cur_profile,
            full_profile=profile.content.full_profile,
            retpath=retpath,
            referer=referer,
        )
        return ufo_closest, ufo_distance, estimate_result

    @staticmethod
    def web_decide_on_challenge(ufo_distance, estimate_result, profile):
        decision_source = DecisionSource.UFO
        is_challenge_required = ufo_distance >= settings.WEB_PROFILE_DISTANCE_THRESHOLD
        log.debug('Web challenge: is fresh profile passed? {}'.format(not is_challenge_required))

        # Если расстояние до fresh-профиля достаточно мало, челлендж не нужен
        # В этом случае мы однозначно успешно сходили в ufo, и нам не важно,
        # успешно ли мы сходили в модель.

        if is_challenge_required:
            if not profile.success or not estimate_result.status:
                # Если у нас не все данные для похода в модель (сломался ufo) либо сам поход в модель
                # неуспешен, необходимо требовать челлендж, чтобы нельзя было пробрутфорсить
                # наш механизм, поскольку отказов хоть и мало, но они регулярно происходят.
                # На случай масштабного сбоя, поддерживаем счетчик сбоев, при срабатывании которого
                # перестаем требовать челлендж.
                if is_profile_broken():
                    log.error('Profile is broken, skipping challenge')
                    is_challenge_required = False

                decision_source = DecisionSource.UFO_FAILED
            else:
                # И ufo, и модель отработали штатно - можем полагаться на решение модели
                is_challenge_required = estimate_result.estimate >= estimate_result.threshold

        return decision_source, is_challenge_required

    @staticmethod
    def mobile_estimate(cur_profile, profile):
        ufo_closest, ufo_distance = profile.content.find_closest(cur_profile)

        return ufo_closest, ufo_distance

    def mobile_compare_cur_profile_to_full_profile(self, cur_profile):
        full_profile = self.profile_load_result.content.full_profile or dict()
        period = '3m'  # 3 месяца - взято на глаз, до появления более лучшей информации
        full_AS_list = extract_full_profile_as_list(full_profile, period)

        full_AS_list_str = ','.join(map(str, sorted(full_AS_list))) or '-'
        self.statbox.bind(full_profile_AS_list=full_AS_list_str)
        log.debug('Full profile AS list: {}'.format(full_AS_list_str))

        # Если список AS не сошёлся за нужный период, просим челлендж.
        # Это правило также будет срабатывать, если вообще нет в профиле данных.
        if cur_profile.AS_list and set(cur_profile.AS_list).issubset(set(full_AS_list)):
            log.debug('Current profile AS list {} is found in full profile, check passed'.format(
                repr(cur_profile.AS_list)),
            )
        else:
            log.debug('Current profile AS list {} is not found in full profile, check not passed'.format(
                repr(cur_profile.AS_list)),
            )
            return False

        return True

    def check_avatar_secret_on_autologin(self, avatar_secret):
        """
        :param avatar_secret: секрет из урла аватарки
        :return: True если секрет верный и челендж не требуется
        """
        uid = self.account.uid

        if not is_experiment_enabled_by_uid(
            uid,
            settings.CHECK_AVATARS_SECRETS_DENOMINATOR,
        ):
            return True

        if avatar_secret is None:
            log.warning('Avatar secret is absent on auto-login.')
            return False

        try:
            signed_data = self.blackbox.check_sign(avatar_secret[0], settings.AVATAR_SECRET_SIGN_SPACE)
        except BaseBlackboxError as ex:
            log.warning('Could not check sign in blackbox: %s', ex)
            # не получилось провалидировать секрет, запросим челендж на всякий случай
            return False
        if signed_data['status'] != BLACKBOX_CHECK_SIGN_STATUS_OK:
            log.warning(
                'Avatar secret is invalid. Status: %s.',
                signed_data['status'],
            )
            return False

        signed_uid = signed_data['value'].split(':', 1)[0]
        if int(signed_uid) != uid:
            log.warning(
                'Avatar secret is valid, but contain another uid: %s, user uid: %s.',
                signed_uid,
                uid,
            )
            return False
        self.track.is_avatar_secret_checked = True
        log.info('Avatar secret is valid.')
        return True

    def calculate_available_challenges(self, allow_new_challenge=True):
        _, available_challenges_our_opinion = challenges_available(self, AF_ALL_CHALLENGES, ignore_antifraud_tags=True)
        available_challenges = []
        if allow_new_challenge:
            self.account_challenges_enabled = [challenge.name for challenge in available_challenges_our_opinion.values() if challenge.is_enabled]
            available_challenges = [challenge.name for challenge in available_challenges_our_opinion.values() if challenge.name in self.account_challenges_enabled and not challenge.is_passed]

        return available_challenges

    def make_antifraud_fallback_response(self):
        # тут все челленджи, которые можем показать, за исключением банковского номера и может чего-то еще
        available_challenges = self.antifraud_features.features['available_challenges']
        tags = []
        for challenge_name in settings.ANTIFRAUD_FALLBACK_CHALLENGES:
            if challenge_name not in available_challenges:
                continue
            if challenge_name == 'phone_confirmation':
                tags.extend(settings.ANTIFRAUD_FALLBACK_PHONE_CHALLENGES)
            else:
                tags.append(challenge_name)

        af_response = ScoreResponse(
            action=ScoreAction.ALLOW if tags else ScoreAction.DENY,
            reason=None,
            tags=tags,
        )

        return af_response, DecisionSource.ANTIFRAUD_FALLBACK

    def antifraud_decide_on_challenge(self):
        if (
            self.antifraud_action is None and
            settings.ANTIFRAUD_ON_CHALLENGE_ENABLED and
            (not is_test_yandex_login(self.account.login) or settings.ALLOW_ANTIFRAUD_ON_CHALLENGE_ON_TEST_YANDEX_LOGIN)
        ):
            af_api = get_antifraud_api()
            decision_source = DecisionSource.ANTIFRAUD_API

            try:
                af_response = af_api.score(
                    self.antifraud_features.as_dict(),
                )
            except BaseAntifraudApiError as e:
                incr_af_calls_counter(self.kolmogor)
                log.debug('Antifraud API response ended with error: {}'.format(str(e)))
                self.statbox.bind(af_request_error=trim_message(str(e)))

                is_exceeded = is_af_limit_exceeded(self.kolmogor)
                if settings.ANTIFRAUD_SHOW_CHALLENGE_ON_FAIL and not is_exceeded:
                    af_response, decision_source = self.make_antifraud_fallback_response()
                    log.debug(
                        'Antifraud fallback challenges will be applied: challenges available - %s, action - %s' % (
                            'yes' if len(af_response.tags) else 'no', af_response.action
                        ))
                elif is_exceeded:
                    log.debug('Antifraud fallback challenges will not be applied due to exceeded counters')
                    self.statbox.bind(af_is_limit_exceeded=True)
                    return
                else:
                    log.debug('Antifraud fallback challenges will not be applied due to False flag')
                    return

            log.debug('Antifraud API response: {}'.format(repr(af_response)))
            self.statbox.bind(
                af_is_challenge_required=bool(af_response.tags),
                af_is_auth_forbidden=af_response.action == ScoreAction.DENY,
                af_action=af_response.action,
                af_reason=af_response.reason,
                af_tags=' '.join(map(str, af_response.tags)),
            )
            self.antifraud_action = af_response.action
            self.antifraud_tags = af_response.tags
        elif is_test_yandex_login(self.account.login) and self.track.antifraud_tags:
            self.antifraud_tags = self.track.antifraud_tags

        is_challenge_required = False
        is_auth_forbidden = False
        if self.antifraud_action is None:
            return
        elif self.antifraud_action == ScoreAction.DENY:
            is_auth_forbidden = True
        elif self.antifraud_action == ScoreAction.ALLOW:
            if self.antifraud_tags:
                is_challenge_required = True
                self.track.antifraud_tags = self.antifraud_tags
            else:
                # Пока при allow не отменяем челленж, а используем наше решение
                return
        else:
            raise NotImplementedError('Unknown antifraud action: {}'.format(repr(self.antifraud_action)))

        return decision_source, is_challenge_required, is_auth_forbidden

    def mobile_decide_on_challenge(self, cur_profile, mobile_profile, avatar_secret=None):
        ufo_is_consistently_down = False

        if self.profile_load_result.success:
            # Если удалось загрузить профиль из хранилища,
            # считаем расстояние от fresh профиля и сравниваем текущий профиль с полным
            ufo_closest, ufo_distance = self.mobile_estimate(
                cur_profile,
                self.profile_load_result,
            )
            is_fresh_profile_passed = ufo_distance < settings.MOBILE_PROFILE_DISTANCE_THRESHOLD
            log.debug('Mobile challenge: is fresh profile passed? {}'.format(is_fresh_profile_passed))
            is_full_profile_passed = self.mobile_compare_cur_profile_to_full_profile(cur_profile)
            log.debug('Mobile challenge: is full profile passed? {}'.format(is_full_profile_passed))
        else:
            is_fresh_profile_passed = False
            is_full_profile_passed = False
            ufo_closest = None
            ufo_distance = EnvDistance.Min

            # Если сломался ufo, необходимо требовать челлендж, чтобы нельзя было пробрутфорсить
            # наш механизм, поскольку отказов хоть и мало, но они регулярно происходят.
            # На случай масштабного сбоя, поддерживаем счетчик сбоев, при срабатывании которого
            # перестаем требовать челлендж.
            if is_profile_broken():
                log.error('Profile is broken, skipping challenge')
                ufo_is_consistently_down = True

        self.statbox.bind(
            is_fresh_profile_passed=is_fresh_profile_passed,
            is_full_profile_passed=is_full_profile_passed,
        )

        # Дерево решений о челлендже на мобилках
        # challenge_reason - причина, по которой решили показывать челендж, даже если потом передумали и решили не показывать
        # decision_source - причина, по которой is_challenge_required имеет то значение, которое имеет
        challenge_reason = None
        if is_fresh_profile_passed:
            is_challenge_required = False
            # на самом деле строчка ниже перекроется условием else на проверке device_status
            # пишу её здесь для ясности
            decision_source = DecisionSource.UFO
        else:
            decision_source = DecisionSource.UFO
            if is_full_profile_passed:
                is_challenge_required = False
            else:
                is_challenge_required = True
                challenge_reason = decision_source = DecisionSource.MOBILE_FULL_PROFILE
        if not self.profile_load_result.success:
            challenge_reason = decision_source = DecisionSource.UFO_FAILED
            is_challenge_required = True
        if ufo_is_consistently_down:
            is_challenge_required = False

        log.debug('Mobile challenge: is new mobile device? {}'.format(MOBILE_DEVICE_STATUS_TO_STR[mobile_profile.device_status].capitalize()))
        has_cards = False
        if self.profile_load_result.success and self.profile_load_result.content.full_profile:
            has_cards = self.profile_load_result.content.full_profile.get('has_cards')
            log.debug('Mobile challenge: has cards? {}'.format(bool(has_cards)))

        # Вторая часть проверок, не зависящая от профиля уже
        app_id = (mobile_profile.app_id or '').lower()
        log.debug('Mobile challenge: app id "{}"'.format(app_id))

        password_source = (mobile_profile.password_source or '').lower()
        log.debug('Mobile challenge: password source? {}'.format(password_source or 'unknown'))
        if password_source in (MobilePasswordSource.AUTO_LOGIN, MobilePasswordSource.SMARTLOCK):
            if not self.track.account_manager_version or not self.track.device_os_id:
                is_challenge_required = True
                decision_source = DecisionSource.AM_SMARTLOCK_FRAUD
            else:
                is_challenge_required = not self.check_avatar_secret_on_autologin(avatar_secret)
                decision_source = DecisionSource.AM_SMARTLOCK
            if is_challenge_required:
                challenge_reason = decision_source

        elif mobile_profile.device_status == MobileDeviceStatus.new:  # устройство точно новое
            # Если есть платёжная карта привязанная к аккаунту, то надо бы попросить челлендж на новом устройстве
            if has_cards:
                if settings.CHALLENGE_ON_HAS_CARDS_FOR_ALL_APPS:
                    reason = 'for all apps => challenge'
                    decision_source, is_challenge_required = DecisionSource.HAS_CARDS, True
                    challenge_reason = decision_source
                elif app_id in settings.CHALLENGE_ON_HAS_CARDS_APP_IDS:
                    reason = '{} is in set ({}) => challenge'.format(
                        app_id,
                        ', '.join(sorted(settings.CHALLENGE_ON_HAS_CARDS_APP_IDS)),
                    )
                    decision_source, is_challenge_required = DecisionSource.HAS_CARDS, True
                    challenge_reason = decision_source
                else:
                    reason = '{} is not in set ({}) => no challenge'.format(
                        app_id,
                        ', '.join(sorted(settings.CHALLENGE_ON_HAS_CARDS_APP_IDS)),
                    )

                log.debug('Mobile challenge: new device and has cards: {}'.format(reason))
            else:
                log.debug('Mobile challenge: new device and no cards => no challenge')
            # Если AS входит в чёрный список, всегда требуем челлендж, в остальных кейсах как профиль выше решил
            if is_ip_blacklisted(self.client_ip):
                is_challenge_required = True
                challenge_reason = decision_source = DecisionSource.BLACKLISTED_AS
        elif mobile_profile.device_status == MobileDeviceStatus.old:  # устройство точно старое
            # если устройство известное, челлендж никогда не просим
            decision_source, is_challenge_required = DecisionSource.MOBILE_KNOWN_DEVICE, False
        else:  # не удалось определить (oauth не ответил)
            decision_source, is_challenge_required = DecisionSource.DEVICE_STATUS_FAILED, True
            challenge_reason = decision_source

        return challenge_reason, decision_source, is_challenge_required, ufo_closest, ufo_distance

    def setup_antifraud_features(self):
        self.antifraud_features = AntifraudFeatures(
            get_unixtime(),
            self.account.uid,
            self.track.track_id,
            self.client_ip,
            input_login=self.track.user_entered_login,
            user_agent=self.user_agent,
            retpath=self.form_values.get('retpath'),
            traffic_fp=self.track.ysa_mirror_resolution,
            badauth_counts=self.track.badauth_counts,
            login_id=self.track.login_id,
            surface=self.antifraud_auth_type
        )
        self.antifraud_features.add_account(self.account)
        self.antifraud_features.add_device_params_from_track(self.track)

    def show_challenge_if_necessary(
        self,
        allow_new_challenge=False,
        auth_by_x_token=False,
        mobile_profile=None,
        avatar_secret=None,
        captcha_reason=None,
        allow_captcha=True,
        is_magnitola=False,
    ):
        if not self.antifraud_features:
            self.setup_antifraud_features()
        self.antifraud_features.add_mobile_profile(mobile_profile)
        self.antifraud_features.add_lah_cookie_uids(self.cookies.get('lah'))

        is_mobile = mobile_profile is not None
        mobile_params = dict()
        if is_mobile and self.track is not None:
            mobile_params = extract_mobile_params_from_track(self.track)

        current_profile = self.request.env.make_profile(is_mobile=is_mobile, mobile_params=mobile_params)
        log.debug('Current profile is {}'.format(current_profile.as_json or '-'))
        self.antifraud_features.add_cur_profile(current_profile)

        state = self._try_show_challenge_for_sms_2fa(
            current_profile=current_profile,
            mobile_profile=mobile_profile,
            is_magnitola=is_magnitola,
            auth_by_x_token=auth_by_x_token,
        ) or self._try_show_challenge_for_hacked(
            allow_captcha=allow_captcha,
            allow_new_challenge=allow_new_challenge,
            auth_by_x_token=auth_by_x_token,
            avatar_secret=avatar_secret,
            captcha_reason=captcha_reason,
            current_profile=current_profile,
            mobile_profile=mobile_profile,
            )
        return state

    def _is_new_challenge_available(self):
        if self.account.phones.secure:
            return True

        if get_emails_for_challenge(self.account):
            return True

        return False

    def _log_challenge_to_statbox(self, action, decision_source, is_challenge_required,
                                  current_profile, is_fresh_account=None, mobile_profile=None,
                                  ufo_closest=None, ufo_distance=None, estimate_result=None,
                                  challenge_reason=None, captcha_reason=None):
        if bool(mobile_profile):
            self.statbox.bind(
                device_id=self.track.device_id,
            )
        if estimate_result is not None:
            self.statbox.bind(
                is_model_passed=estimate_result.status and estimate_result.estimate < estimate_result.threshold,
                tensornet_estimate=estimate_result.estimate,
                tensornet_model='-'.join(estimate_result.model) if estimate_result.model else None,
                tensornet_status=estimate_result.status,
            )
        if bool(mobile_profile) and not is_challenge_required:
            self.statbox.bind(
                challenge_reason=challenge_reason,
            )
        if captcha_reason:
            self.statbox.bind(
                captcha_reason=captcha_reason,
            )
        self.statbox.log(**{
            'action': action,
            'current': current_profile.as_json or '-',
            'decision_source': decision_source,
            'input_login': self.track.user_entered_login,
            'ip': self.client_ip,
            'is_challenge_required': is_challenge_required,
            'is_fresh_account': is_fresh_account,
            'is_mobile': bool(mobile_profile),
            'mobile_password_source': mobile_profile.password_source if mobile_profile else None,
            'kind': self.profile_load_result.source,
            'mode': 'any_auth',
            'track_id': self.track.track_id,
            'ufo_closest': json.dumps(ufo_closest.as_dict) if ufo_closest else None,
            'ufo_distance': ufo_distance,
            'ufo_status': self.profile_load_result.success,
            'uid': self.account.uid,
            'user_agent': self.user_agent,
            'yandexuid': self.cookies.get('yandexuid'),
        })

    def _log_skip_challenge_to_statbox(self, decision_source, challenge_reason=None, captcha_reason=None):
        if challenge_reason:
            self.statbox.bind(
                challenge_reason=challenge_reason,
            )
        if captcha_reason:
            self.statbox.bind(
                captcha_reason=captcha_reason,
            )
        self.statbox.log(
            action='skip_challenge',
            decision_source=decision_source,
            device_id=self.track.device_id,
            uid=self.account.uid,
        )

    def _try_show_challenge_for_sms_2fa(self, current_profile, mobile_profile,
                                        is_magnitola=False, auth_by_x_token=False):
        is_mobile = mobile_profile is not None

        if not self.account.sms_2fa_on:
            log.debug('SMS-2FA not enabled')
            return

        # Намеренно не проверяем allow_new_challenge: не должно быть возможности обойти sms-2fa,
        # придя в старую ручку или со старого АМ
        if not self._is_new_challenge_available():
            log.debug('Skipping SMS-2FA challenges: no challenges available')
            return

        if is_magnitola:
            log.debug('Skipping SMS-2FA challenges: app password for magnitola was used')
            return

        if self.is_secure_phone_confirmed_in_track(allow_by_flash_call=True):
            # Не показываем Sms2Fa челлендж даже если пользователь подтвердил
            # телефон флеш-коллом, потому что ручка /auth/password/challenge/submit/
            # разрешает проходить Sms2Fa челлендж даже флеш-коллом.
            log.debug('Skipping SMS-2FA challenges: phone already confirmed')
            return

        if auth_by_x_token:
            log.debug('Skipping SMS-2FA challenges: auth by x-token (PASSP-33654)')
            return

        if self.profile_load_result is None:
            self.profile_load_result = self.load_profile(self.account, is_mobile, self.track, self.statbox)

        on_all_devices = is_experiment_enabled_by_uid(
            self.account.uid,
            settings.DISABLE_SMS_2FA_CHALLENGE_ONLY_ON_KNOWN_DEVICES_DENOMINATOR,
        )
        if not on_all_devices and is_known_device(
            account=self.account,
            ydb_profile=self.profile_load_result.content,
            cookies=self.cookies,
            max_lah_age=None,  # устроит любой возраст куки
            device_id=self.track.device_id,
            cloud_token=self.track.cloud_token,
        ):
            log.debug('Skipping SMS-2FA challenges: device is known')
            return

        self.antifraud_features.add_profile(self.profile_load_result)

        decision_source = 'sms_2fa'
        is_auth_forbidden = False

        self.antifraud_features.add_available_challenges(self.calculate_available_challenges())

        antifraud_resolution = self.antifraud_decide_on_challenge()
        if antifraud_resolution:
            # не учитываем is_challenge_required, так как смягчить нашу резолюцию он не может
            decision_source, _, is_auth_forbidden = antifraud_resolution

        log_auth_challenge_shown(
            self.request.env,
            self.account.uid,
            auth_type=AUTH_TYPE_OAUTH_CREATE if is_mobile else None,
            retpath=self.form_values.get('retpath'),
        )

        self._log_challenge_to_statbox(
            action='sms_2fa_challenge',
            decision_source=decision_source,
            is_challenge_required=True,
            current_profile=current_profile,
            mobile_profile=mobile_profile,
        )

        if is_auth_forbidden:
            # Честно запретить вход мы не можем, поэтому сделаем вид, что пароль неверный
            raise PasswordNotMatchedError()

        self.track.is_auth_challenge_shown = True
        log.debug('Showing SMS-2FA challenge')
        return AuthChallenge()

    def _try_show_challenge_for_hacked(
        self,
        allow_new_challenge,
        current_profile,
        auth_by_x_token,
        mobile_profile,
        avatar_secret=None,
        captcha_reason=None,
        allow_captcha=True,
    ):
        """
        Здесь принимается решение о том, нужно ли показать челлендж пользователю или нет.

        Решение принимается на основании следующих факторов:
        - включены ли челленджи в настройках
        - параметры аккаунта (force_challenge, есть ли 2FA, совместно используемый ли аккаунт, т.п.)
        - оценка похожести профилей моделью
        - тестовый пользователь или нет

        Информация о челленджах сохраняется в statbox-логе и отмечается в треке.

        :param allow_new_challenge есть ли техническая возможность показать челлендж (например, поддержка его в AM)
        :param allow_captcha есть ли техническая возможность показать CAPTCHA
        """

        is_mobile = mobile_profile is not None
        log.debug('Will use {} rules for challenge'.format('mobile' if is_mobile else 'web'))

        # список челленджей что можем показать по требованию АФ шире, надо использовать его в расчетах, иначе есть шанс выйти из метода раньше времени
        af_available_challenges = self.calculate_available_challenges(allow_new_challenge)

        use_new_challenge = allow_new_challenge and bool(af_available_challenges)
        should_send_email = False

        if is_mobile and not use_new_challenge:
            captcha_reason = 'new_challenge_not_available_for_account'

        is_lucky_test_login = is_yandex_lucky_test_login(self.account.login)

        is_profile_test_login = is_yandex_profile_test_login(self.account.login) or \
            is_yandex_force_new_challenge_test_login(self.account.login)
        is_fresh_account = self.is_fresh_account(is_profile_test_login, is_lucky_test_login)

        # Если у пользователя уже запрашивали капчу, а челлендж ему не грозит, то нам делать тут нечего
        if self.is_captcha_passed and not use_new_challenge:
            log.debug('Skipping challenges, captcha already passed')
            if is_mobile:
                self._log_skip_challenge_to_statbox(
                    decision_source=DecisionSource.CAPTCHA_ALREADY_PASSED,
                    captcha_reason=captcha_reason,
                )
            return
        # Если это тестовый lucky пользователь и каптча пройдена, то тоже выходим
        if self.is_captcha_passed and is_lucky_test_login:
            log.debug('Skipping challenges, captcha already passed')
            if is_mobile:
                self._log_skip_challenge_to_statbox(
                    decision_source=DecisionSource.CAPTCHA_ALREADY_PASSED_LUCKY,
                    captcha_reason=captcha_reason,
                )
            return

        # Защита отключена в следующих случаях:
        # - в интранете;
        # - для совместно используемых аккаунтов;
        # - для аккаунтов с принудительной сменой пароля;
        # - для 2FA;
        # - для входа по QR-коду с X-Token в вебе.
        # (но только если не стоит атрибут force_challenge)
        for predicate, decision_source in [
            (not settings.AUTH_PROFILE_ENABLED, DecisionSource.SETTINGS),
            (self.account.is_shared, DecisionSource.ACCOUNT_IS_SHARED),
            (self.account.password.is_changing_required, DecisionSource.PASSWORD_CHANGING_REQUIRED),
            (self.account.totp_secret.is_set, DecisionSource.TOTP_SECRET_IS_SET),
            (auth_by_x_token, DecisionSource.AUTH_BY_X_TOKEN),  # сейчас этот параметр не используется в мобильной авторизации
        ]:
            if predicate and not self.account.force_challenge:
                log.debug('Skipping challenges, not applicable. Decision_source: {}'.format(decision_source))
                if is_mobile:
                    self._log_skip_challenge_to_statbox(
                        decision_source=decision_source,
                        captcha_reason=captcha_reason,
                    )
                return

        if self.profile_load_result is None:
            self.profile_load_result = self.load_profile(self.account, is_mobile, self.track, self.statbox)

        self.antifraud_features.add_profile(self.profile_load_result)

        if is_mobile:
            estimate_result = NO_ESTIMATE  # модель не используется для мобилок

            log.debug('Applying mobile rules for challenge...')
            if settings.ALLOW_PROFILE_CHECK_FOR_MOBILE:

                challenge_reason, decision_source, is_challenge_required, ufo_closest, ufo_distance = self.mobile_decide_on_challenge(
                    current_profile,
                    mobile_profile,
                    avatar_secret,
                )

            else:
                is_challenge_required = False
                log.debug('Skipping mobile profiles check for challenge. This is engaged in antifraud.')
                challenge_reason, decision_source, ufo_closest, ufo_distance = ('', DecisionSource.UFO, None, EnvDistance.Min)

                password_source = (mobile_profile.password_source or '').lower()
                log.debug('Mobile challenge: password source? {}'.format(password_source or 'unknown'))
                if password_source in (MobilePasswordSource.AUTO_LOGIN, MobilePasswordSource.SMARTLOCK):
                    is_challenge_required = not self.check_avatar_secret_on_autologin(avatar_secret)
                    decision_source = DecisionSource.AM_SMARTLOCK
                    if is_challenge_required:
                        challenge_reason = decision_source

            if is_challenge_required:
                # для авторизации на мобилках всегда шлём письмо
                should_send_email = True
            log.debug('Done applying mobile rules for challenge, should {}show challenge'.format(
                '' if is_challenge_required else 'not ',
            ))
        else:
            # для веба должны отправлять письмо только для нового челленджа и только если он показан
            should_send_email = allow_new_challenge
            challenge_reason = None

            log.debug('Applying web rules for challenge...')
            if settings.ALLOW_PROFILE_CHECK_FOR_WEB:
                ufo_closest, ufo_distance, estimate_result = self.web_estimate(
                    current_profile,
                    self.profile_load_result,
                    self.account,
                    self.form_values.get('retpath'),
                    self.referer,
                )
                is_fresh_profile_passed = ufo_distance < settings.WEB_PROFILE_DISTANCE_THRESHOLD
                decision_source, is_challenge_required = self.web_decide_on_challenge(
                    ufo_distance,
                    estimate_result,
                    self.profile_load_result,
                )
                self.statbox.bind(is_fresh_profile_passed=is_fresh_profile_passed)
            else:
                is_challenge_required = False
                estimate_result = NO_ESTIMATE
                log.debug('Skipping web profiles check for challenge. This is engaged in antifraud.')
                decision_source, ufo_closest, ufo_distance = (DecisionSource.UFO, None, EnvDistance.Min)

            log.debug('Done applying web rules for challenge, should {}show challenge'.format(
                '' if is_challenge_required else 'not ',
            ))

        challenge_required_due_to_blacklisted_as = is_challenge_required and decision_source == DecisionSource.BLACKLISTED_AS

        if not is_challenge_required and not is_lucky_test_login:
            # Случайный показ каптчи нужен, чтобы получать больше данных для обучения модели.
            # Если ufo принял решение не показывать челлендж, то здесь всё равно может быть принято решение
            # показать челлендж "случайно", в зависимости от значения FORCED_CHALLENGE_CHANCE и других факторов.
            is_lucky = self.is_lucky_person(self.account.uid)
        else:
            is_lucky = False

        # Если пользователь тестовый - то точно покажем каптчу по этому поводу
        # В противном случае покажем ему каптчу только если is_lucky
        # !Этот кусок с if вырезать, когда уберем флаги ALLOW_PROFILE_CHECK_FOR_MOBILE, ALLOW_PROFILE_CHECK_FOR_WEB!
        if (
            (is_lucky_test_login or is_lucky) and
            (
                is_mobile and settings.ALLOW_PROFILE_CHECK_FOR_MOBILE or
                not is_mobile and settings.ALLOW_PROFILE_CHECK_FOR_WEB
            )
        ):
            decision_source = DecisionSource.RANDOM
            is_challenge_required = True
            # показываем только каптчу, не используем новый челлендж, чтобы ненавязчиво было
            use_new_challenge = False
            # не надо посылать письмо в духе "внимание, кто-то заходит в ваш аккаунт",
            # потому что никакой нештатной ситуации нет, которая требовала бы привлечения внимания.
            should_send_email = False
            challenge_reason = decision_source
            self.statbox.bind(captcha_reason=decision_source)

        for predicate, source in [
            (not settings.AUTH_PROFILE_CHALLENGE_ENABLED, DecisionSource.SETTINGS),
            (is_yandex_server_ip(str(self.client_ip)), DecisionSource.YANDEX_IP),
            (
                not is_profile_test_login and not is_lucky_test_login and
                is_test_yandex_login(self.account.login) and
                is_yandex_ip(str(self.client_ip)),
                DecisionSource.TEST_LOGIN,
            ),
        ]:
            # Выключаем челлендж:
            #  - по настройке (для тестинга)
            #  - при вызове из серверных сетей Яндекса
            #  - для тестовых логинов, пришедших из сетей яндекса
            if predicate:
                is_challenge_required = False
                decision_source = source
                log.debug('Challenge is not applicable, decision_source: {}'.format(source))
                break

        # Включаем челлендж для отдельных тестовых пользователей, независимо от того, что у них с профилем
        if is_profile_test_login:
            challenge_reason = decision_source = DecisionSource.TEST_LOGIN
            is_challenge_required = True
            should_send_email = allow_new_challenge
        elif is_lucky_test_login:
            is_challenge_required = True
            challenge_reason = DecisionSource.TEST_LOGIN
        # если тестировщики из blacklisted AS, включаем им челлендж
        elif is_test_yandex_login(self.account.login) and challenge_required_due_to_blacklisted_as:
            is_challenge_required = True
            challenge_reason = DecisionSource.TEST_LOGIN
        # если стоит флаг force_challenge, включаем челлендж
        elif self.account.force_challenge:
            log.debug('Got force_challenge flag, enforcing challenge')
            challenge_reason = decision_source = DecisionSource.FORCED_CHALLENGE
            is_challenge_required = True
            should_send_email = True

        self.antifraud_features.add_available_challenges(af_available_challenges)

        is_auth_forbidden = False
        if (
            self.is_secure_phone_confirmed_in_track(allow_by_flash_call=True) or
            (not self.account.phones.secure and self.is_bank_phone_confirmed_in_track(allow_by_flash_call=True))
        ):
            # Считаем, что подтверждение телефона (пусть даже флешколлом) - достаточная причина не просить
            # дополнительные челленжи
            log.debug('Skipping antifraud check: phone already confirmed')
        else:
            antifraud_resolution = self.antifraud_decide_on_challenge()
            if antifraud_resolution:
                decision_source, is_challenge_required, is_auth_forbidden = antifraud_resolution
                if is_challenge_required:
                    challenge_reason = decision_source

            log.debug('Profile check: show challenge={}, decision source={}, fresh_account={}'.format(
                is_challenge_required,
                decision_source,
                is_fresh_account,
            ))

        # Необходимость челленджа проверили, теперь проверим есть ли что показать
        if is_challenge_required and use_new_challenge and not is_auth_forbidden:
            skip_challenge_reason = None
            has_available_challenges_af = False
            has_available_challenges_our_opinion = False
            if self.antifraud_tags:
                # если антифрод угадал в тэги, то сразу знаем что есть что показывать
                if set(af_available_challenges) & set({AF_TAG_TO_CHALLENGE_NAME_MAP[tag] for tag in self.antifraud_tags}):
                    has_available_challenges_af = True
                else:
                    log.debug('Can\'t find any available challenges for this account based on antifraud tags: [{}]. Skipping challenge'.format(' '.join(map(str, self.antifraud_tags))))
                    skip_challenge_reason = DecisionSource.ANTIFRAUD_INVALID_TAGS
            else:
                # если же в антифрод не ходили, проверим что можем показать сами
                _, available_challenges_our_opinion = challenges_available(
                    self,
                    AF_ALL_CHALLENGES,
                )
                has_available_challenges_our_opinion = any([challenge.is_enabled and not challenge.is_passed for challenge in available_challenges_our_opinion.values()])
                if not has_available_challenges_our_opinion:
                    skip_challenge_reason = DecisionSource.ALL_CHALLENGES_PASSED

        # сформируем осмысленное сообщение в лог, занулим captcha_reason, если точно не будет каптчи
        if is_auth_forbidden:
            captcha_reason = None

        self._log_challenge_to_statbox(
            action='ufo_profile_checked',
            decision_source=decision_source,
            is_challenge_required=is_challenge_required,
            current_profile=current_profile,
            is_fresh_account=is_fresh_account,
            mobile_profile=mobile_profile,
            ufo_closest=ufo_closest,
            ufo_distance=ufo_distance,
            estimate_result=estimate_result,
            challenge_reason=challenge_reason,
            captcha_reason=captcha_reason,
        )

        # Если пользователь зарегистрировался недавно - челлендж ему не показываем и молча собираем статистику
        # его профилей. Не распространяется на плохие AS, запрет от ФО и прочие отягчающие обстоятельства.
        # !Этот кусок с if вырезать, когда уберем флаги ALLOW_PROFILE_CHECK_FOR_MOBILE, ALLOW_PROFILE_CHECK_FOR_WEB!
        if (
            (
                is_mobile and settings.ALLOW_PROFILE_CHECK_FOR_MOBILE or
                not is_mobile and settings.ALLOW_PROFILE_CHECK_FOR_WEB
            ) and
            is_fresh_account and
            not challenge_required_due_to_blacklisted_as and
            not self.account.force_challenge and
            not is_auth_forbidden and
            not self.antifraud_tags
        ):
            log.debug('Profile is fresh, will not show challenge')
            self._log_skip_challenge_to_statbox(
                decision_source=DecisionSource.FRESH_ACCOUNT,
                challenge_reason=challenge_reason,
                captcha_reason=captcha_reason,
            )
            return

        if is_challenge_required or is_auth_forbidden:
            challenge_time = datetime.now()
            notifications_sent = False
            if should_send_email:
                # вебная ветка: письма отправляем только тем, кому могли бы показать новые челленжи
                # мобильная ветка: письмо шлём всегда, если хотим показать челлендж
                # ToDo: разделить отправку пушей и почты, логировать по отдельности
                # TODO: добавить в письмо ссылку, содержащую challenge_time (PASSP-29869)
                device_name = mobile_profile.device_name if mobile_profile else None
                notifications_sent = self.try_send_auth_notifications(
                    is_challenged=True,
                    device_name=device_name,
                )

            if not is_auth_forbidden:
                log_auth_challenge_shown(
                    self.request.env,
                    self.account.uid,
                    auth_type=AUTH_TYPE_OAUTH_CREATE if is_mobile else None,
                    retpath=self.form_values.get('retpath'),
                    time=challenge_time,
                )
            if settings.MESSENGER_FANOUT_API_ENABLED and not is_lucky:
                try:
                    was_online_sec_ago = get_mssngr_fanout_api().check_user_lastseen(uid=self.account.uid)
                except BaseMessengerApiError as ex:
                    log.warning('Request to Messenger Api failed: {}'.format(ex.message))
                    self.statbox.log(
                        status='error',
                        error='messenger_api.request_failed',
                    )
                else:
                    self.statbox.bind(was_online_sec_ago=was_online_sec_ago)

            self.statbox.log(**{
                'uid': self.account.uid,
                'action': 'profile_threshold_exceeded',
                'track_id': self.track.track_id,
                'input_login': self.track.user_entered_login,
                'ip': self.client_ip,
                'user_agent': self.user_agent,
                'yandexuid': self.cookies.get('yandexuid'),
                'mode': 'any_auth',
                'is_password_change_required': (
                    self.account.password and
                    self.account.password.is_changing_required
                ),
                'decision_source': decision_source,
                'email_sent': notifications_sent,
                'kind': self.profile_load_result.source,
                'is_mobile': is_mobile,
                'mobile_password_source': mobile_profile.password_source if mobile_profile else None,
            })

            if decision_source == DecisionSource.ANTIFRAUD_FALLBACK and is_auth_forbidden:
                # В случае таймаута АФ и отсутствия челленджей говорим "Повторите позже"
                raise InternalTemporaryError()
            if is_auth_forbidden:
                # Честно запретить вход мы не можем, поэтому сделаем вид, что пароль неверный
                raise PasswordNotMatchedError()

            if use_new_challenge:
                if has_available_challenges_our_opinion or has_available_challenges_af:
                    log.debug('Showing new challenge')
                    self.track.is_auth_challenge_shown = True
                    # показываем новый-кленовый челленж
                    return AuthChallenge()
                else:
                    log.debug('User already passed all available challenges')
                    self._log_skip_challenge_to_statbox(
                        decision_source=skip_challenge_reason,
                        challenge_reason=challenge_reason,
                    )
                    return
            elif self.account_challenges_enabled and not af_available_challenges:
                # если прошел 1 или более новых челленджей и более не осталось, не будем отправлять на каптчу
                log.debug('User already passed all available challenges')
                self._log_skip_challenge_to_statbox(
                    decision_source=DecisionSource.ALL_CHALLENGES_PASSED,
                    challenge_reason=challenge_reason,
                )
            elif allow_captcha:
                log.debug('Showing captcha')
                self.track.is_auth_challenge_shown = True
                # фолбечимся на старый "челленж" - капчу
                self.track.is_captcha_required = True
                self.track.is_captcha_recognized = False
                if is_mobile:
                    self.generate_mobile_captcha()
                raise CaptchaRequiredError()
            else:
                log.debug('Skipping challenges, last resort challenge is CAPTCHA, but it is not allowed')
